<?php
// /app/operations/workorders_list.php
declare(strict_types=1);

require_once __DIR__ . '/_lib.php';

$pageTitle = 'Work Orders';
$activeNav = 'operations';

$companyId = ops_require_company_scope();
if (!ops_can_view()) { http_response_code(403); exit('Forbidden'); }

$pdo = db_safe();
$canManage = ops_can_manage();

/* Filters */
$status    = (string)($_GET['status'] ?? '');
$serviceId = (int)($_GET['service_id'] ?? 0);
$q         = trim((string)($_GET['q'] ?? ''));
$from      = (string)($_GET['from'] ?? '');
$to        = (string)($_GET['to'] ?? '');
$view      = (string)($_GET['view'] ?? 'active'); // active|all|completed
if (!in_array($view, ['active','all','completed'], true)) $view = 'active';

$where = ["wo.company_id = :cid"];
$params = [':cid'=>$companyId];

if ($status !== '' && in_array($status, ['new','scheduled','in_progress','blocked','completed','canceled'], true)) {
  $where[] = "wo.status = :status";
  $params[':status'] = $status;
} else {
  if ($view === 'active') {
    $where[] = "wo.status NOT IN ('completed','canceled')";
  } elseif ($view === 'completed') {
    $where[] = "wo.status = 'completed'";
  }
}

if ($serviceId > 0) {
  $where[] = "wo.service_id = :sid";
  $params[':sid'] = $serviceId;
}

if ($q !== '') {
  $where[] = "(wo.customer_name LIKE :q OR wo.serial_number LIKE :q OR wo.external_order_number LIKE :q OR wo.external_order_id LIKE :q OR wo.city LIKE :q OR wo.state LIKE :q)";
  $params[':q'] = '%'.$q.'%';
}

if ($from !== '' && preg_match('/^\d{4}-\d{2}-\d{2}$/', $from)) {
  $where[] = "DATE(wo.created_at) >= :from";
  $params[':from'] = $from;
}
if ($to !== '' && preg_match('/^\d{4}-\d{2}-\d{2}$/', $to)) {
  $where[] = "DATE(wo.created_at) <= :to";
  $params[':to'] = $to;
}

$whereSql = implode(' AND ', $where);

/* Pagination */
$perPage = 50;
$page = max(1, (int)($_GET['page'] ?? 1));
$offset = ($page - 1) * $perPage;

$totalStmt = $pdo->prepare("SELECT COUNT(*) FROM ops_work_orders wo WHERE $whereSql");
$totalStmt->execute($params);
$total = (int)$totalStmt->fetchColumn();
$pages = (int)ceil($total / $perPage);

/* Data */
$sql = "
  SELECT
    wo.*,
    s.service_name,
    lead.user_id AS lead_user_id,
    u.first_name AS lead_first,
    u.last_name  AS lead_last
  FROM ops_work_orders wo
  JOIN ops_services s ON s.id = wo.service_id
  LEFT JOIN ops_work_order_assignees lead ON lead.work_order_id = wo.id AND lead.role='lead'
  LEFT JOIN users u ON u.id = lead.user_id
  WHERE $whereSql
  ORDER BY
    (wo.status='blocked') DESC,
    (wo.status='new') DESC,
    (wo.status='scheduled') DESC,
    wo.scheduled_date IS NULL,
    wo.scheduled_date ASC,
    wo.created_at DESC
  LIMIT :lim OFFSET :off
";
$stmt = $pdo->prepare($sql);
foreach ($params as $k=>$v) $stmt->bindValue($k, $v);
$stmt->bindValue(':lim', $perPage, PDO::PARAM_INT);
$stmt->bindValue(':off', $offset, PDO::PARAM_INT);
$stmt->execute();
$rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

/* Services + status counts for toggle */
$svc = $pdo->prepare("SELECT id, service_name FROM ops_services WHERE company_id=:cid AND is_active=1 ORDER BY service_name");
$svc->execute([':cid'=>$companyId]);
$services = $svc->fetchAll(PDO::FETCH_ASSOC);

$countStmt = $pdo->prepare("
  SELECT
    SUM(status NOT IN ('completed','canceled')) AS active_cnt,
    SUM(status='completed') AS completed_cnt,
    COUNT(*) AS all_cnt
  FROM ops_work_orders
  WHERE company_id=:cid
");
$countStmt->execute([':cid'=>$companyId]);
$counts = $countStmt->fetch(PDO::FETCH_ASSOC) ?: ['active_cnt'=>0,'completed_cnt'=>0,'all_cnt'=>0];

/* Tech list for assignment */
$techs = [];
if ($canManage) {
  // Supports multiple roles (installer/electrician/repair_tech/etc). Adjust as needed.
  $techs = $pdo->prepare("
    SELECT id, CONCAT(first_name,' ',last_name) AS name
    FROM users
    WHERE tenant_type='company' AND tenant_id=:cid
      AND is_active=1
      AND role IN ('electrician','installer','repair_tech','tech','crew_lead')
    ORDER BY first_name, last_name
  ");
  try {
    $techs->execute([':cid'=>$companyId]);
    $techs = $techs->fetchAll(PDO::FETCH_KEY_PAIR);
  } catch (Throwable $e) {
    // If users table doesn't have tenant_type/tenant_id/is_active, fall back to role-only.
    $techs = $pdo->query("
      SELECT id, CONCAT(first_name,' ',last_name) AS name
      FROM users
      WHERE role IN ('electrician','installer','repair_tech','tech','crew_lead')
      ORDER BY first_name, last_name
    ")->fetchAll(PDO::FETCH_KEY_PAIR);
  }
}

require_once __DIR__ . '/includes/header.php';
$activeOps = 'workorders';
require __DIR__ . '/_subnav.php';
?>
<div class="d-flex flex-wrap align-items-center gap-2 mb-3">
  <div class="btn-group" role="group">
    <a class="btn btn-sm <?= $view==='active'?'btn-dark':'btn-outline-dark' ?>" href="?<?=h(http_build_query(array_merge($_GET,['view'=>'active','page'=>1])))?>">
      Active <span class="opacity-75">(<?= (int)$counts['active_cnt'] ?>)</span>
    </a>
    <a class="btn btn-sm <?= $view==='completed'?'btn-dark':'btn-outline-dark' ?>" href="?<?=h(http_build_query(array_merge($_GET,['view'=>'completed','page'=>1])))?>">
      Completed <span class="opacity-75">(<?= (int)$counts['completed_cnt'] ?>)</span>
    </a>
    <a class="btn btn-sm <?= $view==='all'?'btn-dark':'btn-outline-dark' ?>" href="?<?=h(http_build_query(array_merge($_GET,['view'=>'all','page'=>1])))?>">
      All <span class="opacity-75">(<?= (int)$counts['all_cnt'] ?>)</span>
    </a>
  </div>

  <a class="btn btn-primary btn-sm ms-auto" href="/app/operations/workorder_view.php?id=new">Create Work Order</a>
</div>

<form class="card shadow-sm mb-3" method="get">
  <div class="card-body">
    <div class="row g-2 align-items-end">
      <div class="col-12 col-md-3">
        <label class="form-label small text-muted">Search</label>
        <input class="form-control form-control-sm" name="q" value="<?=h($q)?>" placeholder="Customer, serial, order #, city...">
      </div>

      <div class="col-12 col-md-2">
        <label class="form-label small text-muted">Service</label>
        <select class="form-select form-select-sm" name="service_id">
          <option value="0">All services</option>
          <?php foreach ($services as $s): ?>
            <option value="<?= (int)$s['id'] ?>" <?= $serviceId===(int)$s['id']?'selected':'' ?>><?=h($s['service_name'])?></option>
          <?php endforeach; ?>
        </select>
      </div>

      <div class="col-12 col-md-2">
        <label class="form-label small text-muted">Status</label>
        <select class="form-select form-select-sm" name="status">
          <option value="">Any (based on view)</option>
          <?php foreach (['new','scheduled','in_progress','blocked','completed','canceled'] as $st): ?>
            <option value="<?=$st?>" <?=$status===$st?'selected':''?>><?=h(ops_status_label($st))?></option>
          <?php endforeach; ?>
        </select>
      </div>

      <div class="col-6 col-md-2">
        <label class="form-label small text-muted">From</label>
        <input type="date" class="form-control form-control-sm" name="from" value="<?=h($from)?>">
      </div>
      <div class="col-6 col-md-2">
        <label class="form-label small text-muted">To</label>
        <input type="date" class="form-control form-control-sm" name="to" value="<?=h($to)?>">
      </div>

      <div class="col-12 col-md-1 d-grid">
        <button class="btn btn-sm btn-dark" type="submit">Filter</button>
      </div>

      <input type="hidden" name="view" value="<?=h($view)?>">
    </div>
  </div>
</form>

<?php if (!$rows): ?>
  <div class="alert alert-secondary">No work orders found.</div>
<?php else: ?>

<!-- Desktop table -->
<div class="table-responsive d-none d-lg-block">
  <table class="table table-bordered table-striped table-sm align-middle">
    <thead class="table-dark">
      <tr>
        <th>Status</th>
        <th>Service</th>
        <th>Customer</th>
        <th>Serial</th>
        <th>City/State</th>
        <th>Scheduled</th>
        <th>Assigned</th>
        <th>Created</th>
        <th></th>
      </tr>
    </thead>
    <tbody>
      <?php foreach ($rows as $r): ?>
        <?php
          $badge = ops_status_badge((string)$r['status']);
          $assigned = trim((string)($r['lead_first'] ?? '').' '.(string)($r['lead_last'] ?? ''));
          $cs = trim(trim((string)($r['city'] ?? '')).', '.trim((string)($r['state'] ?? '')), ' ,');
        ?>
        <tr>
          <td>
            <?php if ($canManage): ?>
              <select class="form-select form-select-sm js-status" data-id="<?= (int)$r['id'] ?>" data-prev="<?=h((string)$r['status'])?>">
                <?php foreach (['new','scheduled','in_progress','blocked','completed','canceled'] as $st): ?>
                  <option value="<?=$st?>" <?=$st===$r['status']?'selected':''?>><?=h(ops_status_label($st))?></option>
                <?php endforeach; ?>
              </select>
            <?php else: ?>
              <span class="badge bg-<?=$badge?>"><?=h(ops_status_label((string)$r['status']))?></span>
            <?php endif; ?>
          </td>
          <td><?=h((string)$r['service_name'])?></td>
          <td><?=h((string)($r['customer_name'] ?? '—'))?></td>
          <td><?=h((string)($r['serial_number'] ?? '—'))?></td>
          <td><?=h($cs ?: '—')?></td>
          <td><?=h((string)($r['scheduled_date'] ?: '—'))?></td>
          <td>
            <?php if ($canManage): ?>
              <select class="form-select form-select-sm js-lead" data-id="<?= (int)$r['id'] ?>">
                <option value="">— Unassigned —</option>
                <?php foreach ($techs as $uid=>$name): ?>
                  <option value="<?= (int)$uid ?>" <?= ((int)($r['lead_user_id'] ?? 0) === (int)$uid) ? 'selected' : '' ?>><?=h((string)$name)?></option>
                <?php endforeach; ?>
              </select>
            <?php else: ?>
              <?=h($assigned ?: '—')?>
            <?php endif; ?>
          </td>
          <td><?=h((string)date('Y-m-d', strtotime((string)$r['created_at'])))?></td>
          <td><a class="btn btn-sm btn-outline-primary" href="/app/operations/workorder_view.php?id=<?= (int)$r['id'] ?>">Open</a></td>
        </tr>
      <?php endforeach; ?>
    </tbody>
  </table>
</div>

<!-- Mobile cards -->
<div class="d-lg-none">
  <?php foreach ($rows as $r): ?>
    <?php
      $badge = ops_status_badge((string)$r['status']);
      $assigned = trim((string)($r['lead_first'] ?? '').' '.(string)($r['lead_last'] ?? ''));
      $cs = trim(trim((string)($r['city'] ?? '')).', '.trim((string)($r['state'] ?? '')), ' ,');
      $title = (string)$r['service_name'] . ' • ' . ((string)($r['serial_number'] ?? '') !== '' ? (string)$r['serial_number'] : ('WO #'.(int)$r['id']));
    ?>
    <div class="card shadow-sm mb-3" style="border-left:4px solid #111;">
      <div class="card-header d-flex justify-content-between align-items-center bg-dark text-white">
        <div class="fw-semibold"><?=h($title)?></div>
        <span class="badge bg-<?=$badge?>"><?=h(ops_status_label((string)$r['status']))?></span>
      </div>
      <div class="card-body py-2">
        <div class="small text-muted mb-1"><?=h((string)($r['customer_name'] ?? '—'))?> • <?=h($cs ?: '—')?></div>
        <div class="d-flex justify-content-between gap-2">
          <div class="small"><strong>Scheduled:</strong> <?=h((string)($r['scheduled_date'] ?: '—'))?></div>
          <div class="small"><strong>Assigned:</strong> <?=h($assigned ?: '—')?></div>
        </div>
        <a class="btn btn-primary btn-sm w-100 mt-2" href="/app/operations/workorder_view.php?id=<?= (int)$r['id'] ?>">Open Work Order</a>
      </div>
    </div>
  <?php endforeach; ?>
</div>

<?php if ($pages > 1): ?>
  <nav class="mt-3">
    <ul class="pagination pagination-sm flex-wrap">
      <?php for ($i=1; $i<=$pages; $i++): ?>
        <li class="page-item <?=$i===$page?'active':''?>">
          <a class="page-link" href="?<?=h(http_build_query(array_merge($_GET,['page'=>$i])))?>"><?=$i?></a>
        </li>
      <?php endfor; ?>
    </ul>
  </nav>
<?php endif; ?>

<?php endif; ?>

<?php if ($canManage): ?>
<script>
(function(){
  async function postUpdate(action, id, value){
    const body = new URLSearchParams({ action, id, value, csrf: <?= json_encode(csrf_token()) ?> });
    const res = await fetch('/app/operations/workorders_ajax.php', {
      method:'POST',
      headers:{ 'Content-Type':'application/x-www-form-urlencoded' },
      body: body.toString()
    });
    const text = await res.text();
    let j = null;
    try { j = JSON.parse(text); } catch(e) {}
    if (!res.ok) throw new Error((j && j.msg) ? j.msg : text || ('HTTP '+res.status));
    if (!j || !j.ok) throw new Error((j && j.msg) ? j.msg : 'Update failed');
    return j;
  }

  document.addEventListener('change', async (e)=>{
    const el = e.target;
    if (el.classList.contains('js-status')) {
      const id = el.dataset.id;
      const prev = el.dataset.prev || '';
      el.disabled = true;
      try {
        await postUpdate('status', id, el.value);
        el.dataset.prev = el.value;
      } catch (err) {
        alert(err.message || 'Failed to update status');
        if (prev) el.value = prev;
      } finally {
        el.disabled = false;
      }
    }
    if (el.classList.contains('js-lead')) {
      const id = el.dataset.id;
      el.disabled = true;
      try {
        await postUpdate('lead', id, el.value);
      } catch (err) {
        alert(err.message || 'Failed to update assignee');
      } finally {
        el.disabled = false;
      }
    }
  });
})();
</script>
<?php endif; ?>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
