<?php
// /app/operations/webhooks/shedsuite_order.php
declare(strict_types=1);

/**
 * ShedSuite -> ShedOffice webhook receiver (multi-tenant).
 *
 * Authentication:
 *   Header: X-ShedOffice-Webhook-Key: <40-char api key>
 *   Fallback: ?key=<api key>
 *
 * Behavior:
 *   - Stores payload into ops_webhook_inbox (status=pending)
 *   - Processing (mapping addons -> services -> workorders) is performed by cron:
 *       /app/operations/cron/cron_process_webhook_orders.php
 */

require_once __DIR__ . '/../includes/db_config.php'; // provides $pdo in most installs
require_once __DIR__ . '/../includes/db.php';        // optional (db() helper) - ignored if missing

header('Content-Type: application/json; charset=utf-8');

function dbw(): PDO {
  if (function_exists('db')) return db();
  if (isset($GLOBALS['pdo']) && $GLOBALS['pdo'] instanceof PDO) return $GLOBALS['pdo'];
  throw new RuntimeException('No DB connection');
}

$pdo = dbw();

$key = '';
if (!empty($_SERVER['HTTP_X_SHEDOFFICE_WEBHOOK_KEY'])) $key = trim((string)$_SERVER['HTTP_X_SHEDOFFICE_WEBHOOK_KEY']);
if ($key === '' && !empty($_GET['key'])) $key = trim((string)$_GET['key']);

if ($key === '') { http_response_code(401); echo json_encode(['ok'=>false,'msg'=>'Missing webhook key']); exit; }

$k = $pdo->prepare("SELECT * FROM ops_webhook_keys WHERE api_key=:k AND is_enabled=1 LIMIT 1");
$k->execute([':k'=>$key]);
$kr = $k->fetch(PDO::FETCH_ASSOC);
if (!$kr) { http_response_code(403); echo json_encode(['ok'=>false,'msg'=>'Invalid webhook key']); exit; }

$companyId = (int)$kr['company_id'];

$raw = file_get_contents('php://input');
if ($raw === false || trim($raw) === '') { http_response_code(400); echo json_encode(['ok'=>false,'msg'=>'Empty body']); exit; }

$payload = json_decode($raw, true);
if (!is_array($payload)) { http_response_code(400); echo json_encode(['ok'=>false,'msg'=>'Invalid JSON']); exit; }

/* Extract a best-effort external order id and event id */
$externalOrderId = null;
$externalEventId = null;

$externalEventId = $payload['event_id'] ?? $payload['id'] ?? null;

$order = $payload['order'] ?? $payload['data']['order'] ?? $payload['data'] ?? $payload;
if (is_array($order)) {
  $externalOrderId = $order['order_id'] ?? $order['id'] ?? $order['uuid'] ?? null;
}
if ($externalOrderId === null && isset($payload['order_id'])) $externalOrderId = $payload['order_id'];

$ip = $_SERVER['REMOTE_ADDR'] ?? null;

/* Header capture (safe subset) */
$hdrs = [];
foreach ($_SERVER as $kname=>$v) {
  if (str_starts_with($kname, 'HTTP_')) {
    $hdrs[$kname] = (string)$v;
  }
}

$ins = $pdo->prepare("
  INSERT INTO ops_webhook_inbox
    (company_id, provider, external_event_id, external_order_id, source_ip, headers_json, payload_json, status)
  VALUES
    (:cid, 'shedsuite', :eid, :oid, :ip, :hdrs, :payload, 'pending')
");

$ins->execute([
  ':cid'=>$companyId,
  ':eid'=>$externalEventId ? (string)$externalEventId : null,
  ':oid'=>$externalOrderId ? (string)$externalOrderId : null,
  ':ip'=>$ip,
  ':hdrs'=>json_encode($hdrs, JSON_UNESCAPED_SLASHES),
  ':payload'=>json_encode($payload, JSON_UNESCAPED_SLASHES),
]);

$inboxId = (int)$pdo->lastInsertId();

/* Update last used */
$pdo->prepare("UPDATE ops_webhook_keys SET last_used_at=UTC_TIMESTAMP(), last_used_ip=:ip WHERE id=:id")
    ->execute([':ip'=>$ip, ':id'=>(int)$kr['id']]);

echo json_encode(['ok'=>true, 'inbox_id'=>$inboxId]);
