<?php
// /app/repos/repo_store.php
declare(strict_types=1);

require_once __DIR__ . '/../includes/auth.php';
require_company_context_if_rto();

require_once __DIR__ . '/../includes/acl.php';
require_once __DIR__ . '/../includes/scope.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  header('Location: ' . app_path('repos/repos_list.php'));
  exit;
}

if (!csrf_validate((string)($_POST['_csrf'] ?? ''))) {
  http_response_code(403);
  echo 'CSRF validation failed.';
  exit;
}

$pdo = db();
acl_bootstrap($pdo);

$tenantType = current_tenant_type(); // company|rto
$companyTenantId = company_context_id();
$rtoTenantId = (int) current_tenant_id();

$userId = (int) current_user_id();

$isCompanyAdmin = user_has_role('company_admin');
$isManager = user_has_role('manager');
$canAssignDriver = ($tenantType !== 'rto') && ($isCompanyAdmin || $isManager || user_can('repos.assign_driver') || user_can('repos.view_all'));

// Collect inputs
$assetId = (int)($_POST['asset_id'] ?? 0);
$priority = trim((string)($_POST['priority'] ?? 'medium'));
$driverUserId = (int)($_POST['driver_user_id'] ?? 0);

$pickup1 = trim((string)($_POST['pickup_address1'] ?? ''));
$pickup2 = trim((string)($_POST['pickup_address2'] ?? ''));
$city = trim((string)($_POST['pickup_city'] ?? ''));
$state = trim((string)($_POST['pickup_state'] ?? ''));
$zip = trim((string)($_POST['pickup_zip'] ?? ''));
$notes = trim((string)($_POST['notes'] ?? ''));

$validPriorities = ['low','medium','high'];
if ($assetId < 1 || !in_array($priority, $validPriorities, true) || $pickup1 === '' || $city === '' || $state === '' || $zip === '') {
  http_response_code(400);
  echo 'Missing or invalid fields.';
  exit;
}

// Normalize state
$state = strtoupper($state);

// Driver assignment rules
if ($tenantType === 'rto') {
  $driverUserId = 0; // RTO cannot assign drivers
}
if (!$canAssignDriver) {
  $driverUserId = 0; // enforce server-side
}

try {
  $pdo->beginTransaction();

  // If RTO, verify link to company tenant
  if ($tenantType === 'rto') {
    $stmt = $pdo->prepare("
      SELECT id
      FROM rto_company_links
      WHERE rto_tenant_id = :rto1
        AND company_tenant_id = :co1
        AND status='active'
      LIMIT 1
    ");
    $stmt->execute([':rto1' => $rtoTenantId, ':co1' => $companyTenantId]);
    if (!$stmt->fetchColumn()) {
      $pdo->rollBack();
      http_response_code(403);
      echo 'Forbidden: RTO is not linked to this company.';
      exit;
    }
  }

  // Validate asset belongs to company tenant
  $stmt = $pdo->prepare("
    SELECT id, status
    FROM assets
    WHERE tenant_id = :tid1 AND id = :aid1
    LIMIT 1
    FOR UPDATE
  ");
  $stmt->execute([':tid1' => $companyTenantId, ':aid1' => $assetId]);
  $asset = $stmt->fetch();

  if (!$asset) {
    $pdo->rollBack();
    http_response_code(400);
    echo 'Invalid asset selection.';
    exit;
  }

  // Enforce one active repo per asset (any status except closed/canceled)
  $stmt = $pdo->prepare("
    SELECT id
    FROM repos
    WHERE tenant_id = :tid1
      AND asset_id = :aid1
      AND status NOT IN ('closed','canceled')
    LIMIT 1
    FOR UPDATE
  ");
  $stmt->execute([':tid1' => $companyTenantId, ':aid1' => $assetId]);
  $existingRepoId = (int)($stmt->fetchColumn() ?: 0);

  if ($existingRepoId) {
    $pdo->rollBack();
    http_response_code(409);
    echo "This asset already has an active repo (#{$existingRepoId}). Close/cancel it first.";
    exit;
  }

  // Validate driver belongs to company tenant if assigned
  $assignedDriver = null;
  if ($driverUserId > 0) {
    $stmt = $pdo->prepare("
      SELECT id
      FROM users
      WHERE tenant_id = :tid1 AND id = :uid1 AND status='active'
      LIMIT 1
    ");
    $stmt->execute([':tid1' => $companyTenantId, ':uid1' => $driverUserId]);
    if (!$stmt->fetchColumn()) {
      $pdo->rollBack();
      http_response_code(400);
      echo 'Invalid driver selection.';
      exit;
    }
    $assignedDriver = $driverUserId;
  }

  // Insert repo
  $stmt = $pdo->prepare("
    INSERT INTO repos
      (tenant_id, rto_tenant_id, asset_id, location_id, status, priority,
       assigned_driver_user_id,
       pickup_address1, pickup_address2, pickup_city, pickup_state, pickup_zip,
       picked_up_at, closed_at, notes, created_by_user_id, created_at, updated_at)
    VALUES
      (:tid1, :rto1, :aid1, NULL, 'new', :prio1,
       :driver1,
       :p1, :p2, :city1, :state1, :zip1,
       NULL, NULL, :notes1, :cb1, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
  ");

  $stmt->execute([
    ':tid1' => $companyTenantId,
    ':rto1' => ($tenantType === 'rto') ? $rtoTenantId : null,
    ':aid1' => $assetId,
    ':prio1' => $priority,
    ':driver1' => $assignedDriver,
    ':p1' => $pickup1,
    ':p2' => ($pickup2 === '' ? null : $pickup2),
    ':city1' => $city,
    ':state1' => $state,
    ':zip1' => $zip,
    ':notes1' => ($notes === '' ? null : $notes),
    ':cb1' => $userId,
  ]);

  $repoId = (int)$pdo->lastInsertId();

  // Update asset status to repo_in_progress (enterprise-friendly)
  $stmt = $pdo->prepare("
    UPDATE assets
    SET status = 'repo_in_progress',
        updated_at = CURRENT_TIMESTAMP
    WHERE tenant_id = :tid1 AND id = :aid1
  ");
  $stmt->execute([':tid1' => $companyTenantId, ':aid1' => $assetId]);

  // Log repo event
  $meta = [
    'action' => 'created',
    'tenant_type' => $tenantType,
    'priority' => $priority,
    'driver_assigned' => $assignedDriver ? true : false
  ];

  $stmt = $pdo->prepare("
    INSERT INTO repo_events (tenant_id, repo_id, event_type, meta_json, created_by_user_id, created_at)
    VALUES (:tid1, :rid1, 'created', :meta1, :uid1, CURRENT_TIMESTAMP)
  ");
  $stmt->execute([
    ':tid1' => $companyTenantId,
    ':rid1' => $repoId,
    ':meta1' => json_encode($meta, JSON_UNESCAPED_SLASHES),
    ':uid1' => $userId,
  ]);

  $pdo->commit();

  header('Location: ' . app_path('repos/repo_view.php?id=' . $repoId . '&ok=1'));
  exit;

} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  http_response_code(500);
  echo 'Failed to create repo. ' . htmlspecialchars($e->getMessage());
  exit;
}
