<?php
// /app/repos/repo_confirm_pickup.php
declare(strict_types=1);

require_once __DIR__ . '/../includes/auth.php';
require_company_context_if_rto();

require_once __DIR__ . '/../includes/acl.php';
require_once __DIR__ . '/../includes/scope.php';
require_once __DIR__ . '/../includes/bunny.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  header('Location: ' . app_path('repos/repos_list.php'));
  exit;
}

if (!csrf_validate((string)($_POST['_csrf'] ?? ''))) {
  http_response_code(403);
  echo 'CSRF validation failed.';
  exit;
}

$pdo = db();
acl_bootstrap($pdo);

$tenantType = current_tenant_type(); // company|rto
$companyTenantId = company_context_id();
$rtoTenantId = (int) current_tenant_id();
$userId = (int) current_user_id();

$isCompanyAdmin = user_has_role('company_admin');
$isManager = user_has_role('manager');
$isDriver = user_has_role('driver');

// Inputs
$repoId = (int)($_POST['repo_id'] ?? 0);
$dropoffLocationId = (int)($_POST['dropoff_location_id'] ?? 0);
$miles = trim((string)($_POST['miles_pickup_to_dropoff'] ?? ''));
$repairsNeeded = (int)($_POST['repairs_needed'] ?? 0);
$repairsNotes = trim((string)($_POST['repairs_notes'] ?? ''));
$conditionNotes = trim((string)($_POST['condition_notes'] ?? ''));

$overrideTotal = trim((string)($_POST['override_total'] ?? '')); // optional
$overrideReason = trim((string)($_POST['override_reason'] ?? ''));

// Add-ons payload
$addonIds = $_POST['addon_ids'] ?? [];           // array of IDs (strings)
$addonAmount = $_POST['addon_amount'] ?? [];     // map addon_id => amount string

if ($repoId < 1 || $dropoffLocationId < 1) {
  http_response_code(400);
  echo "Missing repo_id or dropoff location.";
  exit;
}

$milesVal = null;
if ($miles !== '') {
  if (!is_numeric($miles)) {
    http_response_code(400);
    echo "Miles must be numeric.";
    exit;
  }
  $milesVal = round((float)$miles, 2);
  if ($milesVal < 0) $milesVal = 0;
}

function fetch_company_policy(PDO $pdo, int $companyTid): array {
  $billTo = 'rto';
  $driverCanOverride = 0;
  $driverCanCustom = 0;

  $stmt = $pdo->prepare("
    SELECT repo_invoice_bill_to, driver_can_override_repo_invoice, driver_can_submit_custom_invoice
    FROM tenant_settings
    WHERE tenant_id = :tid1
    LIMIT 1
  ");
  $stmt->execute([':tid1' => $companyTid]);
  if ($row = $stmt->fetch()) {
    $billTo = (string)$row['repo_invoice_bill_to'];
    $driverCanOverride = (int)$row['driver_can_override_repo_invoice'];
    $driverCanCustom = (int)$row['driver_can_submit_custom_invoice'];
  }

  return [$billTo, $driverCanOverride, $driverCanCustom];
}

/**
 * Load add-on fee types for this repo context.
 * Precedence: RTO-specific types override company types by NAME.
 */
function load_addon_types(PDO $pdo, int $companyTid, ?int $repoRtoTenantId): array {
  $params = [':tid1' => $companyTid];
  $where = "tenant_id = :tid1 AND is_active = 1 AND (applies_to='company')";

  if ($repoRtoTenantId) {
    $where = "tenant_id = :tid1 AND is_active = 1 AND (
      (applies_to='company')
      OR (applies_to='rto' AND rto_tenant_id = :rto1)
    )";
    $params[':rto1'] = $repoRtoTenantId;
  }

  $stmt = $pdo->prepare("
    SELECT id, name, description, default_amount, max_amount, driver_can_add, driver_can_override_amount, applies_to, rto_tenant_id
    FROM repo_addon_fee_types
    WHERE $where
    ORDER BY
      (applies_to='rto') DESC,  -- prefer rto versions first
      name ASC
  ");
  $stmt->execute($params);
  $rows = $stmt->fetchAll();

  // Deduplicate by name, preferring rto version
  $byName = [];
  foreach ($rows as $r) {
    $nm = (string)$r['name'];
    if (!isset($byName[$nm])) $byName[$nm] = $r;
  }

  // Return keyed by id for validation
  $out = [];
  foreach ($byName as $r) {
    $out[(int)$r['id']] = $r;
  }
  return $out;
}

try {
  $pdo->beginTransaction();

  // Load repo with tenant scope
  $where = "r.tenant_id = :tid1 AND r.id = :rid1";
  $params = [':tid1' => $companyTenantId, ':rid1' => $repoId];
  if ($tenantType === 'rto') {
    $where .= " AND r.rto_tenant_id = :rto1";
    $params[':rto1'] = $rtoTenantId;
  }

  $stmt = $pdo->prepare("
    SELECT r.*, a.width_ft, a.length_ft, a.stock_number
    FROM repos r
    JOIN assets a ON a.id = r.asset_id
    WHERE $where
    LIMIT 1
    FOR UPDATE
  ");
  $stmt->execute($params);
  $repo = $stmt->fetch();

  if (!$repo) {
    $pdo->rollBack();
    http_response_code(404);
    echo "Repo not found.";
    exit;
  }

  // Dropoff location must belong to company tenant
  $stmt = $pdo->prepare("
    SELECT id FROM locations
    WHERE tenant_id = :tid1 AND id = :lid1 AND status='active'
    LIMIT 1
  ");
  $stmt->execute([':tid1' => $companyTenantId, ':lid1' => $dropoffLocationId]);
  if (!$stmt->fetchColumn()) {
    $pdo->rollBack();
    http_response_code(400);
    echo "Invalid drop-off location.";
    exit;
  }

  // Policy lookup
  [$billTo, $driverCanOverrideInvoice, $driverCanCustom] = fetch_company_policy($pdo, $companyTenantId);

  // Apply link overrides if repo has rto_tenant_id
  $rtoLinkId = 0;
  if (!empty($repo['rto_tenant_id'])) {
    $stmt = $pdo->prepare("
      SELECT id
      FROM rto_company_links
      WHERE rto_tenant_id = :rto1 AND company_tenant_id = :co1 AND status='active'
      LIMIT 1
    ");
    $stmt->execute([':rto1' => (int)$repo['rto_tenant_id'], ':co1' => $companyTenantId]);
    $rtoLinkId = (int)($stmt->fetchColumn() ?: 0);

    if ($rtoLinkId) {
      $stmt = $pdo->prepare("
        SELECT repo_invoice_bill_to, driver_can_override_repo_invoice, driver_can_submit_custom_invoice
        FROM rto_company_policy_settings
        WHERE rto_company_link_id = :lid1
        LIMIT 1
      ");
      $stmt->execute([':lid1' => $rtoLinkId]);
      if ($ov = $stmt->fetch()) {
        if ($ov['repo_invoice_bill_to'] !== null) $billTo = (string)$ov['repo_invoice_bill_to'];
        if ($ov['driver_can_override_repo_invoice'] !== null) $driverCanOverrideInvoice = (int)$ov['driver_can_override_repo_invoice'];
        if ($ov['driver_can_submit_custom_invoice'] !== null) $driverCanCustom = (int)$ov['driver_can_submit_custom_invoice'];
      }
    }
  }

  // Required photos: building + serial
  $building = bunny_require_uploaded_image('building_photo');
  $serial   = bunny_require_uploaded_image('serial_photo');

  // Upload required to Bunny
  $buildingKey = bunny_repo_storage_key($companyTenantId, $repoId, 'building', $building['ext']);
  $serialKey   = bunny_repo_storage_key($companyTenantId, $repoId, 'serial', $serial['ext']);

  $buildingUp = bunny_upload_file($buildingKey, $building['tmp']);
  $serialUp   = bunny_upload_file($serialKey, $serial['tmp']);

  // Optional corner photos
  $cornerPhotos = bunny_optional_multi_images('corner_photos', ['jpg','jpeg','png','webp'], 10);

  // Store media
  $insMedia = $pdo->prepare("
    INSERT INTO repo_media (tenant_id, repo_id, media_type, storage_key, public_url, uploaded_by_user_id, created_at)
    VALUES (:tid1, :rid1, :mt1, :sk1, :url1, :uid1, CURRENT_TIMESTAMP)
  ");

  $insMedia->execute([
    ':tid1' => $companyTenantId,
    ':rid1' => $repoId,
    ':mt1' => 'building_photo',
    ':sk1' => $buildingUp['storage_key'],
    ':url1' => $buildingUp['public_url'],
    ':uid1' => $userId,
  ]);

  $insMedia->execute([
    ':tid1' => $companyTenantId,
    ':rid1' => $repoId,
    ':mt1' => 'serial_photo',
    ':sk1' => $serialUp['storage_key'],
    ':url1' => $serialUp['public_url'],
    ':uid1' => $userId,
  ]);

  $cornerCount = 0;
  foreach ($cornerPhotos as $cp) {
    $cornerKey = bunny_repo_storage_key($companyTenantId, $repoId, 'corner', $cp['ext']);
    $up = bunny_upload_file($cornerKey, $cp['tmp']);
    $cornerCount++;

    $insMedia->execute([
      ':tid1' => $companyTenantId,
      ':rid1' => $repoId,
      ':mt1' => 'corner_photo',
      ':sk1' => $up['storage_key'],
      ':url1' => $up['public_url'],
      ':uid1' => $userId,
    ]);
  }

  // Save dropoff
  $stmt = $pdo->prepare("
    INSERT INTO repo_dropoffs (tenant_id, repo_id, dropoff_location_id, created_at)
    VALUES (:tid1, :rid1, :loc1, CURRENT_TIMESTAMP)
    ON DUPLICATE KEY UPDATE dropoff_location_id=VALUES(dropoff_location_id)
  ");
  $stmt->execute([':tid1' => $companyTenantId, ':rid1' => $repoId, ':loc1' => $dropoffLocationId]);

  // Save pickup record
  $pickedUpAt = date('Y-m-d H:i:s');
  $stmt = $pdo->prepare("
    INSERT INTO repo_pickups
      (tenant_id, repo_id, picked_up_at, miles_pickup_to_dropoff, repairs_needed, repairs_notes, condition_notes, created_by_user_id, created_at, updated_at)
    VALUES
      (:tid1, :rid1, :p1, :m1, :rn1, :rnotes1, :cnotes1, :uid1, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
    ON DUPLICATE KEY UPDATE
      picked_up_at=VALUES(picked_up_at),
      miles_pickup_to_dropoff=VALUES(miles_pickup_to_dropoff),
      repairs_needed=VALUES(repairs_needed),
      repairs_notes=VALUES(repairs_notes),
      condition_notes=VALUES(condition_notes),
      updated_at=CURRENT_TIMESTAMP
  ");
  $stmt->execute([
    ':tid1' => $companyTenantId,
    ':rid1' => $repoId,
    ':p1' => $pickedUpAt,
    ':m1' => $milesVal,
    ':rn1' => ($repairsNeeded ? 1 : 0),
    ':rnotes1' => ($repairsNotes === '' ? null : $repairsNotes),
    ':cnotes1' => ($conditionNotes === '' ? null : $conditionNotes),
    ':uid1' => $userId,
  ]);

  // Update repo status
  $stmt = $pdo->prepare("
    UPDATE repos
    SET status='picked_up',
        picked_up_at=COALESCE(picked_up_at, :p1),
        updated_at=CURRENT_TIMESTAMP
    WHERE tenant_id=:tid1 AND id=:rid1
  ");
  $stmt->execute([':p1' => $pickedUpAt, ':tid1' => $companyTenantId, ':rid1' => $repoId]);

  // Create repair if needed
  if ($repairsNeeded) {
    $stmt = $pdo->prepare("SELECT id FROM repairs WHERE tenant_id=:tid1 AND repo_id=:rid1 LIMIT 1");
    $stmt->execute([':tid1' => $companyTenantId, ':rid1' => $repoId]);
    $existingRepair = (int)($stmt->fetchColumn() ?: 0);

    if (!$existingRepair) {
      $stmt = $pdo->prepare("
        INSERT INTO repairs
          (tenant_id, rto_tenant_id, asset_id, repo_id, location_id, status, priority, source, created_by_user_id, created_at, updated_at)
        VALUES
          (:tid1, :rto1, :aid1, :rid1, NULL, 'new', 'high', 'auto_from_repo_pickup', :uid1, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
      ");
      $stmt->execute([
        ':tid1' => $companyTenantId,
        ':rto1' => $repo['rto_tenant_id'] ?? null,
        ':aid1' => (int)$repo['asset_id'],
        ':rid1' => $repoId,
        ':uid1' => $userId,
      ]);
    }
  }

  // Choose schedule (prefer RTO)
  $scheduleId = null;

  if (!empty($repo['rto_tenant_id'])) {
    $stmt = $pdo->prepare("
      SELECT id
      FROM repo_fee_schedules
      WHERE tenant_id=:tid1 AND applies_to='rto' AND rto_tenant_id=:rto1 AND is_active=1
      ORDER BY id DESC
      LIMIT 1
    ");
    $stmt->execute([':tid1' => $companyTenantId, ':rto1' => (int)$repo['rto_tenant_id']]);
    $scheduleId = $stmt->fetchColumn() ?: null;
  }

  if (!$scheduleId) {
    $stmt = $pdo->prepare("
      SELECT id
      FROM repo_fee_schedules
      WHERE tenant_id=:tid1 AND applies_to='company' AND is_active=1
      ORDER BY id DESC
      LIMIT 1
    ");
    $stmt->execute([':tid1' => $companyTenantId]);
    $scheduleId = $stmt->fetchColumn() ?: null;
  }

  if (!$scheduleId) {
    $pdo->rollBack();
    http_response_code(500);
    echo "No active repo fee schedule found.";
    exit;
  }

  // Load schedule
  $stmt = $pdo->prepare("
    SELECT included_miles, max_billable_miles, per_mile_fee
    FROM repo_fee_schedules
    WHERE id=:sid1 AND tenant_id=:tid1
    LIMIT 1
  ");
  $stmt->execute([':sid1' => (int)$scheduleId, ':tid1' => $companyTenantId]);
  $sched = $stmt->fetch();
  if (!$sched) {
    $pdo->rollBack();
    http_response_code(500);
    echo "Invalid fee schedule.";
    exit;
  }

  $includedMiles = (int)$sched['included_miles'];
  $maxMiles = (int)$sched['max_billable_miles'];
  $perMile = (float)$sched['per_mile_fee'];

  $widthFt = (int)$repo['width_ft'];
  $lengthFt = (int)$repo['length_ft'];

  $stmt = $pdo->prepare("
    SELECT base_fee, base_length_max_ft, extra_length_step_ft, extra_length_fee_per_step
    FROM repo_fee_width_rules
    WHERE schedule_id=:sid1 AND width_ft=:w1
    LIMIT 1
  ");
  $stmt->execute([':sid1' => (int)$scheduleId, ':w1' => $widthFt]);
  $rule = $stmt->fetch();

  if (!$rule) {
    $pdo->rollBack();
    http_response_code(500);
    echo "No width rule found for {$widthFt}ft wide on schedule.";
    exit;
  }

  $baseFee = (float)$rule['base_fee'];
  $baseLenMax = (int)$rule['base_length_max_ft'];
  $stepFt = (int)$rule['extra_length_step_ft'];
  $stepFee = (float)$rule['extra_length_fee_per_step'];

  $extraLenFee = 0.0;
  if ($lengthFt > $baseLenMax && $stepFt > 0 && $stepFee > 0) {
    $over = $lengthFt - $baseLenMax;
    $steps = (int)ceil($over / $stepFt);
    $extraLenFee = $steps * $stepFee;
  }

  $milesTotal = $milesVal;
  $milesBillable = 0.0;
  $mileageFee = 0.0;

  if ($milesTotal !== null) {
    $billableRange = max($maxMiles - $includedMiles, 0);
    $milesBillable = min(max($milesTotal - $includedMiles, 0), $billableRange);
    $mileageFee = $milesBillable * $perMile;
  }

  $billedTo = $billTo;
  if ($billedTo === 'rto' && empty($repo['rto_tenant_id'])) $billedTo = 'company';

  // Load allowed add-ons (dedup by name; prefer RTO-specific)
  $addonTypesById = load_addon_types($pdo, $companyTenantId, !empty($repo['rto_tenant_id']) ? (int)$repo['rto_tenant_id'] : null);

  // Normalize selected addon IDs
  $selectedAddonIds = [];
  if (is_array($addonIds)) {
    foreach ($addonIds as $raw) {
      $id = (int)$raw;
      if ($id > 0) $selectedAddonIds[$id] = true;
    }
  }

  // Create/replace invoice header
  $stmt = $pdo->prepare("
    INSERT INTO repo_invoices
      (tenant_id, repo_id, schedule_id, billed_to, status, miles_total, miles_billable, subtotal, total,
       overridden, override_reason, submitted_by_user_id, approved_by_user_id, approval_reason,
       created_by_user_id, created_at, updated_at)
    VALUES
      (:tid1, :rid1, :sid1, :bt1, 'draft', :mt1, :mb1, 0.00, 0.00,
       0, NULL, NULL, NULL, NULL,
       :uid1, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
    ON DUPLICATE KEY UPDATE
      schedule_id=VALUES(schedule_id),
      billed_to=VALUES(billed_to),
      status='draft',
      miles_total=VALUES(miles_total),
      miles_billable=VALUES(miles_billable),
      subtotal=0.00,
      total=0.00,
      overridden=0,
      override_reason=NULL,
      submitted_by_user_id=NULL,
      approved_by_user_id=NULL,
      approval_reason=NULL,
      updated_at=CURRENT_TIMESTAMP
  ");
  $stmt->execute([
    ':tid1' => $companyTenantId,
    ':rid1' => $repoId,
    ':sid1' => (int)$scheduleId,
    ':bt1' => $billedTo,
    ':mt1' => $milesTotal,
    ':mb1' => $milesBillable,
    ':uid1' => $userId,
  ]);

  $stmt = $pdo->prepare("SELECT id FROM repo_invoices WHERE tenant_id=:tid1 AND repo_id=:rid1 LIMIT 1");
  $stmt->execute([':tid1' => $companyTenantId, ':rid1' => $repoId]);
  $invoiceId = (int)$stmt->fetchColumn();

  // Clear invoice items
  $stmt = $pdo->prepare("DELETE FROM repo_invoice_items WHERE tenant_id=:tid1 AND repo_invoice_id=:iid1");
  $stmt->execute([':tid1' => $companyTenantId, ':iid1' => $invoiceId]);

  // Item inserter
  $insItem = $pdo->prepare("
    INSERT INTO repo_invoice_items
      (tenant_id, repo_invoice_id, item_type, addon_fee_type_id, description, quantity, unit_price, amount, added_by_user_id, created_at)
    VALUES
      (:tid1, :iid1, :type1, :addon1, :desc1, :qty1, :unit1, :amt1, :uid1, CURRENT_TIMESTAMP)
  ");

  $subtotal = 0.0;

  // Base
  $subtotal += $baseFee;
  $insItem->execute([
    ':tid1' => $companyTenantId,
    ':iid1' => $invoiceId,
    ':type1' => 'base',
    ':addon1' => null,
    ':desc1' => "Base Repo Fee ({$widthFt}ft wide up to {$baseLenMax}ft)",
    ':qty1' => 1,
    ':unit1' => $baseFee,
    ':amt1' => $baseFee,
    ':uid1' => $userId,
  ]);

  // Extra length
  if ($extraLenFee > 0) {
    $subtotal += $extraLenFee;
    $insItem->execute([
      ':tid1' => $companyTenantId,
      ':iid1' => $invoiceId,
      ':type1' => 'length_extra',
      ':addon1' => null,
      ':desc1' => "Extra Length Fee (over {$baseLenMax}ft)",
      ':qty1' => 1,
      ':unit1' => $extraLenFee,
      ':amt1' => $extraLenFee,
      ':uid1' => $userId,
    ]);
  }

  // Mileage
  if ($mileageFee > 0) {
    $subtotal += $mileageFee;
    $insItem->execute([
      ':tid1' => $companyTenantId,
      ':iid1' => $invoiceId,
      ':type1' => 'mileage',
      ':addon1' => null,
      ':desc1' => "Mileage ({$milesBillable} billable miles @ $" . number_format($perMile,2) . "/mi)",
      ':qty1' => $milesBillable,
      ':unit1' => $perMile,
      ':amt1' => $mileageFee,
      ':uid1' => $userId,
    ]);
  }

  // Add-ons
  $addonTotal = 0.0;

  foreach ($selectedAddonIds as $aid => $_true) {
    if (!isset($addonTypesById[$aid])) continue;
    $t = $addonTypesById[$aid];

    // Driver can only add if driver_can_add=1
    if ($isDriver && !$isCompanyAdmin && !$isManager) {
      if ((int)$t['driver_can_add'] !== 1) continue;
    }

    $defaultAmt = (float)$t['default_amount'];
    $maxAmt = (float)$t['max_amount'];
    $driverCanOverrideAmt = (int)$t['driver_can_override_amount'];

    $amt = $defaultAmt;

    // if amount provided, treat as override attempt
    $amtRaw = isset($addonAmount[(string)$aid]) ? trim((string)$addonAmount[(string)$aid]) : '';
    if ($amtRaw !== '') {
      if (!is_numeric($amtRaw) || (float)$amtRaw < 0) {
        $pdo->rollBack();
        http_response_code(400);
        echo "Invalid amount for add-on: " . htmlspecialchars((string)$t['name']);
        exit;
      }
      $requested = round((float)$amtRaw, 2);

      // Enforce max rule (if max_amount > 0)
      if ($maxAmt > 0 && $requested > $maxAmt) {
        $pdo->rollBack();
        http_response_code(400);
        echo "Add-on amount exceeds max for: " . htmlspecialchars((string)$t['name']);
        exit;
      }

      // Company admin/manager can override; driver only if allowed
      if ($isCompanyAdmin || $isManager) {
        // If maxAmt=0, only allow default
        if ($maxAmt <= 0 && $requested != $defaultAmt) {
          $pdo->rollBack();
          http_response_code(400);
          echo "Add-on amount cannot be overridden for: " . htmlspecialchars((string)$t['name']);
          exit;
        }
        $amt = $requested;
      } else {
        // Driver path
        if ($requested != $defaultAmt) {
          if ($driverCanOverrideAmt !== 1) {
            $pdo->rollBack();
            http_response_code(403);
            echo "Driver is not allowed to override amount for: " . htmlspecialchars((string)$t['name']);
            exit;
          }
          if ($maxAmt <= 0) {
            $pdo->rollBack();
            http_response_code(403);
            echo "Add-on amount cannot be overridden for: " . htmlspecialchars((string)$t['name']);
            exit;
          }
        }
        $amt = $requested;
      }
    } else {
      // no amount field given: use default
      $amt = $defaultAmt;
    }

    $addonTotal += $amt;

    $insItem->execute([
      ':tid1' => $companyTenantId,
      ':iid1' => $invoiceId,
      ':type1' => 'addon',
      ':addon1' => (int)$aid,
      ':desc1' => (string)$t['name'],
      ':qty1' => 1,
      ':unit1' => $amt,
      ':amt1' => $amt,
      ':uid1' => $userId,
    ]);
  }

  $subtotal += $addonTotal;
  $total = $subtotal;

  // Total override (requires reason)
  if ($overrideTotal !== '') {
    if (!is_numeric($overrideTotal) || (float)$overrideTotal < 0) {
      $pdo->rollBack();
      http_response_code(400);
      echo "Override total must be numeric.";
      exit;
    }
    $overrideTotalVal = round((float)$overrideTotal, 2);
    if ($overrideReason === '') {
      $pdo->rollBack();
      http_response_code(400);
      echo "Override reason is required.";
      exit;
    }

    if ($isCompanyAdmin || $isManager) {
      // immediate approve
      $stmt = $pdo->prepare("
        UPDATE repo_invoices
        SET overridden=1,
            override_reason=:or1,
            status='approved',
            approved_by_user_id=:ab1,
            approval_reason=:ar1,
            subtotal=:sub1,
            total=:tot1,
            updated_at=CURRENT_TIMESTAMP
        WHERE tenant_id=:tid1 AND id=:iid1
      ");
      $stmt->execute([
        ':or1' => $overrideReason,
        ':ab1' => $userId,
        ':ar1' => 'Approved by admin/manager at pickup submission.',
        ':sub1' => $subtotal,
        ':tot1' => $overrideTotalVal,
        ':tid1' => $companyTenantId,
        ':iid1' => $invoiceId,
      ]);

      // record override line item
      $insItem->execute([
        ':tid1' => $companyTenantId,
        ':iid1' => $invoiceId,
        ':type1' => 'custom',
        ':addon1' => null,
        ':desc1' => "OVERRIDE TOTAL (Approved) — Reason: " . mb_substr($overrideReason, 0, 200),
        ':qty1' => 1,
        ':unit1' => $overrideTotalVal,
        ':amt1' => $overrideTotalVal,
        ':uid1' => $userId,
      ]);

      $total = $overrideTotalVal;

    } else {
      // driver override only if allowed
      if (!$isDriver || $driverCanOverrideInvoice !== 1) {
        $pdo->rollBack();
        http_response_code(403);
        echo "You are not allowed to override invoice totals.";
        exit;
      }

      $stmt = $pdo->prepare("
        UPDATE repo_invoices
        SET overridden=1,
            override_reason=:or1,
            status='submitted',
            submitted_by_user_id=:sb1,
            subtotal=:sub1,
            total=:tot1,
            updated_at=CURRENT_TIMESTAMP
        WHERE tenant_id=:tid1 AND id=:iid1
      ");
      $stmt->execute([
        ':or1' => $overrideReason,
        ':sb1' => $userId,
        ':sub1' => $subtotal,
        ':tot1' => $overrideTotalVal,
        ':tid1' => $companyTenantId,
        ':iid1' => $invoiceId,
      ]);

      $insItem->execute([
        ':tid1' => $companyTenantId,
        ':iid1' => $invoiceId,
        ':type1' => 'custom',
        ':addon1' => null,
        ':desc1' => "OVERRIDE TOTAL (Submitted) — Reason: " . mb_substr($overrideReason, 0, 200),
        ':qty1' => 1,
        ':unit1' => $overrideTotalVal,
        ':amt1' => $overrideTotalVal,
        ':uid1' => $userId,
      ]);

      $total = $overrideTotalVal;
    }
  } else {
    // Normal totals
    $stmt = $pdo->prepare("
      UPDATE repo_invoices
      SET subtotal=:sub1, total=:tot1, updated_at=CURRENT_TIMESTAMP
      WHERE tenant_id=:tid1 AND id=:iid1
    ");
    $stmt->execute([
      ':sub1' => $subtotal,
      ':tot1' => $total,
      ':tid1' => $companyTenantId,
      ':iid1' => $invoiceId,
    ]);
  }

  // Log event
  $meta = [
    'picked_up' => true,
    'dropoff_location_id' => $dropoffLocationId,
    'miles' => $milesVal,
    'repairs_needed' => (bool)$repairsNeeded,
    'invoice_id' => $invoiceId,
    'addon_total' => $addonTotal,
    'addon_count' => count($selectedAddonIds),
    'corner_photos_uploaded' => $cornerCount
  ];

  $stmt = $pdo->prepare("
    INSERT INTO repo_events (tenant_id, repo_id, event_type, meta_json, created_by_user_id, created_at)
    VALUES (:tid1, :rid1, 'pickup_package_submitted', :m1, :uid1, CURRENT_TIMESTAMP)
  ");
  $stmt->execute([
    ':tid1' => $companyTenantId,
    ':rid1' => $repoId,
    ':m1' => json_encode($meta, JSON_UNESCAPED_SLASHES),
    ':uid1' => $userId,
  ]);

  $pdo->commit();

  header('Location: ' . app_path('repos/repo_view.php?id=' . $repoId . '&ok=1'));
  exit;

} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  http_response_code(500);
  echo "Failed to confirm pickup: " . htmlspecialchars($e->getMessage());
  exit;
}
