<?php
// /app/repos/invoice_approve.php
declare(strict_types=1);

require_once __DIR__ . '/../includes/auth.php';
require_company_context_if_rto();

require_once __DIR__ . '/../includes/acl.php';
require_once __DIR__ . '/../includes/scope.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  header('Location: ' . app_path('repos/repos_list.php'));
  exit;
}

if (!csrf_validate((string)($_POST['_csrf'] ?? ''))) {
  http_response_code(403);
  echo 'CSRF validation failed.';
  exit;
}

$pdo = db();
acl_bootstrap($pdo);

$tenantType = current_tenant_type();
$companyTenantId = company_context_id();
$userId = (int) current_user_id();

$isCompanyAdmin = user_has_role('company_admin');
$isManager = user_has_role('manager');

if ($tenantType === 'rto') {
  http_response_code(403);
  echo "RTO accounts cannot approve invoices.";
  exit;
}

if (!$isCompanyAdmin && !$isManager) {
  http_response_code(403);
  echo "Only admin/manager can approve invoices.";
  exit;
}

$repoId = (int)($_POST['repo_id'] ?? 0);
$invoiceId = (int)($_POST['invoice_id'] ?? 0);
$approvalReason = trim((string)($_POST['approval_reason'] ?? ''));

if ($repoId < 1 || $invoiceId < 1) {
  http_response_code(400);
  echo "Missing repo_id or invoice_id.";
  exit;
}

if ($approvalReason === '') {
  http_response_code(400);
  echo "Approval reason is required.";
  exit;
}

try {
  $pdo->beginTransaction();

  // Lock invoice
  $stmt = $pdo->prepare("
    SELECT *
    FROM repo_invoices
    WHERE tenant_id = :tid1 AND id = :iid1 AND repo_id = :rid1
    LIMIT 1
    FOR UPDATE
  ");
  $stmt->execute([
    ':tid1' => $companyTenantId,
    ':iid1' => $invoiceId,
    ':rid1' => $repoId,
  ]);
  $inv = $stmt->fetch();

  if (!$inv) {
    $pdo->rollBack();
    http_response_code(404);
    echo "Invoice not found.";
    exit;
  }

  if ((string)$inv['status'] !== 'submitted') {
    $pdo->rollBack();
    http_response_code(400);
    echo "Invoice is not in 'submitted' status.";
    exit;
  }

  // Approve invoice
  $stmt = $pdo->prepare("
    UPDATE repo_invoices
    SET status = 'approved',
        approved_by_user_id = :uid1,
        approval_reason = :ar1,
        updated_at = CURRENT_TIMESTAMP
    WHERE tenant_id = :tid1 AND id = :iid1
  ");
  $stmt->execute([
    ':uid1' => $userId,
    ':ar1' => $approvalReason,
    ':tid1' => $companyTenantId,
    ':iid1' => $invoiceId,
  ]);

  // Log repo event
  $meta = [
    'invoice_id' => $invoiceId,
    'action' => 'invoice_approved',
    'approved_by_user_id' => $userId,
    'approval_reason' => $approvalReason,
  ];

  $stmt = $pdo->prepare("
    INSERT INTO repo_events (tenant_id, repo_id, event_type, meta_json, created_by_user_id, created_at)
    VALUES (:tid1, :rid1, 'invoice_approved', :m1, :uid1, CURRENT_TIMESTAMP)
  ");
  $stmt->execute([
    ':tid1' => $companyTenantId,
    ':rid1' => $repoId,
    ':m1' => json_encode($meta, JSON_UNESCAPED_SLASHES),
    ':uid1' => $userId,
  ]);

  $pdo->commit();

  header('Location: ' . app_path('repos/repo_view.php?id=' . $repoId . '&ok=1'));
  exit;

} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  http_response_code(500);
  echo "Invoice approval failed: " . htmlspecialchars($e->getMessage());
  exit;
}
