<?php
// /app/repairs/repairs_list.php
declare(strict_types=1);

ini_set('display_errors', '0'); // keep off so we don't leak errors publicly
ini_set('log_errors', '1');
ini_set('error_log', __DIR__ . '/repos_debug.log');
error_reporting(E_ALL);

error_log("---- REPOS_LIST HIT " . date('c') . " ----");
error_log("URI=" . ($_SERVER['REQUEST_URI'] ?? ''));
error_log("HTTPS=" . ($_SERVER['HTTPS'] ?? ''));

require_once __DIR__ . '/../includes/auth.php';
error_log("loaded auth.php");
require_company_context_if_rto();
error_log("passed require_company_context_if_rto");

require_once __DIR__ . '/../includes/acl.php';
error_log("loaded acl.php");
require_once __DIR__ . '/../includes/scope.php';
error_log("loaded scope.php");

$pageTitle = 'Repairs';
$activeNav = 'repairs';

$pdo = db();
acl_bootstrap($pdo);

$user = current_user();

$tenantType = current_tenant_type(); // company|rto
$companyTenantId = company_context_id();

$isCompanyAdmin = user_has_role('company_admin');
$isManager      = user_has_role('manager');
$isTech         = user_has_role('repair_tech');

// Default repair techs to "Assigned to me" unless explicitly set
$assignedToMe = isset($_GET['mine'])
  ? (int)($_GET['mine'] ?? 0)
  : ($isTech ? 1 : 0);

$status = trim((string)($_GET['status'] ?? ''));
$q      = trim((string)($_GET['q'] ?? ''));

// Company-only filter
$techId = (int)($_GET['tech_id'] ?? 0);

// Validate status
$validStatuses = ['new','assigned','in_progress','waiting_parts','completed','canceled'];
if ($status !== '' && !in_array($status, $validStatuses, true)) $status = '';

/**
 * Base WHERE from centralized scope (tenant + rto + tech assigned-only)
 * apply_repair_scope uses alias rp.
 */
$where = [];
$params = [];
apply_repair_scope($where, $params);

// Manual filters
if ($status !== '') {
  $where[] = "rp.status = :status";
  $params[':status'] = $status;
}

if ($q !== '') {
  $where[] = "(a.stock_number LIKE :q OR a.serial_number LIKE :q OR CAST(rp.id AS CHAR) LIKE :q OR CAST(rp.repo_id AS CHAR) LIKE :q)";
  $params[':q'] = '%' . $q . '%';
}

/**
 * Assigned-to-me override (company only)
 */
if ($tenantType !== 'rto' && $assignedToMe === 1) {
  $where[] = "rp.assigned_tech_user_id = :mine_tech";
  $params[':mine_tech'] = (int) current_user_id();
}

/**
 * Tech filter:
 * - company only
 * - only if user can view all repairs and not forcing mine=1
 */
$canViewAllRepairs = $isCompanyAdmin || $isManager || user_can('repairs.view_all');
if ($tenantType !== 'rto' && $techId > 0 && $assignedToMe !== 1 && $canViewAllRepairs) {
  $where[] = "rp.assigned_tech_user_id = :tech_id";
  $params[':tech_id'] = $techId;
}

$whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

/**
 * Tech dropdown (role-filtered) for company users that can view all
 * IMPORTANT: unique placeholders (:tid1, :tid2)
 */
$techs = [];
if ($tenantType !== 'rto' && $canViewAllRepairs) {
  $stmt = $pdo->prepare("
    SELECT u.id, CONCAT(COALESCE(u.first_name,''), ' ', COALESCE(u.last_name,'')) AS name
    FROM users u
    JOIN user_roles ur ON ur.user_id = u.id
    JOIN roles r ON r.id = ur.role_id
    WHERE u.tenant_id = :tid1
      AND u.status = 'active'
      AND r.tenant_id = :tid2
      AND r.name = 'repair_tech'
    ORDER BY name ASC
    LIMIT 300
  ");
  $stmt->execute([
    ':tid1' => $companyTenantId,
    ':tid2' => $companyTenantId,
  ]);
  $techs = $stmt->fetchAll();
}

function status_pill_repair(string $status): string {
  return match ($status) {
    'new' => 'blue',
    'assigned', 'waiting_parts' => 'orange',
    'in_progress' => 'red',
    'completed', 'canceled' => 'gray',
    default => 'gray'
  };
}

function priority_pill_repair(string $p): string {
  return match ($p) {
    'high' => 'red',
    'medium' => 'orange',
    'low' => 'blue',
    default => 'gray'
  };
}

$sql = "
  SELECT
    rp.id,
    rp.status,
    rp.priority,
    rp.source,
    rp.repo_id,
    rp.created_at,
    rp.updated_at,
    rp.assigned_tech_user_id,
    a.stock_number,
    a.asset_type,
    a.length_ft,
    a.width_ft,
    t_rto.name AS rto_name,
    CONCAT(COALESCE(u.first_name,''), ' ', COALESCE(u.last_name,'')) AS tech_name
  FROM repairs rp
  JOIN assets a ON a.id = rp.asset_id
  LEFT JOIN tenants t_rto ON t_rto.id = rp.rto_tenant_id
  LEFT JOIN users u ON u.id = rp.assigned_tech_user_id
  $whereSql
  ORDER BY
    FIELD(rp.status,'new','assigned','in_progress','waiting_parts','completed','canceled'),
    FIELD(rp.priority,'high','medium','low'),
    rp.updated_at DESC
  LIMIT 250
";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$rows = $stmt->fetchAll();

require_once __DIR__ . '/../includes/header.php';
?>
<div class="container-fluid">

  <div class="d-flex flex-wrap justify-content-between align-items-end gap-2 mb-3">
    <div>
      <div class="so-page-title">Repairs</div>
      <div class="so-page-sub">
        <?= h($tenantType === 'rto' ? ('RTO View: ' . ($user['active_company_name'] ?? '')) : 'Company View') ?>
      </div>
    </div>
    <div class="d-flex gap-2">
      <a class="btn so-btn" href="<?= app_path('repairs/repairs_list.php') ?>"><i class="fa-solid fa-rotate me-2"></i>Refresh</a>
      <a class="btn btn-outline-dark" href="#" style="border-radius:12px;"><i class="fa-solid fa-plus me-2"></i>New Repair</a>
    </div>
  </div>

  <div class="so-card mb-3">
    <div class="so-card-body">
      <form class="row g-2 align-items-end" method="get">

        <div class="col-12 col-md-3">
          <label class="form-label fw-bold">Status</label>
          <select name="status" class="form-select" style="border-radius:12px;">
            <option value="">All</option>
            <?php foreach ($validStatuses as $s): ?>
              <option value="<?= h($s) ?>" <?= $status===$s?'selected':'' ?>>
                <?= h(strtoupper(str_replace('_',' ', $s))) ?>
              </option>
            <?php endforeach; ?>
          </select>
        </div>

        <div class="col-12 col-md-4">
          <label class="form-label fw-bold">Search</label>
          <input type="text" name="q" value="<?= h($q) ?>" class="form-control" style="border-radius:12px;" placeholder="Stock #, Repair ID, Repo ID...">
        </div>

        <div class="col-12 col-md-3">
          <label class="form-label fw-bold">Technician</label>
          <select name="tech_id" class="form-select" style="border-radius:12px;" <?= ($tenantType==='rto' || !$canViewAllRepairs) ? 'disabled' : '' ?>>
            <option value="0">All</option>
            <?php foreach ($techs as $t): ?>
              <?php $tid = (int)$t['id']; ?>
              <option value="<?= $tid ?>" <?= ($techId===$tid)?'selected':'' ?>>
                <?= h(trim((string)$t['name']) ?: ('User #'.$tid)) ?>
              </option>
            <?php endforeach; ?>
          </select>
          <?php if ($tenantType==='rto'): ?>
            <div class="text-muted small mt-1">Technician filter is available for company users only.</div>
          <?php elseif (!$canViewAllRepairs): ?>
            <div class="text-muted small mt-1">Technician filter requires view-all access.</div>
          <?php endif; ?>
        </div>

        <div class="col-12 col-md-2">
          <div class="form-check" style="padding-top: 10px;">
            <input class="form-check-input" type="checkbox" value="1" id="mine" name="mine" <?= ($assignedToMe===1)?'checked':'' ?>>
            <label class="form-check-label fw-bold" for="mine">
              Assigned to me
            </label>
          </div>
        </div>

        <div class="col-12 d-flex gap-2 mt-2">
          <button class="btn btn-dark" style="border-radius:12px;" type="submit">
            <i class="fa-solid fa-filter me-2"></i>Apply Filters
          </button>
          <a class="btn btn-outline-dark" style="border-radius:12px;" href="<?= app_path('repairs/repairs_list.php') ?>">
            Clear
          </a>
        </div>

      </form>
    </div>
  </div>

  <div class="so-card">
    <div class="so-card-h">
      <h5 class="so-card-title">
        <i class="fa-solid fa-screwdriver-wrench me-2" style="color:#cf1f1f"></i>
        Repair Queue
      </h5>
      <div class="text-muted small">Showing up to 250 results</div>
    </div>

    <div class="so-card-body">
      <div class="table-responsive">
        <table class="table so-table align-middle">
          <thead>
            <tr>
              <th>Repair</th>
              <th>RTO Company</th>
              <th>Asset</th>
              <th>Source</th>
              <th>Technician</th>
              <th>Status</th>
              <th class="text-end">Priority</th>
            </tr>
          </thead>
          <tbody>
          <?php if (!$rows): ?>
            <tr>
              <td colspan="7" class="text-muted">No repairs found for the selected filters.</td>
            </tr>
          <?php else: ?>
            <?php foreach ($rows as $r): ?>
              <?php
                $assetLabel = trim(($r['width_ft'] ? $r['width_ft'].'x' : '') . ($r['length_ft'] ?? ''));
                $assetLabel = $assetLabel ?: strtoupper((string)$r['asset_type']);
                $techName = trim((string)$r['tech_name']);
              ?>
              <tr style="cursor:pointer;" onclick="window.location.href='<?= app_path('repairs/repair_view.php?id='.(int)$r['id']) ?>'">
                <td>
                  <strong>#<?= (int)$r['id'] ?></strong>
                  <?php if (!empty($r['repo_id'])): ?>
                    <div class="text-muted small">Repo #<?= (int)$r['repo_id'] ?></div>
                  <?php else: ?>
                    <div class="text-muted small">No repo link</div>
                  <?php endif; ?>
                </td>

                <td><strong><?= h($r['rto_name'] ?? '—') ?></strong></td>

                <td>
                  <strong><?= h((string)$r['stock_number']) ?></strong>
                  <div class="text-muted small"><?= h($assetLabel) ?></div>
                </td>

                <td><span class="so-pill gray"><?= h(str_replace('_',' ', (string)$r['source'])) ?></span></td>

                <td><?= h($techName ?: '—') ?></td>

                <td>
                  <span class="so-pill <?= h(status_pill_repair((string)$r['status'])) ?>">
                    <?= h(strtoupper(str_replace('_',' ', (string)$r['status']))) ?>
                  </span>
                </td>

                <td class="text-end">
                  <span class="so-pill <?= h(priority_pill_repair((string)$r['priority'])) ?>">
                    <?= h(strtoupper((string)$r['priority'])) ?>
                  </span>
                </td>
              </tr>
            <?php endforeach; ?>
          <?php endif; ?>
          </tbody>
        </table>
      </div>
    </div>
  </div>

</div>
<?php require_once __DIR__ . '/../includes/footer.php'; ?>
