<?php
// /app/repairs/repair_update_status.php
declare(strict_types=1);

require_once __DIR__ . '/../includes/auth.php';
require_company_context_if_rto();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  header('Location: ' . app_path('repairs/repairs_list.php'));
  exit;
}

$token = (string)($_POST['_csrf'] ?? '');
if (!csrf_validate($token)) {
  http_response_code(403);
  echo 'CSRF validation failed.';
  exit;
}

$repairId = (int)($_POST['repair_id'] ?? 0);
$newStatus = trim((string)($_POST['status'] ?? ''));
$newPriority = trim((string)($_POST['priority'] ?? ''));
$eventNote = trim((string)($_POST['event_note'] ?? ''));

$validStatuses = ['new','assigned','in_progress','waiting_parts','completed','canceled'];
$validPriorities = ['high','medium','low'];

if ($repairId < 1 || !in_array($newStatus, $validStatuses, true) || !in_array($newPriority, $validPriorities, true)) {
  header('Location: ' . app_path('repairs/repairs_list.php'));
  exit;
}

$pdo = db();
$userId = (int) current_user_id();

$tenantType = current_tenant_type();
$rtoTenantId = (int) current_tenant_id();
$companyTenantId = ($tenantType === 'rto')
  ? (int) active_company_tenant_id()
  : (int) current_tenant_id();

try {
  $pdo->beginTransaction();

  $where = "tenant_id = :company_tid AND id = :rid";
  $params = [':company_tid' => $companyTenantId, ':rid' => $repairId];

  if ($tenantType === 'rto') {
    $where .= " AND rto_tenant_id = :rto_tid";
    $params[':rto_tid'] = $rtoTenantId;
  }

  // Lock row
  $stmt = $pdo->prepare("SELECT * FROM repairs WHERE $where FOR UPDATE");
  $stmt->execute($params);
  $repair = $stmt->fetch();

  if (!$repair) {
    $pdo->rollBack();
    header('Location: ' . app_path('repairs/repairs_list.php'));
    exit;
  }

  $oldStatus = (string)$repair['status'];
  $oldPriority = (string)$repair['priority'];

  $stmt = $pdo->prepare("
    UPDATE repairs
    SET status = :status,
        priority = :priority,
        updated_at = CURRENT_TIMESTAMP
    WHERE tenant_id = :company_tid AND id = :rid
  ");
  $stmt->execute([
    ':status' => $newStatus,
    ':priority' => $newPriority,
    ':company_tid' => $companyTenantId,
    ':rid' => $repairId,
  ]);

  // Log repair event
  $meta = [
    'from_status' => $oldStatus,
    'to_status' => $newStatus,
    'from_priority' => $oldPriority,
    'to_priority' => $newPriority,
  ];
  if ($eventNote !== '') $meta['note'] = $eventNote;

  $stmt = $pdo->prepare("
    INSERT INTO repair_events (tenant_id, repair_id, event_type, meta_json, created_by_user_id, created_at)
    VALUES (:tid, :rid, :etype, :meta, :uid, CURRENT_TIMESTAMP)
  ");
  $stmt->execute([
    ':tid' => $companyTenantId,
    ':rid' => $repairId,
    ':etype' => 'status_update',
    ':meta' => json_encode($meta, JSON_UNESCAPED_SLASHES),
    ':uid' => $userId,
  ]);

  $pdo->commit();

  header('Location: ' . app_path('repairs/repair_view.php?id=' . $repairId));
  exit;

} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  http_response_code(500);
  echo 'Failed to update repair. ' . htmlspecialchars($e->getMessage());
  exit;
}
