<?php
// /app/repairs/repair_assign_tech.php
declare(strict_types=1);

require_once __DIR__ . '/../includes/auth.php';
require_company_context_if_rto();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  header('Location: ' . app_path('repairs/repairs_list.php'));
  exit;
}

if (!csrf_validate((string)($_POST['_csrf'] ?? ''))) {
  http_response_code(403);
  echo 'CSRF validation failed.';
  exit;
}

$repairId = (int)($_POST['repair_id'] ?? 0);
$techUserId = (int)($_POST['tech_user_id'] ?? 0);

if ($repairId < 1) {
  header('Location: ' . app_path('repairs/repairs_list.php'));
  exit;
}

// Business rule: RTO cannot assign technicians
if (current_tenant_type() === 'rto') {
  http_response_code(403);
  echo 'RTO accounts cannot assign technicians.';
  exit;
}

$pdo = db();
$companyTenantId = (int) current_tenant_id();
$actorId = (int) current_user_id();

try {
  $pdo->beginTransaction();

  // Lock repair row
  $stmt = $pdo->prepare("
    SELECT id, assigned_tech_user_id
    FROM repairs
    WHERE tenant_id = :tid AND id = :rid
    FOR UPDATE
  ");
  $stmt->execute([':tid' => $companyTenantId, ':rid' => $repairId]);
  $repair = $stmt->fetch();

  if (!$repair) {
    $pdo->rollBack();
    header('Location: ' . app_path('repairs/repairs_list.php'));
    exit;
  }

  $oldTech = (int)($repair['assigned_tech_user_id'] ?? 0);

  // Validate tech belongs to same company tenant (or allow unassign with 0)
  if ($techUserId > 0) {
    $stmt = $pdo->prepare("
      SELECT id
      FROM users
      WHERE tenant_id = :tid AND id = :uid AND status='active'
      LIMIT 1
    ");
    $stmt->execute([':tid' => $companyTenantId, ':uid' => $techUserId]);
    if (!$stmt->fetchColumn()) {
      $pdo->rollBack();
      http_response_code(400);
      echo 'Invalid technician selection.';
      exit;
    }
  }

  $stmt = $pdo->prepare("
    UPDATE repairs
    SET assigned_tech_user_id = :tuid,
        updated_at = CURRENT_TIMESTAMP
    WHERE tenant_id = :tid AND id = :rid
  ");
  $stmt->execute([
    ':tuid' => ($techUserId > 0 ? $techUserId : null),
    ':tid' => $companyTenantId,
    ':rid' => $repairId,
  ]);

  // Log repair event
  $meta = [
    'from_tech_user_id' => $oldTech ?: null,
    'to_tech_user_id' => $techUserId ?: null,
  ];

  $stmt = $pdo->prepare("
    INSERT INTO repair_events (tenant_id, repair_id, event_type, meta_json, created_by_user_id, created_at)
    VALUES (:tid, :rid, 'tech_assigned', :meta, :uid, CURRENT_TIMESTAMP)
  ");
  $stmt->execute([
    ':tid' => $companyTenantId,
    ':rid' => $repairId,
    ':meta' => json_encode($meta, JSON_UNESCAPED_SLASHES),
    ':uid' => $actorId,
  ]);

  $pdo->commit();

  header('Location: ' . app_path('repairs/repair_view.php?id=' . $repairId . '&ok=1'));
  exit;

} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  http_response_code(500);
  echo 'Failed to assign technician. ' . htmlspecialchars($e->getMessage());
}
