<?php
// /app/index.php
declare(strict_types=1);

require_once __DIR__ . '/includes/auth.php';

start_secure_session();

if (is_logged_in()) {
  // If RTO but no company selected, route to selector
  if (current_tenant_type() === 'rto' && !active_company_tenant_id()) {
    header('Location: ' . app_path('select_company.php'));
    exit;
  }
  header('Location: ' . app_path('dashboard.php'));
  exit;
}

$err = '';
$email = '';
$lockedMsg = '';

if (login_is_locked()) {
  $lockedMsg = 'Too many login attempts. Please wait and try again.';
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && !$lockedMsg) {
  $email = trim((string)($_POST['email'] ?? ''));
  $password = (string)($_POST['password'] ?? '');
  $token = (string)($_POST['_csrf'] ?? '');

  if (!csrf_validate($token)) {
    $err = 'Security check failed. Please refresh and try again.';
  } elseif ($email === '' || $password === '') {
    $err = 'Please enter your email and password.';
  } else {
    try {
      $pdo = db();

      $stmt = $pdo->prepare("
        SELECT u.*
        FROM users u
        WHERE u.email = :email
          AND u.status = 'active'
        LIMIT 1
      ");
      $stmt->execute([':email' => $email]);
      $user = $stmt->fetch();

      if (!$user || !password_verify($password, $user['password_hash'])) {
        login_register_failure();
        $err = 'Invalid credentials.';
      } else {
        // Fetch tenant row
        $stmt = $pdo->prepare("SELECT id, tenant_type, name, status FROM tenants WHERE id = :tid LIMIT 1");
        $stmt->execute([':tid' => (int)$user['tenant_id']]);
        $tenant = $stmt->fetch();

        if (!$tenant || $tenant['status'] !== 'active' && $tenant['status'] !== 'trial') {
          $err = 'Account is not active. Please contact support.';
        } else {
          login_clear_throttle();
          auth_login_user($user, $tenant);

          // If tenant is company, go to dashboard
          if ($tenant['tenant_type'] === 'company') {
            header('Location: ' . app_path('dashboard.php'));
            exit;
          }

          // If tenant is RTO, must select a company context first
          header('Location: ' . app_path('select_company.php'));
          exit;
        }
      }
    } catch (Throwable $e) {
      $err = 'Login failed due to a server error.';
    }
  }
}
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title><?= h(APP_NAME) ?> — Sign In</title>

  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700;800;900&display=swap" rel="stylesheet">

  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css" rel="stylesheet">
  <link href="<?= app_path('assets/css/app.css?v=1') ?>" rel="stylesheet">

  <style>
    .so-login-wrap{
      min-height:100vh;
      display:flex;
      align-items:center;
      justify-content:center;
      padding:24px;
      background:
        radial-gradient(1200px 600px at 70% -10%, rgba(207,31,31,0.18) 0%, rgba(207,31,31,0.00) 55%),
        radial-gradient(1200px 600px at 10% 10%, rgba(0,0,0,0.10) 0%, rgba(0,0,0,0.00) 50%),
        var(--so-bg);
    }
    .so-login-card{
      width: min(980px, 100%);
      border-radius: 18px;
      overflow:hidden;
      box-shadow: var(--so-shadow);
      border: 1px solid var(--so-border);
      background:#fff;
      display:grid;
      grid-template-columns: 1.1fr 0.9fr;
    }
    .so-login-left{
      padding: 34px;
      background: linear-gradient(180deg, #111318 0%, #141a22 100%);
      color:#fff;
      position:relative;
    }
    .so-login-left:after{
      content:"";
      position:absolute; inset:-40%;
      background: radial-gradient(circle at 30% 20%, rgba(207,31,31,0.30) 0%, rgba(207,31,31,0.00) 60%);
      transform: rotate(8deg);
      pointer-events:none;
    }
    .so-login-left-inner{ position:relative; z-index:1; }
    .so-login-right{ padding: 34px; }
    .so-login-badge{
      display:inline-flex; gap:10px; align-items:center;
      padding: 10px 12px;
      border-radius: 14px;
      background: rgba(255,255,255,0.06);
      border: 1px solid rgba(255,255,255,0.10);
    }
    .so-login-badge .mark{
      width:38px;height:38px;border-radius:12px;
      display:flex;align-items:center;justify-content:center;
      background: rgba(207,31,31,0.18);
      border: 1px solid rgba(207,31,31,0.35);
    }
    .so-login-h1{ font-weight: 900; letter-spacing:-0.9px; margin-top:18px; }
    .so-login-p{ color: rgba(255,255,255,0.75); max-width: 420px; }
    .so-field{
      border-radius: 14px !important;
      padding: 12px 14px !important;
      border: 1px solid rgba(18,20,24,0.14) !important;
    }
    .so-login-btn{
      width:100%;
      border-radius: 14px;
      padding: 12px 14px;
      font-weight: 900;
      background: linear-gradient(180deg, rgba(207,31,31,1) 0%, rgba(176,22,22,1) 100%);
      border: 1px solid rgba(207,31,31,0.45);
    }
    .so-login-btn:hover{ filter: brightness(1.03); }
    @media (max-width: 992px){
      .so-login-card{ grid-template-columns: 1fr; }
      .so-login-left{ display:none; }
    }
  </style>
</head>
<body>

<div class="so-login-wrap">
  <div class="so-login-card">
    <div class="so-login-left">
      <div class="so-login-left-inner">
        <div class="so-login-badge">
          <span class="mark"><i class="fa-solid fa-warehouse"></i></span>
          <strong>ShedOffice</strong><span style="color:var(--so-red);font-weight:900;">.com</span>
        </div>

        <h1 class="so-login-h1">Enterprise operations for repos, repairs, and GPS.</h1>
        <p class="so-login-p">
          A secure portal built for manufacturers and RTO partners. Track assets, manage recovery workflows, and keep your teams aligned in one place.
        </p>

        <div class="mt-4" style="color:rgba(255,255,255,0.70); font-size:13px;">
          <div><i class="fa-solid fa-shield-halved me-2"></i>Role-based access</div>
          <div class="mt-2"><i class="fa-solid fa-lock me-2"></i>Session security + CSRF</div>
          <div class="mt-2"><i class="fa-solid fa-chart-line me-2"></i>Operational reporting</div>
        </div>
      </div>
    </div>

    <div class="so-login-right">
      <div class="mb-2">
        <div style="font-weight:900; font-size:24px; letter-spacing:-0.5px;">Sign In</div>
        <div class="text-muted">Access your account.</div>
      </div>

      <?php if ($lockedMsg): ?>
        <div class="alert alert-warning mt-3"><?= h($lockedMsg) ?></div>
      <?php endif; ?>

      <?php if ($err): ?>
        <div class="alert alert-danger mt-3"><?= h($err) ?></div>
      <?php endif; ?>

      <form method="post" class="mt-3" autocomplete="off">
        <input type="hidden" name="_csrf" value="<?= h(csrf_token()) ?>">

        <div class="mb-3">
          <label class="form-label fw-bold">Email</label>
          <input type="email" name="email" class="form-control so-field" value="<?= h($email) ?>" required>
        </div>

        <div class="mb-3">
          <label class="form-label fw-bold">Password</label>
          <input type="password" name="password" class="form-control so-field" required>
        </div>

        <button class="btn btn-danger so-login-btn" type="submit">
          <i class="fa-solid fa-right-to-bracket me-2"></i>Sign In
        </button>

        <div class="mt-3 d-flex justify-content-between align-items-center">
          <a href="#" class="text-muted small text-decoration-none">Forgot password?</a>
          <span class="text-muted small">v0.1</span>
        </div>
      </form>
    </div>
  </div>
</div>

</body>
</html>
