<?php
// /app/includes/acl.php
declare(strict_types=1);

require_once __DIR__ . '/auth.php';
require_once __DIR__ . '/db.php';

/**
 * Returns all role names for the current user within their tenant.
 */
function user_roles_list(PDO $pdo, int $userId, int $tenantId): array {
  $stmt = $pdo->prepare("
    SELECT r.name
    FROM user_roles ur
    JOIN roles r ON r.id = ur.role_id
    WHERE ur.user_id = :uid1
      AND r.tenant_id = :tid1
  ");
  $stmt->execute([':uid1' => $userId, ':tid1' => $tenantId]);
  return array_values(array_filter(array_map('strval', array_column($stmt->fetchAll(), 'name'))));
}

function user_has_role(string $roleName): bool {
  start_secure_session();
  $roles = $_SESSION['auth']['roles'] ?? [];
  return is_array($roles) && in_array($roleName, $roles, true);
}

/**
 * Load & cache roles + permissions for the current session.
 * IMPORTANT: Unique placeholder names everywhere (PDO emulate_prepares is OFF).
 */
function acl_bootstrap(PDO $pdo): void {
  start_secure_session();
  if (!empty($_SESSION['auth']['acl_loaded'])) return;

  $userId = (int)($_SESSION['auth']['user_id'] ?? 0);
  $tenantId = (int)($_SESSION['auth']['tenant_id'] ?? 0);
  if ($userId < 1 || $tenantId < 1) return;

  try {
    // roles
    $roles = user_roles_list($pdo, $userId, $tenantId);
    $_SESSION['auth']['roles'] = $roles;

    // 1) role permissions
    $stmt = $pdo->prepare("
      SELECT p.perm_key, rp.effect
      FROM user_roles ur
      JOIN roles r ON r.id = ur.role_id
      JOIN role_permissions rp ON rp.role_id = r.id
      JOIN permissions p ON p.id = rp.permission_id
      WHERE ur.user_id = :uid1
        AND r.tenant_id = :tid1
    ");
    $stmt->execute([':uid1' => $userId, ':tid1' => $tenantId]);
    $rolePermRows = $stmt->fetchAll();

    // 2) user overrides
    $stmt = $pdo->prepare("
      SELECT p.perm_key, up.effect
      FROM user_permissions up
      JOIN permissions p ON p.id = up.permission_id
      WHERE up.user_id = :uid2
        AND up.tenant_id = :tid2
    ");
    $stmt->execute([':uid2' => $userId, ':tid2' => $tenantId]);
    $userPermRows = $stmt->fetchAll();

    // Build effective permissions
    $allow = [];
    $deny = [];

    foreach ($rolePermRows as $row) {
      $k = (string)$row['perm_key'];
      $e = (string)$row['effect'];
      if ($e === 'allow') $allow[$k] = true;
      if ($e === 'deny')  $deny[$k]  = true;
    }

    foreach ($userPermRows as $row) {
      $k = (string)$row['perm_key'];
      $e = (string)$row['effect'];
      if ($e === 'allow') {
        $allow[$k] = true;
        unset($deny[$k]);
      }
      if ($e === 'deny') {
        $deny[$k] = true;
        unset($allow[$k]);
      }
    }

    $_SESSION['auth']['perm_allow'] = array_keys($allow);
    $_SESSION['auth']['perm_deny']  = array_keys($deny);
    $_SESSION['auth']['acl_loaded'] = 1;

  } catch (Throwable $e) {
    // Do NOT loop redirect on ACL failure. Leave ACL unloaded and allow role-only UI.
    $_SESSION['auth']['roles'] = $_SESSION['auth']['roles'] ?? [];
    $_SESSION['auth']['perm_allow'] = [];
    $_SESSION['auth']['perm_deny'] = [];
    $_SESSION['auth']['acl_loaded'] = 0;

    // Log somewhere safe if possible (optional)
    // error_log("ACL bootstrap failed: " . $e->getMessage());
  }
}

function user_can(string $permKey): bool {
  start_secure_session();
  $deny = $_SESSION['auth']['perm_deny'] ?? [];
  if (is_array($deny) && in_array($permKey, $deny, true)) return false;

  $allow = $_SESSION['auth']['perm_allow'] ?? [];
  return is_array($allow) && in_array($permKey, $allow, true);
}

function require_perm(string $permKey): void {
  $pdo = db();
  acl_bootstrap($pdo);
  if (!user_can($permKey)) {
    http_response_code(403);
    echo "Forbidden: missing permission " . htmlspecialchars($permKey);
    exit;
  }
}
