<?php
// /app/driver/repo_view.php
declare(strict_types=1);

require_once __DIR__ . '/../includes/auth.php';
require_login();

require_once __DIR__ . '/../includes/acl.php';
require_once __DIR__ . '/../includes/scope.php';

$pdo = db();
acl_bootstrap($pdo);

if (current_tenant_type() !== 'company') {
  http_response_code(403);
  echo "Driver mode is for company accounts only.";
  exit;
}

if (!user_has_role('driver')) {
  http_response_code(403);
  echo "Forbidden: driver role required.";
  exit;
}

$pageTitle = 'Pickup Package';
$activeNav = 'repos';

$companyTenantId = company_context_id();
$driverUserId = (int) current_user_id();

$repoId = (int)($_GET['id'] ?? 0);
$autoPickup = (int)($_GET['pickup'] ?? 0) === 1;

if ($repoId < 1) {
  header('Location: ' . app_path('driver/repos.php'));
  exit;
}

// Helpers
function fetch_company_policy(PDO $pdo, int $companyTid): array {
  $driverCanOverrideInvoice = 0;
  $driverCanCustomInvoice = 0;

  $stmt = $pdo->prepare("
    SELECT driver_can_override_repo_invoice, driver_can_submit_custom_invoice
    FROM tenant_settings
    WHERE tenant_id = :tid1
    LIMIT 1
  ");
  $stmt->execute([':tid1' => $companyTid]);
  if ($row = $stmt->fetch()) {
    $driverCanOverrideInvoice = (int)$row['driver_can_override_repo_invoice'];
    $driverCanCustomInvoice = (int)$row['driver_can_submit_custom_invoice'];
  }

  return [$driverCanOverrideInvoice, $driverCanCustomInvoice];
}

/**
 * Load add-on fee types for this repo context.
 * Precedence: if RTO-specific exists, it overrides company by NAME.
 */
function load_addon_types_for_repo(PDO $pdo, int $companyTid, ?int $repoRtoTenantId): array {
  $params = [':tid1' => $companyTid];
  $where = "tenant_id = :tid1 AND is_active = 1 AND applies_to='company'";

  if ($repoRtoTenantId) {
    $where = "tenant_id = :tid1 AND is_active = 1 AND (
      applies_to='company'
      OR (applies_to='rto' AND rto_tenant_id = :rto1)
    )";
    $params[':rto1'] = $repoRtoTenantId;
  }

  $stmt = $pdo->prepare("
    SELECT id, name, description, default_amount, max_amount, driver_can_add, driver_can_override_amount, applies_to, rto_tenant_id
    FROM repo_addon_fee_types
    WHERE $where
    ORDER BY (applies_to='rto') DESC, name ASC
  ");
  $stmt->execute($params);
  $rows = $stmt->fetchAll();

  // Dedup by name, keep first (RTO-specific comes first)
  $byName = [];
  foreach ($rows as $r) {
    $nm = (string)$r['name'];
    if (!isset($byName[$nm])) $byName[$nm] = $r;
  }

  // Driver only: driver_can_add=1
  $filtered = [];
  foreach ($byName as $r) {
    if ((int)$r['driver_can_add'] === 1) $filtered[] = $r;
  }

  return $filtered;
}

// Only allow driver to view assigned repo
$stmt = $pdo->prepare("
  SELECT
    r.*,
    a.stock_number, a.width_ft, a.length_ft, a.asset_type,
    t_rto.name AS rto_name
  FROM repos r
  JOIN assets a ON a.id = r.asset_id
  LEFT JOIN tenants t_rto ON t_rto.id = r.rto_tenant_id
  WHERE r.tenant_id = :tid1
    AND r.id = :rid1
    AND r.assigned_driver_user_id = :uid1
  LIMIT 1
");
$stmt->execute([':tid1' => $companyTenantId, ':rid1' => $repoId, ':uid1' => $driverUserId]);
$repo = $stmt->fetch();

if (!$repo) {
  http_response_code(404);
  echo "Repo not found or not assigned to you.";
  exit;
}

$repoRtoTid = !empty($repo['rto_tenant_id']) ? (int)$repo['rto_tenant_id'] : null;

// Dropoff locations
$stmt = $pdo->prepare("
  SELECT id, name
  FROM locations
  WHERE tenant_id = :tid1 AND status='active'
  ORDER BY name ASC
  LIMIT 500
");
$stmt->execute([':tid1' => $companyTenantId]);
$locations = $stmt->fetchAll();

// Existing pickup/dropoff (if already submitted)
$stmt = $pdo->prepare("SELECT * FROM repo_pickups WHERE tenant_id=:tid1 AND repo_id=:rid1 LIMIT 1");
$stmt->execute([':tid1' => $companyTenantId, ':rid1' => $repoId]);
$pickup = $stmt->fetch();

$stmt = $pdo->prepare("SELECT dropoff_location_id FROM repo_dropoffs WHERE tenant_id=:tid1 AND repo_id=:rid1 LIMIT 1");
$stmt->execute([':tid1' => $companyTenantId, ':rid1' => $repoId]);
$dropoff = $stmt->fetch();

// Driver override policy + RTO link override
[$driverCanOverrideInvoice, $driverCanCustom] = fetch_company_policy($pdo, $companyTenantId);

if ($repoRtoTid) {
  $stmt = $pdo->prepare("
    SELECT id
    FROM rto_company_links
    WHERE rto_tenant_id = :rto1 AND company_tenant_id = :co1 AND status='active'
    LIMIT 1
  ");
  $stmt->execute([':rto1' => $repoRtoTid, ':co1' => $companyTenantId]);
  $linkId = (int)($stmt->fetchColumn() ?: 0);

  if ($linkId) {
    $stmt = $pdo->prepare("
      SELECT driver_can_override_repo_invoice, driver_can_submit_custom_invoice
      FROM rto_company_policy_settings
      WHERE rto_company_link_id = :lid1
      LIMIT 1
    ");
    $stmt->execute([':lid1' => $linkId]);
    if ($ov = $stmt->fetch()) {
      if ($ov['driver_can_override_repo_invoice'] !== null) $driverCanOverrideInvoice = (int)$ov['driver_can_override_repo_invoice'];
      if ($ov['driver_can_submit_custom_invoice'] !== null) $driverCanCustom = (int)$ov['driver_can_submit_custom_invoice'];
    }
  }
}

$addonTypes = load_addon_types_for_repo($pdo, $companyTenantId, $repoRtoTid);

$size = trim(((int)$repo['width_ft'] ? (int)$repo['width_ft'].'x' : '') . ((int)$repo['length_ft'] ?? ''));
$size = $size ?: strtoupper((string)$repo['asset_type']);

require_once __DIR__ . '/../includes/header.php';
?>
<div class="container-fluid">

  <div class="d-flex flex-wrap justify-content-between align-items-end gap-2 mb-3">
    <div>
      <div class="so-page-title">Repo #<?= (int)$repo['id'] ?></div>
      <div class="so-page-sub"><?= h((string)$repo['stock_number']) ?> · <?= h($size) ?><?= !empty($repo['rto_name']) ? (' · ' . h((string)$repo['rto_name'])) : '' ?></div>
    </div>
    <div class="d-flex gap-2">
      <a class="btn so-btn" href="<?= app_path('driver/repos.php') ?>"><i class="fa-solid fa-arrow-left me-2"></i>Back</a>
      <button class="btn btn-dark" style="border-radius:12px;" type="button" data-bs-toggle="modal" data-bs-target="#pickupModal" id="openPickupBtn">
        <i class="fa-solid fa-camera me-2"></i>Pickup Package
      </button>
    </div>
  </div>

  <div class="so-card mb-3">
    <div class="so-card-body">
      <div class="text-muted small">Pickup Address</div>
      <div class="fw-bold"><?= h((string)($repo['pickup_address1'] ?? '—')) ?></div>
      <div class="text-muted small">
        <?= h((string)($repo['pickup_city'] ?? '')) ?> <?= h((string)($repo['pickup_state'] ?? '')) ?> <?= h((string)($repo['pickup_zip'] ?? '')) ?>
      </div>

      <?php if (!empty($repo['notes'])): ?>
        <hr>
        <div class="text-muted small">Notes</div>
        <div><?= nl2br(h((string)$repo['notes'])) ?></div>
      <?php endif; ?>

      <?php if ($pickup): ?>
        <hr>
        <div class="alert alert-success mb-0">
          Pickup package already submitted on <?= h(date('M j, Y g:i A', strtotime((string)$pickup['picked_up_at']))) ?>.
          Re-submit to add additional photos/updates (record will update).
        </div>
      <?php endif; ?>
    </div>
  </div>

</div>

<!-- DRIVER PICKUP MODAL -->
<div class="modal fade" id="pickupModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-xl modal-dialog-centered">
    <div class="modal-content" style="border-radius:16px; overflow:hidden;">
      <div class="modal-header" style="background:#111318; color:#fff;">
        <h5 class="modal-title">
          <i class="fa-solid fa-camera me-2" style="color:#cf1f1f"></i>Submit Pickup Package
        </h5>
        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
      </div>

      <form method="post" action="<?= app_path('repos/repo_confirm_pickup.php') ?>" enctype="multipart/form-data" id="pickupForm">
        <div class="modal-body">
          <input type="hidden" name="_csrf" value="<?= h(csrf_token()) ?>">
          <input type="hidden" name="repo_id" value="<?= (int)$repoId ?>">

          <div class="row g-3">

            <div class="col-12 col-md-6">
              <label class="form-label fw-bold">Drop-off Location (required)</label>
              <select name="dropoff_location_id" class="form-select" style="border-radius:12px;" required>
                <option value="">— Select location —</option>
                <?php foreach ($locations as $l): ?>
                  <option value="<?= (int)$l['id'] ?>" <?= (!empty($dropoff['dropoff_location_id']) && (int)$dropoff['dropoff_location_id']===(int)$l['id']) ? 'selected' : '' ?>>
                    <?= h((string)$l['name']) ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>

            <div class="col-12 col-md-6">
              <label class="form-label fw-bold">Miles Pickup → Drop-off</label>
              <input type="number" step="0.01" min="0" name="miles_pickup_to_dropoff" class="form-control" style="border-radius:12px;"
                     value="<?= h((string)($pickup['miles_pickup_to_dropoff'] ?? '')) ?>" placeholder="e.g. 72.5">
            </div>

            <div class="col-12 col-md-6">
              <label class="form-label fw-bold">Building Photo (required)</label>
              <input type="file" name="building_photo" class="form-control" style="border-radius:12px;" accept="image/*" required>
            </div>

            <div class="col-12 col-md-6">
              <label class="form-label fw-bold">Serial Number Photo (required)</label>
              <input type="file" name="serial_photo" class="form-control" style="border-radius:12px;" accept="image/*" required>
            </div>

            <div class="col-12">
              <label class="form-label fw-bold">Corner Photos (optional)</label>
              <input type="file" name="corner_photos[]" class="form-control" style="border-radius:12px;" accept="image/*" multiple>
            </div>

            <div class="col-12 col-md-3">
              <label class="form-label fw-bold">Repairs Needed?</label>
              <select name="repairs_needed" class="form-select" style="border-radius:12px;">
                <option value="0" <?= (!empty($pickup) && (int)$pickup['repairs_needed']===0) ? 'selected' : '' ?>>No</option>
                <option value="1" <?= (!empty($pickup) && (int)$pickup['repairs_needed']===1) ? 'selected' : '' ?>>Yes</option>
              </select>
            </div>

            <div class="col-12 col-md-9">
              <label class="form-label fw-bold">Repairs Notes</label>
              <input type="text" name="repairs_notes" class="form-control" style="border-radius:12px;"
                     value="<?= h((string)($pickup['repairs_notes'] ?? '')) ?>" maxlength="2000" placeholder="What repairs are needed?">
            </div>

            <div class="col-12">
              <label class="form-label fw-bold">Condition Notes</label>
              <textarea name="condition_notes" class="form-control" style="border-radius:12px;" rows="3" maxlength="2000"
                        placeholder="Condition, damage, site notes..."><?= h((string)($pickup['condition_notes'] ?? '')) ?></textarea>
            </div>

            <!-- DRIVER ADD-ON FEES -->
            <div class="col-12">
              <hr>
              <div class="fw-bold mb-2">Add-on Fees (optional)</div>
              <div class="text-muted small mb-2">
                These fees are controlled by company/RTO policy. Only allowed items are shown.
              </div>

              <?php if (!$addonTypes): ?>
                <div class="text-muted">No driver-addable fee types are configured.</div>
              <?php else: ?>
                <div class="table-responsive">
                  <table class="table so-table align-middle">
                    <thead>
                      <tr>
                        <th style="width:52px;"></th>
                        <th>Fee</th>
                        <th class="text-end" style="width:180px;">Amount</th>
                      </tr>
                    </thead>
                    <tbody>
                      <?php foreach ($addonTypes as $t): ?>
                        <?php
                          $aid = (int)$t['id'];
                          $defaultAmt = (float)$t['default_amount'];
                          $maxAmt = (float)$t['max_amount'];
                          $driverCanOverrideAmt = ((int)$t['driver_can_override_amount'] === 1) && ($maxAmt > 0);

                          $help = $driverCanOverrideAmt ? ('Max $' . number_format($maxAmt, 2)) : 'No override';
                        ?>
                        <tr>
                          <td>
                            <input class="form-check-input" type="checkbox" name="addon_ids[]" value="<?= $aid ?>">
                          </td>
                          <td>
                            <div class="fw-bold"><?= h((string)$t['name']) ?></div>
                            <?php if (!empty($t['description'])): ?>
                              <div class="text-muted small"><?= h((string)$t['description']) ?></div>
                            <?php endif; ?>
                            <div class="text-muted small">Default: $<?= number_format($defaultAmt,2) ?> · <?= h($help) ?></div>
                          </td>
                          <td class="text-end">
                            <input
                              type="number"
                              step="0.01"
                              min="0"
                              name="addon_amount[<?= $aid ?>]"
                              class="form-control"
                              style="border-radius:12px; text-align:right;"
                              value="<?= h((string)$defaultAmt) ?>"
                              <?= $driverCanOverrideAmt ? '' : 'readonly' ?>
                            >
                          </td>
                        </tr>
                      <?php endforeach; ?>
                    </tbody>
                  </table>
                </div>
              <?php endif; ?>
            </div>

            <!-- DRIVER TOTAL OVERRIDE -->
            <?php if ($driverCanOverrideInvoice === 1): ?>
              <div class="col-12">
                <hr>
                <div class="fw-bold mb-2">Invoice Total Override (requires reason)</div>
                <div class="text-muted small mb-2">
                  If you enter an override total, it will be submitted for manager/admin approval.
                </div>
              </div>

              <div class="col-12 col-md-4">
                <label class="form-label fw-bold">Override Total ($)</label>
                <input type="number" step="0.01" min="0" name="override_total" class="form-control" style="border-radius:12px;" placeholder="Leave blank for standard fee" id="overrideTotal">
              </div>

              <div class="col-12 col-md-8">
                <label class="form-label fw-bold">Override Reason</label>
                <input type="text" name="override_reason" class="form-control" style="border-radius:12px;" maxlength="500"
                       placeholder="Required if override total is set" id="overrideReason">
              </div>
            <?php endif; ?>

          </div>
        </div>

        <div class="modal-footer">
          <button class="btn btn-outline-dark" style="border-radius:12px;" type="button" data-bs-dismiss="modal">Cancel</button>
          <button class="btn btn-dark" style="border-radius:12px;" type="submit">
            <i class="fa-solid fa-paper-plane me-2"></i>Submit
          </button>
        </div>
      </form>
    </div>
  </div>
</div>

<?php if ($autoPickup): ?>
<script>
document.addEventListener('DOMContentLoaded', function () {
  const btn = document.getElementById('openPickupBtn');
  if (btn) btn.click();
});
</script>
<?php endif; ?>

<script>
document.addEventListener('DOMContentLoaded', function () {
  // Client-side assist: if override total entered, require a reason
  const total = document.getElementById('overrideTotal');
  const reason = document.getElementById('overrideReason');
  const form = document.getElementById('pickupForm');

  if (form && total && reason) {
    form.addEventListener('submit', function (e) {
      const tv = (total.value || '').trim();
      const rv = (reason.value || '').trim();
      if (tv !== '' && rv === '') {
        e.preventDefault();
        alert('Override reason is required when override total is set.');
        reason.focus();
      }
    });
  }
});
</script>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
