<?php
// /app/underwriting/view.php
declare(strict_types=1);

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/scope.php';

$pageTitle = 'Underwriting Case';
$activeNav = 'underwriting';
require_once __DIR__ . '/../includes/header.php';
require_once __DIR__ . '/_uw.php';

require_company_context_if_rto();

$pdo = db();
$tenantId = uw_company_tid();
$orderId = (int)($_GET['order_id'] ?? 0);
if ($orderId <= 0) {
  header('Location: ' . app_path('underwriting/index.php'));
  exit;
}

$stmt = $pdo->prepare('SELECT * FROM orders WHERE tenant_id = :tid AND id = :id');
$stmt->execute([':tid'=>$tenantId, ':id'=>$orderId]);
$order = $stmt->fetch(PDO::FETCH_ASSOC);
if (!$order) {
  http_response_code(404);
  echo 'Order not found.';
  exit;
}

$case = uw_ensure_case($pdo, $tenantId, $orderId);
$caseId = (int)$case['id'];
$settings = uw_get_settings($pdo, $tenantId);

$stepKeys = [
  'id_ai' => 'AI ID Verification',
  'softpull' => 'Soft Credit Pull (SoftPullSolutions)',
  'blocks' => 'Blocks Acknowledgement',
  'delivery_risk' => 'Delivery Location Risk',
  'public_records' => 'Public Records Scan',
  'signature_match' => 'Signature Match (ID vs Contract)',
];

$steps = [];
foreach ($stepKeys as $k=>$label) {
  $steps[$k] = uw_get_step($pdo, $caseId, $k);
}

$notesStmt = $pdo->prepare('SELECT n.*, u.name AS user_name FROM underwriting_notes n LEFT JOIN users u ON u.id = n.user_id WHERE n.case_id = :cid ORDER BY n.created_at DESC LIMIT 50');
$notesStmt->execute([':cid'=>$caseId]);
$notes = $notesStmt->fetchAll(PDO::FETCH_ASSOC);

$csrf = csrf_token();

function step_badge(string $status): string {
  return match($status) {
    'pass' => 'success',
    'review' => 'warning',
    'fail' => 'danger',
    'done' => 'success',
    default => 'secondary'
  };
}

?>

<div class="container-fluid">
  <div class="row">
    <div class="col-12">
      <div class="so-page-title">Underwriting Â·middot; #<?= h($order['order_number'] ?? '') ?></div>
      <div class="so-page-sub">
        <?= h($order['customer_name'] ?? '') ?> Â·middot; <?= h($order['customer_phone_primary'] ?? '') ?>
        Â·middot; Delivery: <?= h(($order['delivery_city'] ?? '') . ', ' . ($order['delivery_state'] ?? '')) ?>
      </div>
    </div>
  </div>

  <div class="row g-3 mb-3">
    <div class="col-12 col-lg-8">
      <div class="so-card p-3">
        <div class="d-flex justify-content-between align-items-center">
          <div>
            <div class="text-muted">Current UW Status</div>
            <div style="font-weight:900; font-size:18px;">
              <span class="badge bg-<?= ($case['status']==='approved'?'success':($case['status']==='denied'?'danger':($case['status']==='conditional'?'warning':'secondary'))) ?>">
                <?= h(strtoupper((string)$case['status'])) ?>
              </span>
              <span class="ms-2 badge bg-<?= ((int)$case['risk_score']>=55?'danger':((int)$case['risk_score']>=30?'warning':'success')) ?>"><?= (int)$case['risk_score'] ?>/100 Risk</span>
            </div>
          </div>
          <div class="d-flex gap-2">
            <a class="btn btn-outline-light" href="<?= app_path('underwriting/index.php') ?>"><i class="fa-solid fa-arrow-left"></i> Back</a>
            <a class="btn btn-outline-light" href="<?= app_path('underwriting/settings.php') ?>"><i class="fa-solid fa-gear"></i> Settings</a>
          </div>
        </div>
      </div>
    </div>

    <div class="col-12 col-lg-4">
      <div class="so-card p-3">
        <div class="so-side-section">Decision</div>
        <form method="post" action="<?= app_path('underwriting/actions/decide.php') ?>" class="d-grid gap-2">
          <input type="hidden" name="_csrf" value="<?= h($csrf) ?>">
          <input type="hidden" name="order_id" value="<?= (int)$orderId ?>">
          <select class="form-select" name="decision">
            <option value="approved">Approve</option>
            <option value="conditional">Conditional</option>
            <option value="denied">Deny</option>
          </select>
          <input class="form-control" name="reason" placeholder="Reason / notes (optional)">
          <button class="btn btn-primary" type="submit"><i class="fa-solid fa-circle-check"></i> Save Decision</button>
        </form>
      </div>
    </div>
  </div>

  <div class="row g-3">
    <div class="col-12 col-lg-8">
      <?php foreach ($stepKeys as $key => $label):
        $s = $steps[$key];
        $badge = step_badge((string)$s['status']);
        $details = $s['details'] ?? [];
      ?>
      <div class="so-card p-3 mb-3">
        <div class="d-flex justify-content-between align-items-start">
          <div>
            <div style="font-weight:900;"><i class="fa-solid fa-list-check"></i> <?= h($label) ?></div>
            <div class="text-muted" style="font-size:12px;">Status: <span class="badge bg-<?= $badge ?>"><?= h(strtoupper((string)$s['status'])) ?></span> Â·middot; Score: <?= (int)$s['score'] ?></div>
          </div>
          <div class="d-flex gap-2">
            <?php if ($key === 'id_ai'): ?>
              <form method="post" enctype="multipart/form-data" action="<?= app_path('underwriting/actions/upload_id.php') ?>">
                <input type="hidden" name="_csrf" value="<?= h($csrf) ?>">
                <input type="hidden" name="order_id" value="<?= (int)$orderId ?>">
                <div class="d-flex gap-2 align-items-center">
                  <input type="file" name="id_front" class="form-control form-control-sm" accept="image/*" required>
                  <input type="file" name="id_back" class="form-control form-control-sm" accept="image/*">
                  <button class="btn btn-sm btn-outline-light" type="submit"><i class="fa-solid fa-upload"></i> Upload</button>
                </div>
              </form>
              <form method="post" action="<?= app_path('underwriting/actions/run_id_ai.php') ?>">
                <input type="hidden" name="_csrf" value="<?= h($csrf) ?>">
                <input type="hidden" name="order_id" value="<?= (int)$orderId ?>">
                <button class="btn btn-sm btn-primary" type="submit"><i class="fa-solid fa-wand-magic-sparkles"></i> Run AI</button>
              </form>
            <?php elseif ($key === 'softpull'): ?>
              <form method="post" action="<?= app_path('underwriting/actions/run_softpull.php') ?>">
                <input type="hidden" name="_csrf" value="<?= h($csrf) ?>">
                <input type="hidden" name="order_id" value="<?= (int)$orderId ?>">
                <button class="btn btn-sm btn-primary" type="submit"><i class="fa-solid fa-credit-card"></i> Run Soft Pull</button>
              </form>
            <?php elseif ($key === 'blocks'): ?>
              <form method="post" action="<?= app_path('underwriting/actions/step_update.php') ?>">
                <input type="hidden" name="_csrf" value="<?= h($csrf) ?>">
                <input type="hidden" name="order_id" value="<?= (int)$orderId ?>">
                <input type="hidden" name="step_key" value="blocks">
                <input type="hidden" name="status" value="done">
                <input type="hidden" name="score" value="0">
                <input type="hidden" name="note" value="Customer informed about blocks and agreed.">
                <button class="btn btn-sm btn-outline-light" type="submit"><i class="fa-solid fa-check"></i> Mark Complete</button>
              </form>
            <?php elseif ($key === 'delivery_risk'): ?>
              <form method="post" action="<?= app_path('underwriting/actions/run_delivery_risk.php') ?>">
                <input type="hidden" name="_csrf" value="<?= h($csrf) ?>">
                <input type="hidden" name="order_id" value="<?= (int)$orderId ?>">
                <button class="btn btn-sm btn-primary" type="submit"><i class="fa-solid fa-map-location-dot"></i> Run Scan</button>
              </form>
            <?php elseif ($key === 'public_records'): ?>
              <form method="post" action="<?= app_path('underwriting/actions/run_public_records.php') ?>">
                <input type="hidden" name="_csrf" value="<?= h($csrf) ?>">
                <input type="hidden" name="order_id" value="<?= (int)$orderId ?>">
                <button class="btn btn-sm btn-primary" type="submit"><i class="fa-solid fa-gavel"></i> Run Search</button>
              </form>
            <?php elseif ($key === 'signature_match'): ?>
              <form method="post" action="<?= app_path('underwriting/actions/run_signature_match.php') ?>">
                <input type="hidden" name="_csrf" value="<?= h($csrf) ?>">
                <input type="hidden" name="order_id" value="<?= (int)$orderId ?>">
                <button class="btn btn-sm btn-primary" type="submit"><i class="fa-solid fa-signature"></i> Compare Signatures</button>
              </form>
            <?php endif; ?>
          </div>
        </div>

        <?php if (!empty($details)): ?>
          <div class="mt-3">
            <pre style="white-space:pre-wrap; font-size:12px; background:rgba(255,255,255,0.04); border:1px solid rgba(255,255,255,0.08); padding:10px; border-radius:12px;"><?= h(json_encode($details, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES)) ?></pre>
          </div>
        <?php else: ?>
          <div class="mt-2 text-muted" style="font-size:13px;">No data captured yet for this step.</div>
        <?php endif; ?>
      </div>
      <?php endforeach; ?>
    </div>

    <div class="col-12 col-lg-4">
      <div class="so-card p-3 mb-3">
        <div class="so-side-section">Notes</div>
        <form method="post" action="<?= app_path('underwriting/actions/note_add.php') ?>" class="d-grid gap-2">
          <input type="hidden" name="_csrf" value="<?= h($csrf) ?>">
          <input type="hidden" name="order_id" value="<?= (int)$orderId ?>">
          <textarea class="form-control" name="note" rows="3" placeholder="Add underwriting note..."></textarea>
          <button class="btn btn-outline-light" type="submit"><i class="fa-solid fa-comment"></i> Add Note</button>
        </form>
        <hr style="border-color:rgba(255,255,255,0.08);" class="my-3">
        <?php if (!$notes): ?>
          <div class="text-muted" style="font-size:13px;">No notes yet.</div>
        <?php else: foreach ($notes as $n): ?>
          <div class="mb-3">
            <div style="font-weight:700; font-size:13px;"><?= h($n['user_name'] ?? ('User#'.(int)$n['user_id'])) ?></div>
            <div class="text-muted" style="font-size:12px;"><?= h($n['created_at'] ?? '') ?></div>
            <div style="font-size:13px; white-space:pre-wrap;"><?= h($n['note'] ?? '') ?></div>
          </div>
        <?php endforeach; endif; ?>
      </div>

      <div class="so-card p-3">
        <div class="so-side-section">Order docs</div>
        <div class="text-muted" style="font-size:13px;">Invoice URL: <?= $order['invoice_url'] ? '<a href="'.h($order['invoice_url']).'" target="_blank">Open</a>' : 'âmdash;' ?></div>
        <div class="text-muted" style="font-size:13px;">Delivery Certificate URL: <?= $order['delivery_certificate_url'] ? '<a href="'.h($order['delivery_certificate_url']).'" target="_blank">Open</a>' : 'âmdash;' ?></div>
      </div>

    </div>
  </div>
</div>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
