<?php
// /app/underwriting/_uw.php
declare(strict_types=1);

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/scope.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/bunny.php';

function uw_company_tid(): int {
  require_company_context_if_rto();
  return company_context_id();
}

function uw_user_id(): int {
  return (int)(current_user_id() ?? 0);
}

function uw_pdo(): PDO {
  return db();
}

function uw_json_encode($v): string {
  return json_encode($v, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
}

function uw_json_decode(?string $s): array {
  if (!$s) return [];
  $d = json_decode($s, true);
  return is_array($d) ? $d : [];
}

function uw_get_settings(PDO $pdo, int $tenantId): array {
  $stmt = $pdo->prepare('SELECT settings_json FROM underwriting_settings WHERE tenant_id = :tid');
  $stmt->execute([':tid' => $tenantId]);
  $row = $stmt->fetch(PDO::FETCH_ASSOC);
  $settings = $row ? uw_json_decode((string)$row['settings_json']) : [];

  $defaults = [
    'openai_api_key' => '',
    'openai_model' => 'gpt-4.1-mini',
    'google_cse_key' => '',
    'google_cse_cx' => '',
    'softpull_base_url' => '',
    'softpull_username' => '',
    'softpull_password' => '',
    'risk_threshold_pass' => 29,
    'risk_threshold_review' => 54,
    'risk_threshold_fail' => 55,
  ];
  return array_merge($defaults, $settings);
}

function uw_save_settings(PDO $pdo, int $tenantId, array $settings): void {
  $json = uw_json_encode($settings);
  $stmt = $pdo->prepare('INSERT INTO underwriting_settings (tenant_id, settings_json) VALUES (:tid,:json)
    ON DUPLICATE KEY UPDATE settings_json = VALUES(settings_json), updated_at = CURRENT_TIMESTAMP');
  $stmt->execute([':tid' => $tenantId, ':json' => $json]);
}

function uw_ensure_case(PDO $pdo, int $tenantId, int $orderId): array {
  $stmt = $pdo->prepare('SELECT * FROM underwriting_cases WHERE tenant_id=:tid AND order_id=:oid');
  $stmt->execute([':tid'=>$tenantId, ':oid'=>$orderId]);
  $case = $stmt->fetch(PDO::FETCH_ASSOC);
  if ($case) return $case;

  $pdo->prepare('INSERT INTO underwriting_cases (tenant_id, order_id, status, risk_score) VALUES (:tid,:oid,\'open\',0)')
      ->execute([':tid'=>$tenantId, ':oid'=>$orderId]);
  $id = (int)$pdo->lastInsertId();
  $stmt = $pdo->prepare('SELECT * FROM underwriting_cases WHERE id=:id');
  $stmt->execute([':id'=>$id]);
  return (array)$stmt->fetch(PDO::FETCH_ASSOC);
}

function uw_get_step(PDO $pdo, int $caseId, string $stepKey): array {
  $stmt = $pdo->prepare('SELECT * FROM underwriting_steps WHERE case_id=:cid AND step_key=:k');
  $stmt->execute([':cid'=>$caseId, ':k'=>$stepKey]);
  $row = $stmt->fetch(PDO::FETCH_ASSOC);
  if ($row) {
    $row['details'] = uw_json_decode($row['details_json'] ?? null);
    return $row;
  }
  $pdo->prepare('INSERT INTO underwriting_steps (case_id, step_key, status, score) VALUES (:cid,:k,\'pending\',0)')
      ->execute([':cid'=>$caseId, ':k'=>$stepKey]);
  return uw_get_step($pdo, $caseId, $stepKey);
}

function uw_set_step(PDO $pdo, int $caseId, string $stepKey, string $status, int $score, array $details = [], ?int $completedBy = null): void {
  $detailsJson = uw_json_encode($details);
  $sql = 'INSERT INTO underwriting_steps (case_id, step_key, status, score, details_json, completed_by_user_id, completed_at)
          VALUES (:cid,:k,:st,:sc,:dj,:by,IF(:by IS NULL,NULL,NOW()))
          ON DUPLICATE KEY UPDATE status=VALUES(status), score=VALUES(score), details_json=VALUES(details_json),
            completed_by_user_id=VALUES(completed_by_user_id), completed_at=IF(VALUES(completed_by_user_id) IS NULL, completed_at, NOW()), updated_at=NOW()';
  $pdo->prepare($sql)->execute([
    ':cid'=>$caseId,
    ':k'=>$stepKey,
    ':st'=>$status,
    ':sc'=>$score,
    ':dj'=>$detailsJson,
    ':by'=>$completedBy,
  ]);
}

function uw_add_note(PDO $pdo, int $caseId, int $userId, string $note): void {
  $pdo->prepare('INSERT INTO underwriting_notes (case_id, user_id, note) VALUES (:cid,:uid,:n)')
      ->execute([':cid'=>$caseId, ':uid'=>$userId, ':n'=>$note]);
}

function uw_recompute_case_risk(PDO $pdo, int $caseId): void {
  $stmt = $pdo->prepare('SELECT COALESCE(SUM(score),0) AS total FROM underwriting_steps WHERE case_id=:cid');
  $stmt->execute([':cid'=>$caseId]);
  $total = (int)($stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0);
  if ($total < 0) $total = 0;
  if ($total > 100) $total = 100;
  $pdo->prepare('UPDATE underwriting_cases SET risk_score=:r, updated_at=NOW() WHERE id=:id')
      ->execute([':r'=>$total, ':id'=>$caseId]);
}

function uw_http_request(string $url, string $method, array $headers, ?string $body = null, int $timeout = 30): array {
  $ch = curl_init($url);
  curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
  curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
  curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
  curl_setopt($ch, CURLOPT_TIMEOUT, $timeout);
  curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
  curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
  curl_setopt($ch, CURLOPT_MAXREDIRS, 3);
  if ($body !== null) curl_setopt($ch, CURLOPT_POSTFIELDS, $body);
  $resp = curl_exec($ch);
  $err = curl_error($ch);
  $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);
  return ['ok'=>($err==='' && $code>=200 && $code<300), 'code'=>$code, 'body'=>(string)$resp, 'error'=>$err];
}

function uw_http_json(string $url, string $method, array $headers, array $payload, int $timeout = 30): array {
  $headers[] = 'Content-Type: application/json';
  return uw_http_request($url, $method, $headers, json_encode($payload), $timeout);
}

function uw_http_form(string $url, string $method, array $headers, array $fields, int $timeout = 30): array {
  $headers[] = 'Content-Type: application/x-www-form-urlencoded';
  return uw_http_request($url, $method, $headers, http_build_query($fields), $timeout);
}

function uw_openai_upload_file(string $apiKey, string $filePath, string $purpose = 'assistants'): array {
  $ch = curl_init('https://api.openai.com/v1/files');
  $post = [
    'purpose' => $purpose,
    'file' => new CURLFile($filePath),
  ];
  curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
  curl_setopt($ch, CURLOPT_POST, true);
  curl_setopt($ch, CURLOPT_POSTFIELDS, $post);
  curl_setopt($ch, CURLOPT_HTTPHEADER, ['Authorization: Bearer ' . $apiKey]);
  $resp = curl_exec($ch);
  $err = curl_error($ch);
  $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);
  $json = json_decode((string)$resp, true);
  return ['ok'=>($err==='' && $code>=200 && $code<300), 'code'=>$code, 'error'=>$err, 'raw'=>(string)$resp, 'json'=>is_array($json)?$json:[]];
}

function uw_openai_responses(string $apiKey, array $payload): array {
  return uw_http_json('https://api.openai.com/v1/responses', 'POST', ['Authorization: Bearer ' . $apiKey], $payload, 60);
}

function uw_softpull_auth(array $settings): array {
  $base = rtrim((string)$settings['softpull_base_url'], '/');
  $resp = uw_http_json($base . '/api/Authentication/AuthenticationToken', 'POST', ['accept: */*'], [
    'userName' => (string)$settings['softpull_username'],
    'password' => (string)$settings['softpull_password'],
  ], 30);
  $json = json_decode($resp['body'] ?? '', true);
  $token = '';
  if (is_array($json)) $token = (string)($json['token'] ?? $json['access_token'] ?? $json['Token'] ?? '');
  return ['ok'=>$resp['ok'] && $token!=='', 'token'=>$token, 'raw'=>$resp];
}

function uw_softpull_standard_inquiry(array $settings, string $bearerToken, array $fields): array {
  $base = rtrim((string)$settings['softpull_base_url'], '/');
  return uw_http_form($base . '/api/CreditReport/standardInquiry', 'POST', ['Authorization: Bearer ' . $bearerToken], $fields, 60);
}

function uw_softpull_review_report(array $settings, string $bearerToken, array $fields): array {
  $base = rtrim((string)$settings['softpull_base_url'], '/');
  return uw_http_form($base . '/api/SoftPullSolutions/reviewReport', 'POST', ['Authorization: Bearer ' . $bearerToken], $fields, 60);
}

function uw_softpull_score(array $reportJson): array {
  $risk = 0;
  $reasons = [];

  $subject = $reportJson['subjects'][0] ?? [];
  $summary = $subject['bureauCreditSummary'] ?? [];

  $collections = (int)($summary['collections'] ?? 0);
  $publicRecords = (int)($summary['publicRecords'] ?? 0);
  $negativeTrades = (int)($summary['negativeTrades'] ?? 0);
  $tradesAnyNeg = (int)($summary['tradesWithAnyHistoricalNegative'] ?? 0);
  $inq = (int)($summary['inquiries'] ?? 0);

  $scoreVal = null;
  if (!empty($subject['scores'][0]['score'])) $scoreVal = (int)$subject['scores'][0]['score'];

  if ($scoreVal !== null) {
    if ($scoreVal < 520) { $risk += 40; $reasons[] = 'Very low credit score (<520).'; }
    else if ($scoreVal < 580) { $risk += 30; $reasons[] = 'Low credit score (<580).'; }
    else if ($scoreVal < 640) { $risk += 18; $reasons[] = 'Below-average credit score (<640).'; }
    else if ($scoreVal < 700) { $risk += 8; $reasons[] = 'Moderate credit score (<700).'; }
  } else {
    $risk += 12;
    $reasons[] = 'No bureau score returned (thin file or suppressed).';
  }

  if ($collections > 0) { $risk += min(30, 10 + $collections * 5); $reasons[] = 'Collections present.'; }
  if ($publicRecords > 0) { $risk += 25; $reasons[] = 'Public records present.'; }
  if ($negativeTrades > 0) { $risk += min(25, 10 + $negativeTrades * 5); $reasons[] = 'Negative trades present.'; }
  if ($tradesAnyNeg > 0) { $risk += min(20, 5 + $tradesAnyNeg * 3); $reasons[] = 'Historical negatives present.'; }
  if ($inq >= 6) { $risk += 12; $reasons[] = 'High recent inquiry volume.'; }
  else if ($inq >= 3) { $risk += 6; $reasons[] = 'Moderate inquiry volume.'; }

  if ($risk < 0) $risk = 0;
  if ($risk > 100) $risk = 100;

  $grade = 'C';
  if ($risk <= 20) $grade = 'A';
  else if ($risk <= 35) $grade = 'B';
  else if ($risk <= 55) $grade = 'C';
  else if ($risk <= 75) $grade = 'D';
  else $grade = 'F';

  $p3  = min(0.95, max(0.02, 0.05 + ($risk / 100) * 0.35));
  $p6  = min(0.97, max(0.03, 0.08 + ($risk / 100) * 0.45));
  $p9  = min(0.98, max(0.04, 0.10 + ($risk / 100) * 0.52));
  $p12 = min(0.99, max(0.05, 0.12 + ($risk / 100) * 0.60));

  return [
    'risk_score' => $risk,
    'grade' => $grade,
    'bureau_score' => $scoreVal,
    'default_risk' => ['m3'=>round($p3,2), 'm6'=>round($p6,2), 'm9'=>round($p9,2), 'm12'=>round($p12,2)],
    'reasons' => array_values(array_unique($reasons)),
  ];
}

function uw_google_cse_search(array $settings, string $query, int $num = 5): array {
  $key = trim((string)$settings['google_cse_key']);
  $cx = trim((string)$settings['google_cse_cx']);
  if ($key === '' || $cx === '') return ['ok'=>false, 'error'=>'Google CSE key/cx not set.', 'items'=>[]];

  $url = 'https://www.googleapis.com/customsearch/v1?key=' . rawurlencode($key) . '&cx=' . rawurlencode($cx) . '&q=' . rawurlencode($query) . '&num=' . (int)$num;
  $resp = uw_http_request($url, 'GET', ['accept: application/json'], null, 25);
  $json = json_decode($resp['body'] ?? '', true);
  $items = [];
  if (is_array($json) && !empty($json['items']) && is_array($json['items'])) {
    foreach ($json['items'] as $it) {
      $items[] = [
        'title' => (string)($it['title'] ?? ''),
        'link' => (string)($it['link'] ?? ''),
        'snippet' => (string)($it['snippet'] ?? ''),
      ];
    }
  }
  return ['ok'=>$resp['ok'], 'error'=>$resp['error'] ?? '', 'items'=>$items, 'raw'=>$resp['body'] ?? ''];
}
