<?php
// /app/webhooks/shedsuite/order.php
declare(strict_types=1);

/**
 * Single ShedSuite webhook endpoint (SaaS multi-tenant).
 *
 * Each company gets a unique URL by including their api key:
 *   POST /app/webhooks/shedsuite/order.php?key=YOUR_40_CHAR_KEY
 * Or:
 *   Header: X-ShedOffice-Webhook-Key: YOUR_40_CHAR_KEY
 *
 * If key is missing/invalid => request is rejected.
 */

require_once __DIR__ . '/../../includes/config.php';
require_once __DIR__ . '/../../includes/db.php';

header('Content-Type: application/json; charset=utf-8');

$pdo = db();

$key = '';
if (!empty($_SERVER['HTTP_X_SHEDOFFICE_WEBHOOK_KEY'])) $key = trim((string)$_SERVER['HTTP_X_SHEDOFFICE_WEBHOOK_KEY']);
if ($key === '' && isset($_GET['key'])) $key = trim((string)$_GET['key']);

if ($key === '') { http_response_code(401); echo json_encode(['ok'=>false,'msg'=>'Missing webhook key']); exit; }

$k = $pdo->prepare("SELECT id, tenant_id FROM webhook_keys WHERE api_key=:k AND is_enabled=1 LIMIT 1");
$k->execute([':k'=>$key]);
$row = $k->fetch(PDO::FETCH_ASSOC);
if (!$row) { http_response_code(403); echo json_encode(['ok'=>false,'msg'=>'Invalid webhook key']); exit; }

$tenantId = (int)$row['tenant_id'];
$sourceIp = $_SERVER['REMOTE_ADDR'] ?? null;

$raw = file_get_contents('php://input');
$payload = json_decode((string)$raw, true);
if (!is_array($payload)) $payload = [];

$eventName = (string)($payload['event'] ?? $payload['event_name'] ?? $payload['type'] ?? '');
$order = $payload['order'] ?? ($payload['data']['order'] ?? ($payload['data'] ?? $payload));
if (!is_array($order)) $order = [];

$status = 'unknown';
$eventLower = strtolower($eventName);
if (str_contains($eventLower, 'submitted')) $status = 'submitted';
elseif (str_contains($eventLower, 'processed')) $status = 'processed';
elseif (str_contains($eventLower, 'delivered')) $status = 'delivered';
elseif (str_contains($eventLower, 'cancel')) $status = 'canceled';

$orderNumber = (string)($order['order_number'] ?? $order['number'] ?? $order['orderNumber'] ?? '');
if ($orderNumber === '') $orderNumber = (string)($order['id'] ?? $order['order_id'] ?? '');

$externalOrderId = (string)($order['order_id'] ?? $order['id'] ?? '');
if ($externalOrderId === '') $externalOrderId = $orderNumber;

$headers = [];
foreach ($_SERVER as $hk=>$hv) {
  if (str_starts_with($hk, 'HTTP_')) $headers[$hk] = (string)$hv;
}

$inboxId = null;
try {
  $ins = $pdo->prepare("
    INSERT INTO webhook_inbox
      (tenant_id, provider, event_name, external_order_id, source_ip, headers_json, payload_json)
    VALUES
      (:tid, 'shedsuite', :event, :oid, :ip, :hdrs, :payload)
  ");
  $ins->execute([
    ':tid'=>$tenantId,
    ':event'=>$eventName ?: null,
    ':oid'=>$externalOrderId ?: null,
    ':ip'=>$sourceIp,
    ':hdrs'=>json_encode($headers, JSON_UNESCAPED_SLASHES),
    ':payload'=>json_encode($payload, JSON_UNESCAPED_SLASHES),
  ]);
  $inboxId = (int)$pdo->lastInsertId();
} catch (Throwable $e) {
  // inbox is optional; ignore failures
}

/* Upsert order into tenant-scoped orders table */
try {
  $customer = $order['customer'] ?? [];
  $delivery = $order['delivery_address'] ?? $order['delivery'] ?? [];

  $tsNow = gmdate('Y-m-d H:i:s');
  $submittedAt = $status==='submitted' ? $tsNow : null;
  $processedAt = $status==='processed' ? $tsNow : null;
  $deliveredAt = $status==='delivered' ? $tsNow : null;
  $cancelAt    = $status==='canceled'  ? $tsNow : null;

  $sql = "
    INSERT INTO orders
      (tenant_id, order_number, order_event, status, serial_number, model, size,
       customer_name, customer_phone_primary, customer_email,
       delivery_street_line_one, delivery_street_line_two, delivery_city, delivery_state, delivery_zip,
       dest_lat, dest_lng, addons_json, custom_addons_json, idempotency_key,
       submitted_at, processed_at, delivered_at, cancel_date)
    VALUES
      (:tenant_id, :order_number, :order_event, :status, :serial_number, :model, :size,
       :customer_name, :customer_phone_primary, :customer_email,
       :delivery_street_line_one, :delivery_street_line_two, :delivery_city, :delivery_state, :delivery_zip,
       :dest_lat, :dest_lng, :addons_json, :custom_addons_json, :idempotency_key,
       :submitted_at, :processed_at, :delivered_at, :cancel_date)
    ON DUPLICATE KEY UPDATE
      order_event=VALUES(order_event),
      status=VALUES(status),
      serial_number=COALESCE(NULLIF(VALUES(serial_number),''), serial_number),
      model=COALESCE(NULLIF(VALUES(model),''), model),
      size=COALESCE(NULLIF(VALUES(size),''), size),
      customer_name=COALESCE(NULLIF(VALUES(customer_name),''), customer_name),
      customer_phone_primary=COALESCE(NULLIF(VALUES(customer_phone_primary),''), customer_phone_primary),
      customer_email=COALESCE(NULLIF(VALUES(customer_email),''), customer_email),
      delivery_street_line_one=COALESCE(NULLIF(VALUES(delivery_street_line_one),''), delivery_street_line_one),
      delivery_street_line_two=COALESCE(NULLIF(VALUES(delivery_street_line_two),''), delivery_street_line_two),
      delivery_city=COALESCE(NULLIF(VALUES(delivery_city),''), delivery_city),
      delivery_state=COALESCE(NULLIF(VALUES(delivery_state),''), delivery_state),
      delivery_zip=COALESCE(NULLIF(VALUES(delivery_zip),''), delivery_zip),
      dest_lat=COALESCE(VALUES(dest_lat), dest_lat),
      dest_lng=COALESCE(VALUES(dest_lng), dest_lng),
      addons_json=COALESCE(VALUES(addons_json), addons_json),
      custom_addons_json=COALESCE(VALUES(custom_addons_json), custom_addons_json),
      idempotency_key=COALESCE(NULLIF(VALUES(idempotency_key),''), idempotency_key),
      submitted_at=COALESCE(VALUES(submitted_at), submitted_at),
      processed_at=COALESCE(VALUES(processed_at), processed_at),
      delivered_at=COALESCE(VALUES(delivered_at), delivered_at),
      cancel_date=COALESCE(VALUES(cancel_date), cancel_date)
  ";

  $stmt = $pdo->prepare($sql);
  $stmt->execute([
    ':tenant_id'=>$tenantId,
    ':order_number'=>$orderNumber,
    ':order_event'=>$eventName ?: 'order',
    ':status'=>$status,
    ':serial_number'=>(string)($order['serial_number'] ?? $order['building_serial_number'] ?? ''),
    ':model'=>(string)($order['model'] ?? ''),
    ':size'=>(string)($order['size'] ?? ''),
    ':customer_name'=>(string)($customer['name'] ?? $order['customer_name'] ?? ''),
    ':customer_phone_primary'=>(string)($customer['phone'] ?? $order['customer_phone_primary'] ?? ''),
    ':customer_email'=>(string)($customer['email'] ?? $order['customer_email'] ?? ''),
    ':delivery_street_line_one'=>(string)($delivery['street_line_one'] ?? $delivery['line1'] ?? $delivery['address1'] ?? ''),
    ':delivery_street_line_two'=>(string)($delivery['street_line_two'] ?? $delivery['line2'] ?? $delivery['address2'] ?? ''),
    ':delivery_city'=>(string)($delivery['city'] ?? ''),
    ':delivery_state'=>(string)($delivery['state'] ?? ''),
    ':delivery_zip'=>(string)($delivery['zip'] ?? $delivery['postal_code'] ?? ''),
    ':dest_lat'=>isset($order['dest_lat']) ? (float)$order['dest_lat'] : (isset($delivery['lat']) ? (float)$delivery['lat'] : None),
    ':dest_lng'=>isset($order['dest_lng']) ? (float)$order['dest_lng'] : (isset($delivery['lng']) ? (float)$delivery['lng'] : None),
    ':addons_json'=>isset($order['addons']) ? json_encode($order['addons']) : (isset($order['addons_json']) ? (string)$order['addons_json'] : null),
    ':custom_addons_json'=>isset($order['custom_addons']) ? json_encode($order['custom_addons']) : (isset($order['custom_addons_json']) ? (string)$order['custom_addons_json'] : null),
    ':idempotency_key'=>(string)($order['idempotency_key'] ?? ''),
    ':submitted_at'=>$submittedAt,
    ':processed_at'=>$processedAt,
    ':delivered_at'=>$deliveredAt,
    ':cancel_date'=>$cancelAt,
  ]);

  $pdo->prepare("UPDATE webhook_keys SET last_used_at=UTC_TIMESTAMP(), last_used_ip=:ip WHERE id=:id")
      ->execute([':ip'=>$sourceIp, ':id'=>(int)$row['id']]);

  if ($inboxId) {
    $pdo->prepare("UPDATE webhook_inbox SET processed=1, processed_at=UTC_TIMESTAMP() WHERE id=:id")->execute([':id'=>$inboxId]);
  }

  echo json_encode(['ok'=>true,'tenant_id'=>$tenantId,'order_number'=>$orderNumber,'status'=>$status]);
} catch (Throwable $e) {
  if ($inboxId) {
    $pdo->prepare("UPDATE webhook_inbox SET error_message=:m WHERE id=:id")->execute([':m'=>substr($e->getMessage(),0,250), ':id'=>$inboxId]);
  }
  http_response_code(400);
  echo json_encode(['ok'=>false,'msg'=>$e->getMessage()]);
}
