<?php
// /app/underwriting/view.php
declare(strict_types=1);

require_once __DIR__ . '/../includes/auth.php';
require_company_context_if_rto();

$pdo = db();

$pageTitle = 'Underwriting';
$activeNav = 'underwriting';
require_once __DIR__ . '/../includes/header.php';

/**
 * Underwriting step keys (canonical)
 */
const UW_STEPS = [
  'id_verification' => 'AI ID Verification',
  'softpull'        => 'Soft Credit Pull',
  'blocks'          => 'Blocks Acknowledgement',
  'delivery_risk'   => 'Delivery Location Risk',
  'public_records'  => 'Public Records Scan',
  'signature_match' => 'Signature Match',
  'people_history'  => 'Name/Address History',
];

/**
 * Safe users display field detection (NO hard dependency on users.name)
 */
function users_display_expr(PDO $pdo, string $alias = 'u'): string {
  $stmt = $pdo->prepare("
    SELECT column_name
    FROM information_schema.columns
    WHERE table_schema = DATABASE()
      AND table_name = 'users'
      AND column_name IN ('name','user_name','first_name','last_name','email')
  ");
  $stmt->execute();
  $cols = array_flip(array_map('strtolower', $stmt->fetchAll(PDO::FETCH_COLUMN)));

  if (isset($cols['name'])) {
    return "{$alias}.name";
  }
  if (isset($cols['user_name'])) {
    return "{$alias}.user_name";
  }
  if (isset($cols['first_name']) && isset($cols['last_name'])) {
    return "TRIM(CONCAT(COALESCE({$alias}.first_name,''), ' ', COALESCE({$alias}.last_name,'')))";
  }
  if (isset($cols['email'])) {
    return "{$alias}.email";
  }
  // absolute fallback (never breaks)
  return "CONCAT('User #', CAST(n.user_id AS CHAR))";
}

function badge_class_for_step(string $status): string {
  return match ($status) {
    'pass'   => 'bg-success',
    'fail'   => 'bg-danger',
    'review' => 'bg-warning text-dark',
    'done'   => 'bg-info text-dark',
    default  => 'bg-secondary',
  };
}

function label_for_step_status(string $status): string {
  return match ($status) {
    'pass'   => 'Pass',
    'fail'   => 'Fail',
    'review' => 'Review',
    'done'   => 'Done',
    default  => 'Pending',
  };
}

function safe_json_decode(?string $json): array {
  if (!$json) return [];
  $d = json_decode($json, true);
  return is_array($d) ? $d : [];
}

$tenantId = (int) current_tenant_id();
$tenantType = (string) current_tenant_type();

// Inputs
$caseId  = isset($_GET['case_id']) ? (int)$_GET['case_id'] : 0;
$orderId = isset($_GET['order_id']) ? (int)$_GET['order_id'] : 0;

if ($caseId < 1 && $orderId < 1) {
  http_response_code(400);
  echo '<div class="container-fluid"><div class="alert alert-danger">Missing case_id or order_id.</div></div>';
  require_once __DIR__ . '/../includes/footer.php';
  exit;
}

/**
 * Load or create case (if order_id provided)
 */
$case = null;

if ($caseId > 0) {
  $st = $pdo->prepare("SELECT * FROM underwriting_cases WHERE id = ? AND tenant_id = ?");
  $st->execute([$caseId, $tenantId]);
  $case = $st->fetch(PDO::FETCH_ASSOC) ?: null;
  if (!$case) {
    http_response_code(404);
    echo '<div class="container-fluid"><div class="alert alert-danger">Underwriting case not found.</div></div>';
    require_once __DIR__ . '/../includes/footer.php';
    exit;
  }
  $orderId = (int)($case['order_id'] ?? 0);
} else {
  // Find existing case for order, else create
  $st = $pdo->prepare("SELECT * FROM underwriting_cases WHERE tenant_id = ? AND order_id = ? LIMIT 1");
  $st->execute([$tenantId, $orderId]);
  $case = $st->fetch(PDO::FETCH_ASSOC) ?: null;

  if (!$case) {
    $ins = $pdo->prepare("
      INSERT INTO underwriting_cases (tenant_id, order_id, status, risk_score, created_at, updated_at)
      VALUES (?, ?, 'open', 0, NOW(), NOW())
    ");
    $ins->execute([$tenantId, $orderId]);

    $caseId = (int) $pdo->lastInsertId();

    $st = $pdo->prepare("SELECT * FROM underwriting_cases WHERE id = ? AND tenant_id = ?");
    $st->execute([$caseId, $tenantId]);
    $case = $st->fetch(PDO::FETCH_ASSOC) ?: null;
  } else {
    $caseId = (int)$case['id'];
  }
}

if (!$case) {
  http_response_code(500);
  echo '<div class="container-fluid"><div class="alert alert-danger">Failed to load underwriting case.</div></div>';
  require_once __DIR__ . '/../includes/footer.php';
  exit;
}

/**
 * Ensure steps exist
 */
$st = $pdo->prepare("SELECT step_key FROM underwriting_steps WHERE case_id = ?");
$st->execute([$caseId]);
$existingKeys = $st->fetchAll(PDO::FETCH_COLUMN);
$existingKeys = array_map('strval', $existingKeys);
$existingMap = array_flip($existingKeys);

$pdo->beginTransaction();
try {
  foreach (UW_STEPS as $key => $label) {
    if (!isset($existingMap[$key])) {
      $ins = $pdo->prepare("
        INSERT INTO underwriting_steps (case_id, step_key, status, score, details_json, created_at, updated_at)
        VALUES (?, ?, 'pending', 0, NULL, NOW(), NOW())
      ");
      $ins->execute([$caseId, $key]);
    }
  }
  $pdo->commit();
} catch (Throwable $e) {
  $pdo->rollBack();
  // Continue; do not hard-fail the page
}

/**
 * Load steps
 */
$st = $pdo->prepare("SELECT * FROM underwriting_steps WHERE case_id = ? ORDER BY id ASC");
$st->execute([$caseId]);
$steps = $st->fetchAll(PDO::FETCH_ASSOC);

$stepsByKey = [];
foreach ($steps as $row) {
  $stepsByKey[(string)$row['step_key']] = $row;
}

/**
 * Load documents
 */
$st = $pdo->prepare("SELECT * FROM underwriting_documents WHERE case_id = ? ORDER BY id ASC");
$st->execute([$caseId]);
$docs = $st->fetchAll(PDO::FETCH_ASSOC);

$docsByType = [];
foreach ($docs as $d) {
  $docsByType[(string)$d['doc_type']] = $d;
}

/**
 * Load notes (schema-tolerant: no u.name dependency)
 */
$uDisp = users_display_expr($pdo, 'u');

$notesSql = "
  SELECT
    n.*,
    {$uDisp} AS user_display
  FROM underwriting_notes n
  LEFT JOIN users u ON u.id = n.user_id
  WHERE n.case_id = ?
  ORDER BY n.created_at DESC
";
$st = $pdo->prepare($notesSql);
$st->execute([$caseId]);
$notes = $st->fetchAll(PDO::FETCH_ASSOC);

/**
 * Load order (best-effort: SELECT * to avoid unknown-column errors)
 */
$order = null;
if ($orderId > 0) {
  $st = $pdo->prepare("SELECT * FROM orders WHERE id = ? LIMIT 1");
  $st->execute([$orderId]);
  $order = $st->fetch(PDO::FETCH_ASSOC) ?: null;
}

// Derived fields (best-effort)
$caseStatus = (string)($case['status'] ?? 'open');
$riskScore  = (int)($case['risk_score'] ?? 0);

$decisionBadge = match ($caseStatus) {
  'approved'     => 'bg-success',
  'conditional'  => 'bg-warning text-dark',
  'denied'       => 'bg-danger',
  default        => 'bg-secondary',
};

$csrf = csrf_token();

// Helpers to read “order” fields safely across schemas
function ov(?array $order, string $key, string $fallback = ''): string {
  if (!$order) return $fallback;
  $v = $order[$key] ?? $fallback;
  return is_scalar($v) ? (string)$v : $fallback;
}

$orderTitle = 'Order #' . (string)$orderId;
$customerDisplay =
  ov($order, 'customer_name') ? ov($order, 'customer_name') :
  (trim(ov($order, 'first_name') . ' ' . ov($order, 'last_name')) !== '' ? trim(ov($order, 'first_name') . ' ' . ov($order, 'last_name')) :
  (ov($order, 'customer_email') ?: ov($order, 'email', '')));

$deliveryLine1 = ov($order, 'delivery_address', ov($order, 'delivery_address1', ''));
$deliveryCity  = ov($order, 'delivery_city', '');
$deliveryState = ov($order, 'delivery_state', '');
$deliveryZip   = ov($order, 'delivery_zip', '');
$deliveryFull  = trim($deliveryLine1 . ($deliveryCity ? ', ' . $deliveryCity : '') . ($deliveryState ? ', ' . $deliveryState : '') . ($deliveryZip ? ' ' . $deliveryZip : ''));

$dealAmount = ov($order, 'total', ov($order, 'total_price', ov($order, 'order_total', '')));
$orderStatus = ov($order, 'order_status', ov($order, 'status', ''));

// Step summaries
$idStep   = $stepsByKey['id_verification'] ?? null;
$spStep   = $stepsByKey['softpull'] ?? null;
$blkStep  = $stepsByKey['blocks'] ?? null;
$delStep  = $stepsByKey['delivery_risk'] ?? null;
$prsStep  = $stepsByKey['public_records'] ?? null;
$sigStep  = $stepsByKey['signature_match'] ?? null;

$idDetails  = safe_json_decode($idStep['details_json'] ?? null);
$spDetails  = safe_json_decode($spStep['details_json'] ?? null);
$delDetails = safe_json_decode($delStep['details_json'] ?? null);
$prsDetails = safe_json_decode($prsStep['details_json'] ?? null);
$sigDetails = safe_json_decode($sigStep['details_json'] ?? null);

?>
<div class="container-fluid">

  <div class="row mb-3">
    <div class="col-12">
      <div class="d-flex align-items-center justify-content-between flex-wrap gap-2">
        <div>
          <div class="so-page-title">Underwriting</div>
          <div class="so-page-sub">
            <?= h($orderTitle) ?>
            <?php if ($customerDisplay !== ''): ?>
              <span style="opacity:.7;">•</span> <?= h($customerDisplay) ?>
            <?php endif; ?>
            <?php if ($deliveryFull !== ''): ?>
              <span style="opacity:.7;">•</span> <?= h($deliveryFull) ?>
            <?php endif; ?>
          </div>
        </div>

        <div class="d-flex align-items-center gap-2">
          <span class="badge <?= $decisionBadge ?> px-3 py-2"><?= h(strtoupper($caseStatus)) ?></span>
          <span class="badge bg-dark px-3 py-2" style="border:1px solid rgba(255,255,255,.08);">
            Risk Score: <strong><?= (int)$riskScore ?></strong>/100
          </span>
          <a class="btn btn-outline-light" href="<?= app_path('underwriting/index.php') ?>">
            <i class="fa-solid fa-arrow-left me-2"></i>Back to Queue
          </a>
        </div>
      </div>
    </div>
  </div>

  <!-- CASE META -->
  <div class="row g-3 mb-3">
    <div class="col-12 col-xl-8">
      <div class="so-card p-3">
        <div class="d-flex align-items-start justify-content-between flex-wrap gap-2">
          <div>
            <div class="fw-semibold" style="font-size:1.05rem;">Deal Overview</div>
            <div class="mt-2" style="opacity:.85;">
              <div><strong>Order ID:</strong> <?= (int)$orderId ?></div>
              <?php if ($orderStatus !== ''): ?><div><strong>Order Status:</strong> <?= h($orderStatus) ?></div><?php endif; ?>
              <?php if ($dealAmount !== ''): ?><div><strong>Deal Amount:</strong> <?= h($dealAmount) ?></div><?php endif; ?>
              <?php if ($deliveryFull !== ''): ?><div><strong>Delivery:</strong> <?= h($deliveryFull) ?></div><?php endif; ?>
            </div>
          </div>

          <div class="d-flex flex-wrap gap-2">
            <form method="post" action="<?= app_path('underwriting/actions/decide.php') ?>" class="d-flex gap-2">
              <input type="hidden" name="_csrf" value="<?= h($csrf) ?>">
              <input type="hidden" name="case_id" value="<?= (int)$caseId ?>">
              <input type="hidden" name="decision" value="approved">
              <button class="btn btn-success">
                <i class="fa-solid fa-circle-check me-2"></i>Approve
              </button>
            </form>

            <form method="post" action="<?= app_path('underwriting/actions/decide.php') ?>" class="d-flex gap-2">
              <input type="hidden" name="_csrf" value="<?= h($csrf) ?>">
              <input type="hidden" name="case_id" value="<?= (int)$caseId ?>">
              <input type="hidden" name="decision" value="conditional">
              <button class="btn btn-warning">
                <i class="fa-solid fa-triangle-exclamation me-2"></i>Conditional
              </button>
            </form>

            <form method="post" action="<?= app_path('underwriting/actions/decide.php') ?>" class="d-flex gap-2">
              <input type="hidden" name="_csrf" value="<?= h($csrf) ?>">
              <input type="hidden" name="case_id" value="<?= (int)$caseId ?>">
              <input type="hidden" name="decision" value="denied">
              <button class="btn btn-danger">
                <i class="fa-solid fa-ban me-2"></i>Deny
              </button>
            </form>
          </div>
        </div>

        <div class="mt-3">
          <form method="post" action="<?= app_path('underwriting/actions/decide.php') ?>" class="row g-2 align-items-center">
            <input type="hidden" name="_csrf" value="<?= h($csrf) ?>">
            <input type="hidden" name="case_id" value="<?= (int)$caseId ?>">
            <div class="col-12 col-lg-9">
              <input type="text" class="form-control" name="decision_reason" placeholder="Decision notes / reason codes (optional)" value="<?= h((string)($case['decision_reason'] ?? '')) ?>">
            </div>
            <div class="col-12 col-lg-3 d-grid">
              <button class="btn btn-outline-light">
                <i class="fa-solid fa-floppy-disk me-2"></i>Save Decision Notes
              </button>
            </div>
          </form>
        </div>

      </div>
    </div>

    <div class="col-12 col-xl-4">
      <div class="so-card p-3">
        <div class="fw-semibold" style="font-size:1.05rem;">Documents</div>
        <div class="mt-2" style="opacity:.9;">
          <?php
          $docTypes = [
            'id_front' => 'Driver License (Front)',
            'id_back'  => 'Driver License (Back)',
            'contract' => 'Contract / RTO',
            'invoice'  => 'Invoice / Order Doc',
          ];
          ?>
          <?php foreach ($docTypes as $dt => $label): ?>
            <div class="d-flex align-items-center justify-content-between py-1">
              <div><?= h($label) ?></div>
              <div>
                <?php if (!empty($docsByType[$dt]['url'])): ?>
                  <a class="btn btn-sm btn-outline-light" target="_blank" href="<?= h((string)$docsByType[$dt]['url']) ?>">
                    <i class="fa-solid fa-arrow-up-right-from-square me-1"></i>Open
                  </a>
                <?php else: ?>
                  <span class="badge bg-secondary">Not set</span>
                <?php endif; ?>
              </div>
            </div>
          <?php endforeach; ?>
        </div>
      </div>
    </div>
  </div>

  <!-- STEPS -->
  <div class="row g-3 mb-3">
    <div class="col-12 col-xl-8">

      <!-- STEP: ID Verification -->
      <?php $stRow = $stepsByKey['id_verification'] ?? null; ?>
      <div class="so-card p-3 mb-3">
        <div class="d-flex align-items-center justify-content-between flex-wrap gap-2">
          <div class="d-flex align-items-center gap-2">
            <div class="fw-semibold" style="font-size:1.05rem;">
              1) <?= h(UW_STEPS['id_verification']) ?>
            </div>
            <span class="badge <?= badge_class_for_step((string)($stRow['status'] ?? 'pending')) ?>">
              <?= h(label_for_step_status((string)($stRow['status'] ?? 'pending'))) ?>
            </span>
            <span class="badge bg-dark" style="border:1px solid rgba(255,255,255,.08);">
              Score: <?= (int)($stRow['score'] ?? 0) ?>
            </span>
          </div>

          <div class="d-flex gap-2">
            <form method="post" action="<?= app_path('underwriting/actions/run_id_ai.php') ?>">
              <input type="hidden" name="_csrf" value="<?= h($csrf) ?>">
              <input type="hidden" name="case_id" value="<?= (int)$caseId ?>">
              <button class="btn btn-outline-light">
                <i class="fa-solid fa-wand-magic-sparkles me-2"></i>Run AI Check
              </button>
            </form>
          </div>
        </div>

        <div class="mt-3">
          <form method="post" action="<?= app_path('underwriting/actions/upload_id.php') ?>" enctype="multipart/form-data" class="row g-2">
            <input type="hidden" name="_csrf" value="<?= h($csrf) ?>">
            <input type="hidden" name="case_id" value="<?= (int)$caseId ?>">
            <div class="col-12 col-lg-5">
              <label class="form-label" style="opacity:.9;">ID Front</label>
              <input class="form-control" type="file" name="id_front" accept="image/*">
            </div>
            <div class="col-12 col-lg-5">
              <label class="form-label" style="opacity:.9;">ID Back</label>
              <input class="form-control" type="file" name="id_back" accept="image/*">
            </div>
            <div class="col-12 col-lg-2 d-grid align-self-end">
              <button class="btn btn-primary">
                <i class="fa-solid fa-upload me-2"></i>Upload
              </button>
            </div>
          </form>

          <?php if (!empty($idDetails)): ?>
            <div class="mt-3 p-3 rounded" style="background:rgba(255,255,255,0.03); border:1px solid rgba(255,255,255,0.06);">
              <div class="fw-semibold">AI Result</div>
              <div class="mt-2">
                <?php if (!empty($idDetails['verdict'])): ?>
                  <div><strong>Verdict:</strong> <?= h((string)$idDetails['verdict']) ?></div>
                <?php endif; ?>
                <?php if (isset($idDetails['match_confidence'])): ?>
                  <div><strong>Confidence:</strong> <?= (int)$idDetails['match_confidence'] ?></div>
                <?php endif; ?>
                <?php if (!empty($idDetails['reasons']) && is_array($idDetails['reasons'])): ?>
                  <div class="mt-2">
                    <strong>Reasons:</strong>
                    <ul class="mb-0">
                      <?php foreach ($idDetails['reasons'] as $r): ?>
                        <li><?= h((string)$r) ?></li>
                      <?php endforeach; ?>
                    </ul>
                  </div>
                <?php endif; ?>
              </div>
            </div>
          <?php endif; ?>

        </div>
      </div>

      <!-- STEP: SoftPull -->
      <?php $stRow = $stepsByKey['softpull'] ?? null; ?>
      <div class="so-card p-3 mb-3">
        <div class="d-flex align-items-center justify-content-between flex-wrap gap-2">
          <div class="d-flex align-items-center gap-2">
            <div class="fw-semibold" style="font-size:1.05rem;">
              2) <?= h(UW_STEPS['softpull']) ?>
            </div>
            <span class="badge <?= badge_class_for_step((string)($stRow['status'] ?? 'pending')) ?>">
              <?= h(label_for_step_status((string)($stRow['status'] ?? 'pending'))) ?>
            </span>
            <span class="badge bg-dark" style="border:1px solid rgba(255,255,255,.08);">
              Score: <?= (int)($stRow['score'] ?? 0) ?>
            </span>
          </div>

          <form method="post" action="<?= app_path('underwriting/actions/run_softpull.php') ?>">
            <input type="hidden" name="_csrf" value="<?= h($csrf) ?>">
            <input type="hidden" name="case_id" value="<?= (int)$caseId ?>">
            <button class="btn btn-outline-light">
              <i class="fa-solid fa-chart-line me-2"></i>Run Soft Pull
            </button>
          </form>
        </div>

        <?php if (!empty($spDetails)): ?>
          <div class="mt-3 p-3 rounded" style="background:rgba(255,255,255,0.03); border:1px solid rgba(255,255,255,0.06);">
            <div class="fw-semibold">Soft Pull Summary</div>
            <div class="mt-2">
              <?php if (!empty($spDetails['grade'])): ?>
                <div><strong>Grade:</strong> <?= h((string)$spDetails['grade']) ?></div>
              <?php endif; ?>
              <?php if (isset($spDetails['rto_risk_score'])): ?>
                <div><strong>RTO Risk:</strong> <?= (int)$spDetails['rto_risk_score'] ?>/100</div>
              <?php endif; ?>
              <?php if (!empty($spDetails['reason_codes']) && is_array($spDetails['reason_codes'])): ?>
                <div class="mt-2">
                  <strong>Reason Codes:</strong>
                  <ul class="mb-0">
                    <?php foreach ($spDetails['reason_codes'] as $r): ?>
                      <li><?= h((string)$r) ?></li>
                    <?php endforeach; ?>
                  </ul>
                </div>
              <?php endif; ?>
            </div>
          </div>
        <?php endif; ?>
      </div>

      <!-- STEP: Blocks -->
      <?php $stRow = $stepsByKey['blocks'] ?? null; ?>
      <div class="so-card p-3 mb-3">
        <div class="d-flex align-items-center justify-content-between flex-wrap gap-2">
          <div class="d-flex align-items-center gap-2">
            <div class="fw-semibold" style="font-size:1.05rem;">
              3) <?= h(UW_STEPS['blocks']) ?>
            </div>
            <span class="badge <?= badge_class_for_step((string)($stRow['status'] ?? 'pending')) ?>">
              <?= h(label_for_step_status((string)($stRow['status'] ?? 'pending'))) ?>
            </span>
            <span class="badge bg-dark" style="border:1px solid rgba(255,255,255,.08);">
              Score: <?= (int)($stRow['score'] ?? 0) ?>
            </span>
          </div>

          <form method="post" action="<?= app_path('underwriting/actions/step_update.php') ?>" class="d-flex gap-2 align-items-center">
            <input type="hidden" name="_csrf" value="<?= h($csrf) ?>">
            <input type="hidden" name="case_id" value="<?= (int)$caseId ?>">
            <input type="hidden" name="step_key" value="blocks">
            <input type="hidden" name="status" value="done">
            <button class="btn btn-outline-light">
              <i class="fa-solid fa-check me-2"></i>Mark Complete
            </button>
          </form>
        </div>

        <div class="mt-3" style="opacity:.9;">
          Confirm the customer has been told blocks are required and agrees.
        </div>
      </div>

      <!-- STEP: Delivery Risk -->
      <?php $stRow = $stepsByKey['delivery_risk'] ?? null; ?>
      <div class="so-card p-3 mb-3">
        <div class="d-flex align-items-center justify-content-between flex-wrap gap-2">
          <div class="d-flex align-items-center gap-2">
            <div class="fw-semibold" style="font-size:1.05rem;">
              4) <?= h(UW_STEPS['delivery_risk']) ?>
            </div>
            <span class="badge <?= badge_class_for_step((string)($stRow['status'] ?? 'pending')) ?>">
              <?= h(label_for_step_status((string)($stRow['status'] ?? 'pending'))) ?>
            </span>
            <span class="badge bg-dark" style="border:1px solid rgba(255,255,255,.08);">
              Score: <?= (int)($stRow['score'] ?? 0) ?>
            </span>
          </div>

          <form method="post" action="<?= app_path('underwriting/actions/run_delivery_risk.php') ?>">
            <input type="hidden" name="_csrf" value="<?= h($csrf) ?>">
            <input type="hidden" name="case_id" value="<?= (int)$caseId ?>">
            <button class="btn btn-outline-light">
              <i class="fa-solid fa-magnifying-glass-location me-2"></i>Run Search Scan
            </button>
          </form>
        </div>

        <?php if (!empty($delDetails)): ?>
          <div class="mt-3 p-3 rounded" style="background:rgba(255,255,255,0.03); border:1px solid rgba(255,255,255,0.06);">
            <div class="fw-semibold">Delivery Risk Summary</div>
            <div class="mt-2">
              <?php if (!empty($delDetails['verdict'])): ?>
                <div><strong>Verdict:</strong> <?= h((string)$delDetails['verdict']) ?></div>
              <?php endif; ?>
              <?php if (isset($delDetails['risk_score'])): ?>
                <div><strong>Risk:</strong> <?= (int)$delDetails['risk_score'] ?>/100</div>
              <?php endif; ?>
              <?php if (!empty($delDetails['summary'])): ?>
                <div class="mt-2"><?= nl2br(h((string)$delDetails['summary'])) ?></div>
              <?php endif; ?>
              <?php if (!empty($delDetails['sources']) && is_array($delDetails['sources'])): ?>
                <div class="mt-2">
                  <strong>Sources:</strong>
                  <ul class="mb-0">
                    <?php foreach ($delDetails['sources'] as $s): ?>
                      <?php
                        $title = (string)($s['title'] ?? 'Source');
                        $link  = (string)($s['link'] ?? '');
                      ?>
                      <li>
                        <?php if ($link !== ''): ?>
                          <a href="<?= h($link) ?>" target="_blank" rel="noopener"><?= h($title) ?></a>
                        <?php else: ?>
                          <?= h($title) ?>
                        <?php endif; ?>
                      </li>
                    <?php endforeach; ?>
                  </ul>
                </div>
              <?php endif; ?>
            </div>
          </div>
        <?php endif; ?>
      </div>

      <!-- STEP: Public Records -->
      <?php $stRow = $stepsByKey['public_records'] ?? null; ?>
      <div class="so-card p-3 mb-3">
        <div class="d-flex align-items-center justify-content-between flex-wrap gap-2">
          <div class="d-flex align-items-center gap-2">
            <div class="fw-semibold" style="font-size:1.05rem;">
              5) <?= h(UW_STEPS['public_records']) ?>
            </div>
            <span class="badge <?= badge_class_for_step((string)($stRow['status'] ?? 'pending')) ?>">
              <?= h(label_for_step_status((string)($stRow['status'] ?? 'pending'))) ?>
            </span>
            <span class="badge bg-dark" style="border:1px solid rgba(255,255,255,.08);">
              Score: <?= (int)($stRow['score'] ?? 0) ?>
            </span>
          </div>

          <form method="post" action="<?= app_path('underwriting/actions/run_public_records.php') ?>">
            <input type="hidden" name="_csrf" value="<?= h($csrf) ?>">
            <input type="hidden" name="case_id" value="<?= (int)$caseId ?>">
            <button class="btn btn-outline-light">
              <i class="fa-solid fa-scale-balanced me-2"></i>Run Search Scan
            </button>
          </form>
        </div>

        <?php if (!empty($prsDetails)): ?>
          <div class="mt-3 p-3 rounded" style="background:rgba(255,255,255,0.03); border:1px solid rgba(255,255,255,0.06);">
            <div class="fw-semibold">Public Records Summary</div>
            <div class="mt-2">
              <?php if (!empty($prsDetails['verdict'])): ?>
                <div><strong>Verdict:</strong> <?= h((string)$prsDetails['verdict']) ?></div>
              <?php endif; ?>
              <?php if (isset($prsDetails['risk_score'])): ?>
                <div><strong>Risk:</strong> <?= (int)$prsDetails['risk_score'] ?>/100</div>
              <?php endif; ?>
              <?php if (!empty($prsDetails['summary'])): ?>
                <div class="mt-2"><?= nl2br(h((string)$prsDetails['summary'])) ?></div>
              <?php endif; ?>
              <?php if (!empty($prsDetails['sources']) && is_array($prsDetails['sources'])): ?>
                <div class="mt-2">
                  <strong>Sources:</strong>
                  <ul class="mb-0">
                    <?php foreach ($prsDetails['sources'] as $s): ?>
                      <?php
                        $title = (string)($s['title'] ?? 'Source');
                        $link  = (string)($s['link'] ?? '');
                      ?>
                      <li>
                        <?php if ($link !== ''): ?>
                          <a href="<?= h($link) ?>" target="_blank" rel="noopener"><?= h($title) ?></a>
                        <?php else: ?>
                          <?= h($title) ?>
                        <?php endif; ?>
                      </li>
                    <?php endforeach; ?>
                  </ul>
                </div>
              <?php endif; ?>
            </div>
          </div>
        <?php endif; ?>
      </div>

      <!-- STEP: Signature Match -->
      <?php $stRow = $stepsByKey['signature_match'] ?? null; ?>
      <div class="so-card p-3 mb-3">
        <div class="d-flex align-items-center justify-content-between flex-wrap gap-2">
          <div class="d-flex align-items-center gap-2">
            <div class="fw-semibold" style="font-size:1.05rem;">
              6) <?= h(UW_STEPS['signature_match']) ?>
            </div>
            <span class="badge <?= badge_class_for_step((string)($stRow['status'] ?? 'pending')) ?>">
              <?= h(label_for_step_status((string)($stRow['status'] ?? 'pending'))) ?>
            </span>
            <span class="badge bg-dark" style="border:1px solid rgba(255,255,255,.08);">
              Score: <?= (int)($stRow['score'] ?? 0) ?>
            </span>
          </div>

          <form method="post" action="<?= app_path('underwriting/actions/run_signature_match.php') ?>">
            <input type="hidden" name="_csrf" value="<?= h($csrf) ?>">
            <input type="hidden" name="case_id" value="<?= (int)$caseId ?>">
            <button class="btn btn-outline-light">
              <i class="fa-solid fa-signature me-2"></i>Compare Signatures (AI)
            </button>
          </form>
        </div>

        <div class="mt-3" style="opacity:.9;">
          This step compares the signature on the uploaded driver’s license to the signature on the order contract/invoice PDF (best-effort).
        </div>

        <?php if (!empty($sigDetails)): ?>
          <div class="mt-3 p-3 rounded" style="background:rgba(255,255,255,0.03); border:1px solid rgba(255,255,255,0.06);">
            <div class="fw-semibold">Signature Match Result</div>
            <div class="mt-2">
              <?php if (isset($sigDetails['match_confidence'])): ?>
                <div><strong>Match Confidence:</strong> <?= (int)$sigDetails['match_confidence'] ?>/100</div>
              <?php endif; ?>
              <?php if (!empty($sigDetails['verdict'])): ?>
                <div><strong>Verdict:</strong> <?= h((string)$sigDetails['verdict']) ?></div>
              <?php endif; ?>
              <?php if (!empty($sigDetails['reasons']) && is_array($sigDetails['reasons'])): ?>
                <div class="mt-2">
                  <strong>Reasons:</strong>
                  <ul class="mb-0">
                    <?php foreach ($sigDetails['reasons'] as $r): ?>
                      <li><?= h((string)$r) ?></li>
                    <?php endforeach; ?>
                  </ul>
                </div>
              <?php endif; ?>
            </div>
          </div>
        <?php endif; ?>
      </div>

      <!-- STEP: People/Address History (placeholder, no external API required) -->
      <?php $stRow = $stepsByKey['people_history'] ?? null; ?>
      <div class="so-card p-3 mb-3">
        <div class="d-flex align-items-center justify-content-between flex-wrap gap-2">
          <div class="d-flex align-items-center gap-2">
            <div class="fw-semibold" style="font-size:1.05rem;">
              7) <?= h(UW_STEPS['people_history']) ?>
            </div>
            <span class="badge <?= badge_class_for_step((string)($stRow['status'] ?? 'pending')) ?>">
              <?= h(label_for_step_status((string)($stRow['status'] ?? 'pending'))) ?>
            </span>
            <span class="badge bg-dark" style="border:1px solid rgba(255,255,255,.08);">
              Score: <?= (int)($stRow['score'] ?? 0) ?>
            </span>
          </div>

          <form method="post" action="<?= app_path('underwriting/actions/step_update.php') ?>" class="d-flex gap-2 align-items-center">
            <input type="hidden" name="_csrf" value="<?= h($csrf) ?>">
            <input type="hidden" name="case_id" value="<?= (int)$caseId ?>">
            <input type="hidden" name="step_key" value="people_history">
            <input type="hidden" name="status" value="done">
            <button class="btn btn-outline-light">
              <i class="fa-solid fa-check me-2"></i>Mark Complete
            </button>
          </form>
        </div>

        <div class="mt-3" style="opacity:.9;">
          Optional: review consistency (name/phone/address) across your internal history and any low-cost data sources you choose later.
        </div>
      </div>

    </div>

    <!-- NOTES -->
    <div class="col-12 col-xl-4">
      <div class="so-card p-3">
        <div class="d-flex align-items-center justify-content-between">
          <div class="fw-semibold" style="font-size:1.05rem;">Underwriting Notes</div>
        </div>

        <form method="post" action="<?= app_path('underwriting/actions/note_add.php') ?>" class="mt-3">
          <input type="hidden" name="_csrf" value="<?= h($csrf) ?>">
          <input type="hidden" name="case_id" value="<?= (int)$caseId ?>">
          <textarea class="form-control" name="note" rows="3" placeholder="Add a note..."></textarea>
          <div class="d-grid mt-2">
            <button class="btn btn-primary">
              <i class="fa-solid fa-plus me-2"></i>Add Note
            </button>
          </div>
        </form>

        <div class="mt-3" style="max-height: 520px; overflow:auto;">
          <?php if (empty($notes)): ?>
            <div class="text-muted">No notes yet.</div>
          <?php else: ?>
            <?php foreach ($notes as $n): ?>
              <div class="p-3 mb-2 rounded" style="background:rgba(255,255,255,0.03); border:1px solid rgba(255,255,255,0.06);">
                <div class="d-flex align-items-center justify-content-between">
                  <div class="fw-semibold" style="font-size:.95rem;">
                    <?= h((string)($n['user_display'] ?? ('User #' . (string)($n['user_id'] ?? '')))) ?>
                  </div>
                  <div class="text-muted" style="font-size:.85rem;">
                    <?= h((string)($n['created_at'] ?? '')) ?>
                  </div>
                </div>
                <div class="mt-2" style="white-space:pre-wrap;"><?= h((string)($n['note'] ?? '')) ?></div>
              </div>
            <?php endforeach; ?>
          <?php endif; ?>
        </div>

      </div>
    </div>
  </div>

</div>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
