<?php
// /app/underwriting/actions/run_softpull.php
declare(strict_types=1);

require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/scope.php';
require_once __DIR__ . '/../_uw.php';

require_company_context_if_rto();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  header('Location: ' . app_path('underwriting/index.php'));
  exit;
}

if (!csrf_validate((string)($_POST['_csrf'] ?? ''))) {
  http_response_code(403);
  echo 'CSRF validation failed.';
  exit;
}

$pdo = db();
$tenantId = uw_company_tid();
$orderId = (int)($_POST['order_id'] ?? 0);
if ($orderId <= 0) {
  header('Location: ' . app_path('underwriting/index.php'));
  exit;
}

$case = uw_ensure_case($pdo, $tenantId, $orderId);
$caseId = (int)$case['id'];
$settings = uw_get_settings($pdo, $tenantId);

// SoftPull requires SSN + DOB at minimum.
$ssn = preg_replace('/\D+/', '', (string)($_POST['ssn'] ?? ''));
$dob = trim((string)($_POST['dob'] ?? ''));
$nameFirst = trim((string)($_POST['first_name'] ?? ''));
$nameLast = trim((string)($_POST['last_name'] ?? ''));
$address = trim((string)($_POST['address'] ?? ''));
$city = trim((string)($_POST['city'] ?? ''));
$state = trim((string)($_POST['state'] ?? ''));
$zip = trim((string)($_POST['zip'] ?? ''));

if ($ssn === '' || $dob === '' || $nameFirst === '' || $nameLast === '' || $address === '' || $city === '' || $state === '' || $zip === '') {
  // Attempt best-effort prefill from order.
  $o = $pdo->prepare('SELECT customer_name, billing_street_line_one, billing_city, billing_state, billing_zip FROM orders WHERE tenant_id=:tid AND id=:id');
  $o->execute([':tid'=>$tenantId, ':id'=>$orderId]);
  $orow = $o->fetch(PDO::FETCH_ASSOC) ?: [];

  uw_set_step($pdo, $caseId, 'softpull', 'review', 15, [
    'error' => 'Missing required fields for soft pull. Provide SSN, DOB (MM/DD/YYYY), first/last name, and full address.',
    'prefill' => [
      'customer_name' => $orow['customer_name'] ?? null,
      'billing_street_line_one' => $orow['billing_street_line_one'] ?? null,
      'billing_city' => $orow['billing_city'] ?? null,
      'billing_state' => $orow['billing_state'] ?? null,
      'billing_zip' => $orow['billing_zip'] ?? null,
    ]
  ], uw_user_id());
  uw_recompute_case_risk($pdo, $caseId);
  header('Location: ' . app_path('underwriting/view.php?order_id=' . $orderId));
  exit;
}

$auth = uw_softpull_auth($settings);
if (!$auth['ok']) {
  uw_set_step($pdo, $caseId, 'softpull', 'review', 25, ['error'=>'SoftPull auth failed.', 'details'=>$auth], uw_user_id());
  uw_recompute_case_risk($pdo, $caseId);
  header('Location: ' . app_path('underwriting/view.php?order_id=' . $orderId));
  exit;
}

$token = (string)$auth['token'];

// standardInquiry per Postman docs
$inquiryFields = [
  'Pass' => '2',
  'Product' => 'CREDIT',
  'Bureau' => 'TU',
  'SSN' => $ssn,
  'DOB' => $dob,
  'SplitName' => '1',
  'NameFirst' => $nameFirst,
  'NameLast' => $nameLast,
  'Address' => $address,
  'City' => $city,
  'State' => $state,
  'Zip' => $zip,
  'Rbp_Letter' => '1',
  'Rbp_Form' => 'H3',
  'Rbp_Model' => 'TU/001NN',
  'Rbp_Loan_Maker' => 'ShedOffice',
  'Rbp_Output' => 'XML',
];

$inq = uw_softpull_standard_inquiry($settings, $token, $inquiryFields);
$details = [
  'auth' => ['ok'=>$auth['ok']],
  'inquiry_http_code' => $inq['code'] ?? null,
];

if (!$inq['ok']) {
  $details['error'] = $inq['error'] ?? '';
  $details['raw'] = substr((string)($inq['body'] ?? ''), 0, 3000);
  uw_set_step($pdo, $caseId, 'softpull', 'review', 30, $details, uw_user_id());
  uw_recompute_case_risk($pdo, $caseId);
  header('Location: ' . app_path('underwriting/view.php?order_id=' . $orderId));
  exit;
}

// Attempt to extract transid and rbp_token from XML response (best-effort)
$xmlText = (string)$inq['body'];
$details['inquiry_xml_snippet'] = substr($xmlText, 0, 2000);

$transid = '';
$rbpToken = '';
if (preg_match('/<transid[^>]*>([^<]+)<\/transid>/i', $xmlText, $m)) $transid = trim($m[1]);
if (preg_match('/<rbp_token[^>]*>([^<]+)<\/rbp_token>/i', $xmlText, $m)) $rbpToken = trim($m[1]);

$details['transid'] = $transid;
$details['rbp_token'] = $rbpToken;

if ($transid === '') {
  uw_set_step($pdo, $caseId, 'softpull', 'review', 30, array_merge($details, ['error'=>'Could not extract transid from inquiry response.']), uw_user_id());
  uw_recompute_case_risk($pdo, $caseId);
  header('Location: ' . app_path('underwriting/view.php?order_id=' . $orderId));
  exit;
}

$rev = uw_softpull_review_report($settings, $token, ['Pass'=>'109', 'Ref'=>$transid]);
$details['review_http_code'] = $rev['code'] ?? null;

if (!$rev['ok']) {
  $details['error'] = $rev['error'] ?? '';
  $details['raw_review'] = substr((string)($rev['body'] ?? ''), 0, 3000);
  uw_set_step($pdo, $caseId, 'softpull', 'review', 35, $details, uw_user_id());
  uw_recompute_case_risk($pdo, $caseId);
  header('Location: ' . app_path('underwriting/view.php?order_id=' . $orderId));
  exit;
}

$reportJson = json_decode((string)$rev['body'], true);
if (!is_array($reportJson)) $reportJson = ['raw' => (string)$rev['body']];

$score = uw_softpull_score($reportJson);
$details['score'] = $score;

// Store full report for audit
$pdo->prepare('INSERT INTO underwriting_softpull_reports (case_id, transid, rbp_token, inquiry_xml, report_json) VALUES (:cid,:t,:rbp,:ix,:rj)')
  ->execute([
    ':cid'=>$caseId,
    ':t'=>$transid,
    ':rbp'=>$rbpToken,
    ':ix'=>$xmlText,
    ':rj'=>uw_json_encode($reportJson),
  ]);

// Convert report risk score to step contribution (max 40)
$stepScore = (int)round(((int)$score['risk_score']) * 0.4);
$status = ($score['risk_score'] >= 75 ? 'fail' : ($score['risk_score'] >= 35 ? 'review' : 'pass'));

uw_set_step($pdo, $caseId, 'softpull', $status, $stepScore, $details, uw_user_id());
uw_recompute_case_risk($pdo, $caseId);

header('Location: ' . app_path('underwriting/view.php?order_id=' . $orderId));
exit;
