<?php
// /app/underwriting/actions/run_signature_match.php
declare(strict_types=1);

require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/scope.php';
require_once __DIR__ . '/../_uw.php';

require_company_context_if_rto();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  header('Location: ' . app_path('underwriting/index.php'));
  exit;
}

if (!csrf_validate((string)($_POST['_csrf'] ?? ''))) {
  http_response_code(403);
  echo 'CSRF validation failed.';
  exit;
}

$pdo = db();
$tenantId = uw_company_tid();
$orderId = (int)($_POST['order_id'] ?? 0);
if ($orderId <= 0) {
  header('Location: ' . app_path('underwriting/index.php'));
  exit;
}

$case = uw_ensure_case($pdo, $tenantId, $orderId);
$caseId = (int)$case['id'];
$settings = uw_get_settings($pdo, $tenantId);
$apiKey = trim((string)$settings['openai_api_key']);
if ($apiKey === '') {
  uw_set_step($pdo, $caseId, 'signature_match', 'review', 20, ['error'=>'OpenAI API key not configured.'], uw_user_id());
  uw_recompute_case_risk($pdo, $caseId);
  header('Location: ' . app_path('underwriting/view.php?order_id=' . $orderId));
  exit;
}

// Get ID front image
$docs = $pdo->prepare('SELECT doc_type, url FROM underwriting_documents WHERE case_id = :cid');
$docs->execute([':cid'=>$caseId]);
$map = [];
foreach ($docs->fetchAll(PDO::FETCH_ASSOC) as $d) { $map[$d['doc_type']] = $d['url']; }
$idFront = $map['id_front'] ?? '';

$stmt = $pdo->prepare('SELECT invoice_url, delivery_certificate_url FROM orders WHERE tenant_id=:tid AND id=:id');
$stmt->execute([':tid'=>$tenantId, ':id'=>$orderId]);
$o = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];
$pdfUrl = (string)($o['delivery_certificate_url'] ?: $o['invoice_url'] ?: '');

if ($idFront === '' || $pdfUrl === '') {
  uw_set_step($pdo, $caseId, 'signature_match', 'review', 25, [
    'error' => 'Missing inputs. Require ID front image and either delivery_certificate_url or invoice_url on the order.',
    'id_front' => $idFront,
    'pdf_url' => $pdfUrl,
  ], uw_user_id());
  uw_recompute_case_risk($pdo, $caseId);
  header('Location: ' . app_path('underwriting/view.php?order_id=' . $orderId));
  exit;
}

$tmpPdf = sys_get_temp_dir() . '/uw_' . $caseId . '_' . bin2hex(random_bytes(4)) . '.pdf';
$dl = uw_http_request($pdfUrl, 'GET', ['accept: application/pdf'], null, 60);
$details = ['pdf_url'=>$pdfUrl, 'download_http'=>$dl['code'] ?? null];
if (!$dl['ok'] || trim((string)$dl['body']) === '') {
  $details['error'] = 'Failed to download PDF.';
  $details['download_error'] = $dl['error'] ?? '';
  $details['download_body_snip'] = substr((string)$dl['body'], 0, 500);
  uw_set_step($pdo, $caseId, 'signature_match', 'review', 35, $details, uw_user_id());
  uw_recompute_case_risk($pdo, $caseId);
  header('Location: ' . app_path('underwriting/view.php?order_id=' . $orderId));
  exit;
}
file_put_contents($tmpPdf, (string)$dl['body']);

$up = uw_openai_upload_file($apiKey, $tmpPdf, 'assistants');
@unlink($tmpPdf);
$details['file_upload'] = ['ok'=>$up['ok'], 'code'=>$up['code'] ?? null];

if (!$up['ok']) {
  $details['error'] = 'OpenAI file upload failed.';
  $details['raw'] = substr((string)($up['raw'] ?? ''), 0, 1000);
  uw_set_step($pdo, $caseId, 'signature_match', 'review', 35, $details, uw_user_id());
  uw_recompute_case_risk($pdo, $caseId);
  header('Location: ' . app_path('underwriting/view.php?order_id=' . $orderId));
  exit;
}

$fileId = (string)($up['json']['id'] ?? '');
if ($fileId === '') {
  $details['error'] = 'OpenAI upload did not return file_id.';
  uw_set_step($pdo, $caseId, 'signature_match', 'review', 35, $details, uw_user_id());
  uw_recompute_case_risk($pdo, $caseId);
  header('Location: ' . app_path('underwriting/view.php?order_id=' . $orderId));
  exit;
}

$payload = [
  'model' => (string)$settings['openai_model'],
  'input' => [[
    'role' => 'user',
    'content' => [
      ['type'=>'input_text','text'=>"Compare the handwritten/visual signature on the provided driver license image with the signature found in the provided PDF (invoice/contract). Return JSON with found_signature_id, found_signature_pdf, match_confidence (0-100), verdict (pass|review|fail), and reasons. If no signature is visible in either document, say so."],
      ['type'=>'input_image','image_url'=>$idFront],
      ['type'=>'input_file','file_id'=>$fileId],
    ]
  ]],
  'response_format' => [
    'type' => 'json_schema',
    'json_schema' => [
      'name' => 'signature_match',
      'schema' => [
        'type' => 'object',
        'additionalProperties' => true,
        'properties' => [
          'found_signature_id' => ['type'=>'boolean'],
          'found_signature_pdf' => ['type'=>'boolean'],
          'match_confidence' => ['type'=>'integer','minimum'=>0,'maximum'=>100],
          'verdict' => ['type'=>'string','enum'=>['pass','review','fail']],
          'reasons' => ['type'=>'array','items'=>['type'=>'string']],
        ],
        'required' => ['found_signature_id','found_signature_pdf','match_confidence','verdict','reasons']
      ]
    ]
  ]
];

$r = uw_openai_responses($apiKey, $payload);
$details['openai_http_code'] = $r['code'] ?? null;

$status = 'review';
$score = 20;

if (!$r['ok']) {
  $details['error'] = $r['error'] ?? '';
  $details['raw'] = substr((string)($r['body'] ?? ''), 0, 2000);
  $status = 'review';
  $score = 35;
} else {
  $json = json_decode((string)$r['body'], true);
  $details['response'] = $json;
  $out = $json['output'][0]['content'][0]['parsed'] ?? null;
  if (!$out && isset($json['output_text'])) $out = json_decode((string)$json['output_text'], true);

  if (is_array($out)) {
    $details['result'] = $out;
    $conf = (int)($out['match_confidence'] ?? 50);
    $verdict = (string)($out['verdict'] ?? 'review');
    $status = $verdict;
    // Signature mismatch is strong fraud signal; max 25 points.
    $score = (int)round((100 - $conf) * 0.25);
    if ($verdict === 'fail') $score = max($score, 20);
    if ($verdict === 'pass') $score = min($score, 5);
  } else {
    $details['warning'] = 'Could not parse model output.';
    $status = 'review';
    $score = 25;
  }
}

uw_set_step($pdo, $caseId, 'signature_match', $status, $score, $details, uw_user_id());
uw_recompute_case_risk($pdo, $caseId);

header('Location: ' . app_path('underwriting/view.php?order_id=' . $orderId));
exit;
