<?php
// /app/underwriting/actions/run_public_records.php
declare(strict_types=1);

require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/scope.php';
require_once __DIR__ . '/../_uw.php';

require_company_context_if_rto();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  header('Location: ' . app_path('underwriting/index.php'));
  exit;
}

if (!csrf_validate((string)($_POST['_csrf'] ?? ''))) {
  http_response_code(403);
  echo 'CSRF validation failed.';
  exit;
}

$pdo = db();
$tenantId = uw_company_tid();
$orderId = (int)($_POST['order_id'] ?? 0);
if ($orderId <= 0) {
  header('Location: ' . app_path('underwriting/index.php'));
  exit;
}

$case = uw_ensure_case($pdo, $tenantId, $orderId);
$caseId = (int)$case['id'];
$settings = uw_get_settings($pdo, $tenantId);

$stmt = $pdo->prepare('SELECT customer_name, billing_city, billing_state FROM orders WHERE tenant_id=:tid AND id=:id');
$stmt->execute([':tid'=>$tenantId, ':id'=>$orderId]);
$o = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];

$name = trim((string)($o['customer_name'] ?? ''));
$city = trim((string)($o['billing_city'] ?? ''));
$state = trim((string)($o['billing_state'] ?? ''));

if ($name === '') {
  uw_set_step($pdo, $caseId, 'public_records', 'review', 10, ['error'=>'Customer name missing on order.'], uw_user_id());
  uw_recompute_case_risk($pdo, $caseId);
  header('Location: ' . app_path('underwriting/view.php?order_id=' . $orderId));
  exit;
}

$q = $name . ' ' . $city . ' ' . $state . ' arrest court case';
$search = uw_google_cse_search($settings, $q, 5);

$details = ['query'=>$q, 'search'=>$search];
$status = 'review';
$score = 10;

$apiKey = trim((string)$settings['openai_api_key']);
if ($apiKey !== '' && !empty($search['items'])) {
  $payload = [
    'model' => (string)$settings['openai_model'],
    'input' => [[
      'role' => 'user',
      'content' => [
        ['type'=>'input_text','text'=>"Summarize potential public-record risk signals for underwriting. Based ONLY on the snippets/links (may be inaccurate), return JSON with verdict(pass|review|fail), risk_score(0-100), reasons, sources_used (links), and a caution that results may be for different people with same name."],
        ['type'=>'input_text','text'=>uw_json_encode($search['items'])],
      ]
    ]],
    'response_format' => [
      'type'=>'json_schema',
      'json_schema'=>[
        'name'=>'public_records',
        'schema'=>[
          'type'=>'object',
          'additionalProperties'=>true,
          'properties'=>[
            'verdict'=>['type'=>'string','enum'=>['pass','review','fail']],
            'risk_score'=>['type'=>'integer','minimum'=>0,'maximum'=>100],
            'reasons'=>['type'=>'array','items'=>['type'=>'string']],
            'sources_used'=>['type'=>'array','items'=>['type'=>'string']],
            'caution'=>['type'=>'string']
          ],
          'required'=>['verdict','risk_score','reasons']
        ]
      ]
    ]
  ];

  $r = uw_openai_responses($apiKey, $payload);
  $details['openai_http_code'] = $r['code'] ?? null;
  if ($r['ok']) {
    $json = json_decode((string)$r['body'], true);
    $details['response'] = $json;
    $out = $json['output'][0]['content'][0]['parsed'] ?? null;
    if (!$out && isset($json['output_text'])) $out = json_decode((string)$json['output_text'], true);
    if (is_array($out)) {
      $details['result'] = $out;
      $risk = (int)($out['risk_score'] ?? 50);
      $verdict = (string)($out['verdict'] ?? 'review');
      $status = $verdict;
      $score = (int)round($risk * 0.20); // up to 20 pts
      if ($verdict==='pass') $score = min($score, 5);
      if ($verdict==='fail') $score = max($score, 15);
    }
  } else {
    $details['openai_error'] = $r['error'] ?? '';
    $details['openai_raw'] = substr((string)($r['body'] ?? ''), 0, 2000);
    $status = 'review';
    $score = 12;
  }
}

uw_set_step($pdo, $caseId, 'public_records', $status, $score, $details, uw_user_id());
uw_recompute_case_risk($pdo, $caseId);
header('Location: ' . app_path('underwriting/view.php?order_id=' . $orderId));
exit;
