<?php
// /app/underwriting/actions/run_id_ai.php
declare(strict_types=1);

require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/scope.php';
require_once __DIR__ . '/../_uw.php';

require_company_context_if_rto();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  header('Location: ' . app_path('underwriting/index.php'));
  exit;
}

if (!csrf_validate((string)($_POST['_csrf'] ?? ''))) {
  http_response_code(403);
  echo 'CSRF validation failed.';
  exit;
}

$pdo = db();
$tenantId = uw_company_tid();
$orderId = (int)($_POST['order_id'] ?? 0);
if ($orderId <= 0) {
  header('Location: ' . app_path('underwriting/index.php'));
  exit;
}
$case = uw_ensure_case($pdo, $tenantId, $orderId);
$caseId = (int)$case['id'];
$settings = uw_get_settings($pdo, $tenantId);

$apiKey = trim((string)$settings['openai_api_key']);
if ($apiKey === '') {
  uw_set_step($pdo, $caseId, 'id_ai', 'review', 25, ['error'=>'OpenAI API key not configured.'], uw_user_id());
  uw_recompute_case_risk($pdo, $caseId);
  header('Location: ' . app_path('underwriting/view.php?order_id=' . $orderId));
  exit;
}

$docs = $pdo->prepare('SELECT doc_type, url FROM underwriting_documents WHERE case_id = :cid');
$docs->execute([':cid'=>$caseId]);
$map = [];
foreach ($docs->fetchAll(PDO::FETCH_ASSOC) as $d) { $map[$d['doc_type']] = $d['url']; }

$frontUrl = $map['id_front'] ?? '';
if ($frontUrl === '') {
  uw_set_step($pdo, $caseId, 'id_ai', 'review', 25, ['error'=>'No ID front image uploaded.'], uw_user_id());
  uw_recompute_case_risk($pdo, $caseId);
  header('Location: ' . app_path('underwriting/view.php?order_id=' . $orderId));
  exit;
}
$backUrl = $map['id_back'] ?? '';

$payload = [
  'model' => (string)$settings['openai_model'],
  'input' => [
    [
      'role' => 'user',
      'content' => array_values(array_filter([
        ['type'=>'input_text','text'=>"Analyze this driver license image for authenticity indicators. Output JSON with fields: verdict(pass|review|fail), risk_score(0-100), reasons(array), extracted(name,dob,dl_number,address,exp,iss,state), and barcode_expected_back(bool). If only a front image is provided, emphasize limitations and request back for PDF417 cross-check."],
        ['type'=>'input_image','image_url'=>$frontUrl],
        $backUrl ? ['type'=>'input_image','image_url'=>$backUrl] : null,
      ]))
    ]
  ],
  'response_format' => [
    'type' => 'json_schema',
    'json_schema' => [
      'name' => 'id_verification',
      'schema' => [
        'type' => 'object',
        'additionalProperties' => true,
        'properties' => [
          'verdict' => ['type'=>'string','enum'=>['pass','review','fail']],
          'risk_score' => ['type'=>'integer','minimum'=>0,'maximum'=>100],
          'reasons' => ['type'=>'array','items'=>['type'=>'string']],
          'extracted' => ['type'=>'object'],
          'barcode_expected_back' => ['type'=>'boolean']
        ],
        'required' => ['verdict','risk_score','reasons']
      ]
    ]
  ]
];

$resp = uw_openai_responses($apiKey, $payload);
$details = ['openai_http_code'=>$resp['code'] ?? null];
$score = 25;
$status = 'review';

if (!$resp['ok']) {
  $details['error'] = $resp['error'] ?? '';
  $details['raw'] = substr((string)($resp['body'] ?? ''), 0, 2000);
  $status = 'review';
  $score = 35;
} else {
  $json = json_decode((string)$resp['body'], true);
  $details['response'] = $json;

  // Try to find parsed output:
  $out = $json['output'][0]['content'][0]['parsed'] ?? null;
  if (!$out && isset($json['output_text'])) {
    $out = json_decode((string)$json['output_text'], true);
  }

  if (is_array($out)) {
    $details['result'] = $out;
    $risk = (int)($out['risk_score'] ?? 50);
    $verdict = (string)($out['verdict'] ?? 'review');
    $status = $verdict;
    // Convert to step score contribution
    $score = (int)round($risk * 0.4); // max 40 points for ID
    if ($verdict === 'fail') $score = max($score, 35);
    if ($verdict === 'pass') $score = min($score, 8);
  } else {
    $status = 'review';
    $score = 30;
    $details['warning'] = 'Could not parse model output into JSON.';
  }
}

uw_set_step($pdo, $caseId, 'id_ai', $status, $score, $details, uw_user_id());
uw_recompute_case_risk($pdo, $caseId);

header('Location: ' . app_path('underwriting/view.php?order_id=' . $orderId));
exit;
