<?php
// /app/select_company.php
declare(strict_types=1);

require_once __DIR__ . '/includes/auth.php';
require_once __DIR__ . '/includes/acl.php';

require_login();
acl_bootstrap(db());

$pdo = db();
start_secure_session();

$tenantType = current_tenant_type();
$rtoTenantId = (int) current_tenant_id();

if ($tenantType !== 'rto') {
  // Company users don't need to select a company context
  header('Location: ' . app_path('dashboard.php'));
  exit;
}

$err = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $token = (string)($_POST['_csrf'] ?? '');
  $companyTenantId = (int)($_POST['company_tenant_id'] ?? 0);

  if (!csrf_validate($token)) {
    $err = 'Security check failed. Please refresh and try again.';
  } elseif ($companyTenantId < 1) {
    $err = 'Invalid company selection.';
  } else {
    // Validate link exists and is active
    $stmt = $pdo->prepare("
      SELECT t.id, t.name
      FROM rto_company_links l
      JOIN tenants t ON t.id = l.company_tenant_id
      WHERE l.rto_tenant_id = :rto
        AND l.company_tenant_id = :cid
        AND l.status = 'active'
        AND t.tenant_type = 'company'
      LIMIT 1
    ");
    $stmt->execute([
      ':rto' => $rtoTenantId,
      ':cid' => $companyTenantId
    ]);
    $company = $stmt->fetch();

    if (!$company) {
      $err = 'You are not authorized for that company.';
    } else {
      // Set active company context in session
      $_SESSION['auth']['active_company_tenant_id'] = (int)$company['id'];
      $_SESSION['auth']['active_company_name'] = (string)$company['name'];

      header('Location: ' . app_path('dashboard.php'));
      exit;
    }
  }
}

// Load all linked companies
$stmt = $pdo->prepare("
  SELECT t.id, t.name, t.status
  FROM rto_company_links l
  JOIN tenants t ON t.id = l.company_tenant_id
  WHERE l.rto_tenant_id = :rto
    AND l.status = 'active'
    AND t.tenant_type = 'company'
  ORDER BY t.name ASC
");
$stmt->execute([':rto' => $rtoTenantId]);
$companies = $stmt->fetchAll();

$pageTitle = 'Select Company';
$activeNav = 'dashboard';
require_once __DIR__ . '/includes/header.php';
?>
<div class="container-fluid">
  <div class="row g-3">
    <div class="col-12">
      <div class="so-page-title">Select a Company</div>
      <div class="so-page-sub">Choose which manufacturer you want to work in.</div>
    </div>

    <div class="col-12 col-xl-8">
      <div class="so-card">
        <div class="so-card-h">
          <h5 class="so-card-title"><i class="fa-solid fa-building me-2" style="color:#cf1f1f"></i>Linked Companies</h5>
        </div>
        <div class="so-card-body">

          <?php if ($err): ?>
            <div class="alert alert-danger"><?= h($err) ?></div>
          <?php endif; ?>

          <?php if (!$companies): ?>
            <div class="alert alert-warning mb-0">
              No companies are linked to this RTO account yet.
            </div>
          <?php else: ?>
            <div class="table-responsive">
              <table class="table so-table align-middle">
                <thead>
                  <tr>
                    <th>Company</th>
                    <th>Status</th>
                    <th class="text-end">Action</th>
                  </tr>
                </thead>
                <tbody>
                  <?php foreach ($companies as $c): ?>
                    <tr>
                      <td><strong><?= h((string)$c['name']) ?></strong></td>
                      <td><span class="so-pill gray"><?= h((string)$c['status']) ?></span></td>
                      <td class="text-end">
                        <form method="post" class="d-inline">
                          <input type="hidden" name="_csrf" value="<?= h(csrf_token()) ?>">
                          <input type="hidden" name="company_tenant_id" value="<?= (int)$c['id'] ?>">
                          <button class="btn btn-sm so-btn" type="submit">
                            Use Company <i class="fa-solid fa-arrow-right ms-1"></i>
                          </button>
                        </form>
                      </td>
                    </tr>
                  <?php endforeach; ?>
                </tbody>
              </table>
            </div>
          <?php endif; ?>

        </div>
      </div>
    </div>

    <div class="col-12 col-xl-4">
      <div class="so-card">
        <div class="so-card-h">
          <h5 class="so-card-title"><i class="fa-solid fa-circle-info me-2" style="color:#cf1f1f"></i>How this works</h5>
        </div>
        <div class="so-card-body">
          <div class="text-muted">
            RTO accounts can be linked to multiple manufacturers. Selecting a company sets your active context and restricts data access to only your records within that company.
          </div>
        </div>
      </div>
    </div>

  </div>
</div>
<?php require_once __DIR__ . '/includes/footer.php'; ?>
