<?php
// /app/repos/repos_list.php
declare(strict_types=1);

require_once __DIR__ . '/../includes/auth.php';
require_company_context_if_rto();

require_once __DIR__ . '/../includes/acl.php';
require_once __DIR__ . '/../includes/scope.php';

$pageTitle = 'Repos';
$activeNav = 'repos';

$pdo = db();
acl_bootstrap($pdo);

$user = current_user();

$tenantType = current_tenant_type(); // company|rto
$companyTenantId = company_context_id();

$isCompanyAdmin = user_has_role('company_admin');
$isManager      = user_has_role('manager');
$isDriver       = user_has_role('driver');

// Default drivers to "Assigned to me" unless explicitly set
$assignedToMe = isset($_GET['mine'])
  ? (int)($_GET['mine'] ?? 0)
  : ($isDriver ? 1 : 0);

$status = trim((string)($_GET['status'] ?? ''));
$q      = trim((string)($_GET['q'] ?? ''));

// Company-only filters
$driverId = (int)($_GET['driver_id'] ?? 0);

// Validate status
$validStatuses = ['new','assigned','en_route','picked_up','at_yard','closed','canceled'];
if ($status !== '' && !in_array($status, $validStatuses, true)) $status = '';

/**
 * Base WHERE from centralized scope (tenant + rto + driver assigned-only)
 * apply_repo_scope uses alias r.
 */
$where = [];
$params = [];
apply_repo_scope($where, $params);

// Manual filters
if ($status !== '') {
  $where[] = "r.status = :status";
  $params[':status'] = $status;
}

if ($q !== '') {
  $where[] = "(a.stock_number LIKE :q OR a.serial_number LIKE :q OR r.pickup_city LIKE :q OR r.pickup_state LIKE :q OR CAST(r.id AS CHAR) LIKE :q)";
  $params[':q'] = '%' . $q . '%';
}

/**
 * Assigned-to-me override (company only)
 */
if ($tenantType !== 'rto' && $assignedToMe === 1) {
  $where[] = "r.assigned_driver_user_id = :mine_driver";
  $params[':mine_driver'] = (int) current_user_id();
}

/**
 * Driver filter:
 * - company only
 * - only if user can view all repos and not forcing mine=1
 */
$canViewAllRepos = $isCompanyAdmin || $isManager || user_can('repos.view_all');
if ($tenantType !== 'rto' && $driverId > 0 && $assignedToMe !== 1 && $canViewAllRepos) {
  $where[] = "r.assigned_driver_user_id = :driver_id";
  $params[':driver_id'] = $driverId;
}

$whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

/**
 * Driver dropdown (role-filtered) for company users that can view all
 * IMPORTANT: unique placeholders (:tid1, :tid2)
 */
$drivers = [];
if ($tenantType !== 'rto' && $canViewAllRepos) {
  $stmt = $pdo->prepare("
    SELECT u.id, CONCAT(COALESCE(u.first_name,''), ' ', COALESCE(u.last_name,'')) AS name
    FROM users u
    JOIN user_roles ur ON ur.user_id = u.id
    JOIN roles r ON r.id = ur.role_id
    WHERE u.tenant_id = :tid1
      AND u.status = 'active'
      AND r.tenant_id = :tid2
      AND r.name = 'driver'
    ORDER BY name ASC
    LIMIT 300
  ");
  $stmt->execute([
    ':tid1' => $companyTenantId,
    ':tid2' => $companyTenantId,
  ]);
  $drivers = $stmt->fetchAll();
}

function pill_class_repo(string $status): string {
  return match ($status) {
    'new' => 'blue',
    'assigned', 'en_route' => 'orange',
    'picked_up', 'at_yard' => 'red',
    'closed', 'canceled' => 'gray',
    default => 'gray'
  };
}

$sql = "
  SELECT
    r.id,
    r.status,
    r.priority,
    r.created_at,
    r.updated_at,
    r.picked_up_at,
    r.pickup_city,
    r.pickup_state,
    r.assigned_driver_user_id,
    a.stock_number,
    a.asset_type,
    a.length_ft,
    a.width_ft,
    a.height_ft,
    t_rto.name AS rto_name,
    CONCAT(COALESCE(u.first_name,''), ' ', COALESCE(u.last_name,'')) AS driver_name
  FROM repos r
  JOIN assets a ON a.id = r.asset_id
  LEFT JOIN tenants t_rto ON t_rto.id = r.rto_tenant_id
  LEFT JOIN users u ON u.id = r.assigned_driver_user_id
  $whereSql
  ORDER BY
    FIELD(r.status,'new','assigned','en_route','picked_up','at_yard','closed','canceled'),
    r.updated_at DESC
  LIMIT 250
";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$rows = $stmt->fetchAll();

require_once __DIR__ . '/../includes/header.php';
?>
<div class="container-fluid">

  <div class="d-flex flex-wrap justify-content-between align-items-end gap-2 mb-3">
    <div>
      <div class="so-page-title">Repos</div>
      <div class="so-page-sub">
        <?= h($tenantType === 'rto' ? ('RTO View: ' . ($user['active_company_name'] ?? '')) : 'Company View') ?>
      </div>
    </div>
    <div class="d-flex gap-2">
      <a class="btn so-btn" href="<?= app_path('repos/repos_list.php') ?>"><i class="fa-solid fa-rotate me-2"></i>Refresh</a>
      <a class="btn btn-outline-dark" href="<?= app_path('repos/repo_create.php') ?>" style="border-radius:12px;"><i class="fa-solid fa-plus me-2"></i>New Repo</a>
    </div>
  </div>

  <div class="so-card mb-3">
    <div class="so-card-body">
      <form class="row g-2 align-items-end" method="get">

        <div class="col-12 col-md-3">
          <label class="form-label fw-bold">Status</label>
          <select name="status" class="form-select" style="border-radius:12px;">
            <option value="">All</option>
            <?php foreach ($validStatuses as $s): ?>
              <option value="<?= h($s) ?>" <?= $status===$s?'selected':'' ?>>
                <?= h(strtoupper(str_replace('_',' ', $s))) ?>
              </option>
            <?php endforeach; ?>
          </select>
        </div>

        <div class="col-12 col-md-4">
          <label class="form-label fw-bold">Search</label>
          <input type="text" name="q" value="<?= h($q) ?>" class="form-control" style="border-radius:12px;" placeholder="Stock #, City, State, Repo ID...">
        </div>

        <div class="col-12 col-md-3">
          <label class="form-label fw-bold">Driver</label>
          <select name="driver_id" class="form-select" style="border-radius:12px;" <?= ($tenantType==='rto' || !$canViewAllRepos) ? 'disabled' : '' ?>>
            <option value="0">All</option>
            <?php foreach ($drivers as $d): ?>
              <?php $did = (int)$d['id']; ?>
              <option value="<?= $did ?>" <?= ($driverId===$did)?'selected':'' ?>>
                <?= h(trim((string)$d['name']) ?: ('User #'.$did)) ?>
              </option>
            <?php endforeach; ?>
          </select>
          <?php if ($tenantType==='rto'): ?>
            <div class="text-muted small mt-1">Driver filter is available for company users only.</div>
          <?php elseif (!$canViewAllRepos): ?>
            <div class="text-muted small mt-1">Driver filter requires view-all access.</div>
          <?php endif; ?>
        </div>

        <div class="col-12 col-md-2">
          <div class="form-check" style="padding-top: 10px;">
            <input class="form-check-input" type="checkbox" value="1" id="mine" name="mine" <?= ($assignedToMe===1)?'checked':'' ?>>
            <label class="form-check-label fw-bold" for="mine">
              Assigned to me
            </label>
          </div>
        </div>

        <div class="col-12 d-flex gap-2 mt-2">
          <button class="btn btn-dark" style="border-radius:12px;" type="submit">
            <i class="fa-solid fa-filter me-2"></i>Apply Filters
          </button>
          <a class="btn btn-outline-dark" style="border-radius:12px;" href="<?= app_path('repos/repos_list.php') ?>">
            Clear
          </a>
        </div>

      </form>
    </div>
  </div>

  <div class="so-card">
    <div class="so-card-h">
      <h5 class="so-card-title">
        <i class="fa-solid fa-truck-ramp-box me-2" style="color:#cf1f1f"></i>
        Repo Queue
      </h5>
      <div class="text-muted small">Showing up to 250 results</div>
    </div>

    <div class="so-card-body">
      <div class="table-responsive">
        <table class="table so-table align-middle">
          <thead>
            <tr>
              <th>Repo</th>
              <th>RTO Company</th>
              <th>Asset</th>
              <th>Pickup</th>
              <th>Driver</th>
              <th>Status</th>
              <th class="text-end">Updated</th>
            </tr>
          </thead>
          <tbody>
          <?php if (!$rows): ?>
            <tr>
              <td colspan="7" class="text-muted">No repos found for the selected filters.</td>
            </tr>
          <?php else: ?>
            <?php foreach ($rows as $r): ?>
              <?php
                $assetLabel = trim(($r['width_ft'] ? $r['width_ft'].'x' : '') . ($r['length_ft'] ?? ''));
                $assetLabel = $assetLabel ?: strtoupper((string)$r['asset_type']);
                $driverName = trim((string)$r['driver_name']);
                $updated = (string)($r['updated_at'] ?: $r['created_at']);
              ?>
              <tr style="cursor:pointer;" onclick="window.location.href='<?= app_path('repos/repo_view.php?id='.(int)$r['id']) ?>'">
                <td>
                  <strong>#<?= (int)$r['id'] ?></strong>
                  <div class="text-muted small"><?= h((string)$r['priority']) ?> priority</div>
                </td>
                <td><strong><?= h($r['rto_name'] ?? '—') ?></strong></td>
                <td>
                  <strong><?= h((string)$r['stock_number']) ?></strong>
                  <div class="text-muted small"><?= h($assetLabel) ?></div>
                </td>
                <td>
                  <strong><?= h($r['pickup_city'] ?? '—') ?></strong>
                  <div class="text-muted small"><?= h($r['pickup_state'] ?? '') ?></div>
                </td>
                <td><?= h($driverName ?: '—') ?></td>
                <td>
                  <span class="so-pill <?= h(pill_class_repo((string)$r['status'])) ?>">
                    <?= h(strtoupper(str_replace('_',' ', (string)$r['status']))) ?>
                  </span>
                </td>
                <td class="text-end text-muted small">
                  <?= h(date('M j, Y g:i A', strtotime($updated))) ?>
                </td>
              </tr>
            <?php endforeach; ?>
          <?php endif; ?>
          </tbody>
        </table>
      </div>
    </div>
  </div>

</div>
<?php require_once __DIR__ . '/../includes/footer.php'; ?>
