<?php
// /app/repos/repo_view.php
declare(strict_types=1);

require_once __DIR__ . '/../includes/auth.php';
require_company_context_if_rto();

require_once __DIR__ . '/../includes/acl.php';
require_once __DIR__ . '/../includes/scope.php';

$pageTitle = 'Repo Details';
$activeNav = 'repos';

$pdo = db();
acl_bootstrap($pdo);

$user = current_user();

$tenantType = current_tenant_type(); // company|rto
$companyTenantId = company_context_id();
$rtoTenantId = (int) current_tenant_id();

$isCompanyAdmin = user_has_role('company_admin');
$isManager = user_has_role('manager');
$isDriver = user_has_role('driver');

function fetch_company_policy(PDO $pdo, int $companyTid): array {
  $driverCanOverride = 0;
  $driverCanCustom = 0;

  $stmt = $pdo->prepare("
    SELECT driver_can_override_repo_invoice, driver_can_submit_custom_invoice
    FROM tenant_settings
    WHERE tenant_id = :tid1
    LIMIT 1
  ");
  $stmt->execute([':tid1' => $companyTid]);
  if ($row = $stmt->fetch()) {
    $driverCanOverride = (int)$row['driver_can_override_repo_invoice'];
    $driverCanCustom = (int)$row['driver_can_submit_custom_invoice'];
  }

  return [$driverCanOverride, $driverCanCustom];
}

function load_addon_types(PDO $pdo, int $companyTid, ?int $repoRtoTenantId): array {
  $params = [':tid1' => $companyTid];
  $where = "tenant_id = :tid1 AND is_active = 1 AND (applies_to='company')";

  if ($repoRtoTenantId) {
    $where = "tenant_id = :tid1 AND is_active = 1 AND (
      (applies_to='company')
      OR (applies_to='rto' AND rto_tenant_id = :rto1)
    )";
    $params[':rto1'] = $repoRtoTenantId;
  }

  $stmt = $pdo->prepare("
    SELECT id, name, description, default_amount, max_amount, driver_can_add, driver_can_override_amount, applies_to, rto_tenant_id
    FROM repo_addon_fee_types
    WHERE $where
    ORDER BY
      (applies_to='rto') DESC,
      name ASC
  ");
  $stmt->execute($params);
  $rows = $stmt->fetchAll();

  $byName = [];
  foreach ($rows as $r) {
    $nm = (string)$r['name'];
    if (!isset($byName[$nm])) $byName[$nm] = $r;
  }

  return array_values($byName);
}

$repoId = (int)($_GET['id'] ?? 0);
if ($repoId < 1) {
  header('Location: ' . app_path('repos/repos_list.php'));
  exit;
}

$where = "r.tenant_id = :tid1 AND r.id = :rid1";
$params = [':tid1' => $companyTenantId, ':rid1' => $repoId];

if ($tenantType === 'rto') {
  $where .= " AND r.rto_tenant_id = :rto1";
  $params[':rto1'] = $rtoTenantId;
}

$sql = "
  SELECT
    r.*,
    a.stock_number, a.asset_type, a.length_ft, a.width_ft, a.height_ft, a.serial_number,
    t_rto.name AS rto_name,
    CONCAT(COALESCE(u.first_name,''), ' ', COALESCE(u.last_name,'')) AS driver_name
  FROM repos r
  JOIN assets a ON a.id = r.asset_id
  LEFT JOIN tenants t_rto ON t_rto.id = r.rto_tenant_id
  LEFT JOIN users u ON u.id = r.assigned_driver_user_id
  WHERE $where
  LIMIT 1
";
$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$repo = $stmt->fetch();

if (!$repo) {
  header('Location: ' . app_path('repos/repos_list.php'));
  exit;
}

// Drivers list (company users only, role-filtered)
$drivers = [];
if ($tenantType !== 'rto') {
  $stmt = $pdo->prepare("
    SELECT u.id, CONCAT(COALESCE(u.first_name,''), ' ', COALESCE(u.last_name,'')) AS name
    FROM users u
    JOIN user_roles ur ON ur.user_id = u.id
    JOIN roles r ON r.id = ur.role_id
    WHERE u.tenant_id = :tid1
      AND u.status = 'active'
      AND r.tenant_id = :tid2
      AND r.name = 'driver'
    ORDER BY name ASC
    LIMIT 300
  ");
  $stmt->execute([':tid1' => $companyTenantId, ':tid2' => $companyTenantId]);
  $drivers = $stmt->fetchAll();
}

// Pickup record
$stmt = $pdo->prepare("SELECT * FROM repo_pickups WHERE tenant_id=:tid1 AND repo_id=:rid1 LIMIT 1");
$stmt->execute([':tid1' => $companyTenantId, ':rid1' => $repoId]);
$pickup = $stmt->fetch();

// Dropoff record + location
$stmt = $pdo->prepare("
  SELECT d.dropoff_location_id, l.name AS location_name
  FROM repo_dropoffs d
  JOIN locations l ON l.id = d.dropoff_location_id
  WHERE d.tenant_id = :tid1 AND d.repo_id = :rid1
  LIMIT 1
");
$stmt->execute([':tid1' => $companyTenantId, ':rid1' => $repoId]);
$dropoff = $stmt->fetch();

// Media
$stmt = $pdo->prepare("
  SELECT media_type, public_url, created_at
  FROM repo_media
  WHERE tenant_id=:tid1 AND repo_id=:rid1
  ORDER BY created_at DESC
  LIMIT 100
");
$stmt->execute([':tid1' => $companyTenantId, ':rid1' => $repoId]);
$media = $stmt->fetchAll();

// Invoice
$stmt = $pdo->prepare("SELECT * FROM repo_invoices WHERE tenant_id=:tid1 AND repo_id=:rid1 LIMIT 1");
$stmt->execute([':tid1' => $companyTenantId, ':rid1' => $repoId]);
$invoice = $stmt->fetch();

$invoiceItems = [];
if ($invoice) {
  $stmt = $pdo->prepare("
    SELECT item_type, description, quantity, unit_price, amount
    FROM repo_invoice_items
    WHERE tenant_id=:tid1 AND repo_invoice_id=:iid1
    ORDER BY id ASC
  ");
  $stmt->execute([':tid1' => $companyTenantId, ':iid1' => (int)$invoice['id']]);
  $invoiceItems = $stmt->fetchAll();
}

// Events
$stmt = $pdo->prepare("
  SELECT event_type, meta_json, created_at
  FROM repo_events
  WHERE tenant_id=:tid1 AND repo_id=:rid1
  ORDER BY created_at DESC
  LIMIT 50
");
$stmt->execute([':tid1' => $companyTenantId, ':rid1' => $repoId]);
$events = $stmt->fetchAll();

// Locations for dropoff selector
$stmt = $pdo->prepare("SELECT id, name FROM locations WHERE tenant_id=:tid1 AND status='active' ORDER BY name ASC LIMIT 500");
$stmt->execute([':tid1' => $companyTenantId]);
$locations = $stmt->fetchAll();

// Policy + add-ons for modal
[$driverCanOverrideInvoice, $driverCanCustom] = fetch_company_policy($pdo, $companyTenantId);

// RTO link override
if (!empty($repo['rto_tenant_id'])) {
  $stmt = $pdo->prepare("
    SELECT id FROM rto_company_links
    WHERE rto_tenant_id=:rto1 AND company_tenant_id=:co1 AND status='active'
    LIMIT 1
  ");
  $stmt->execute([':rto1' => (int)$repo['rto_tenant_id'], ':co1' => $companyTenantId]);
  $linkId = (int)($stmt->fetchColumn() ?: 0);

  if ($linkId) {
    $stmt = $pdo->prepare("
      SELECT driver_can_override_repo_invoice, driver_can_submit_custom_invoice
      FROM rto_company_policy_settings
      WHERE rto_company_link_id = :lid1
      LIMIT 1
    ");
    $stmt->execute([':lid1' => $linkId]);
    if ($ov = $stmt->fetch()) {
      if ($ov['driver_can_override_repo_invoice'] !== null) $driverCanOverrideInvoice = (int)$ov['driver_can_override_repo_invoice'];
      if ($ov['driver_can_submit_custom_invoice'] !== null) $driverCanCustom = (int)$ov['driver_can_submit_custom_invoice'];
    }
  }
}

$showOverrideFields = ($isCompanyAdmin || $isManager || ($isDriver && $driverCanOverrideInvoice));

$repoRtoTid = !empty($repo['rto_tenant_id']) ? (int)$repo['rto_tenant_id'] : null;
$addonTypes = load_addon_types($pdo, $companyTenantId, $repoRtoTid);

// Driver sees only driver-addable
if ($isDriver && !$isCompanyAdmin && !$isManager) {
  $addonTypes = array_values(array_filter($addonTypes, fn($t) => (int)$t['driver_can_add'] === 1));
}

$flash = '';
if (!empty($_GET['ok'])) $flash = 'Saved successfully.';

require_once __DIR__ . '/../includes/header.php';
?>
<div class="container-fluid">

  <div class="d-flex flex-wrap justify-content-between align-items-end gap-2 mb-3">
    <div>
      <div class="so-page-title">Repo #<?= (int)$repo['id'] ?></div>
      <div class="so-page-sub">
        Asset: <strong><?= h((string)$repo['stock_number']) ?></strong>
        <?php if (!empty($repo['rto_name'])): ?>
          · RTO: <strong><?= h((string)$repo['rto_name']) ?></strong>
        <?php endif; ?>
      </div>
    </div>
    <div class="d-flex gap-2">
      <a class="btn so-btn" href="<?= app_path('repos/repos_list.php') ?>"><i class="fa-solid fa-arrow-left me-2"></i>Back</a>

      <button class="btn btn-dark" style="border-radius:12px;" type="button" data-bs-toggle="modal" data-bs-target="#pickupModal">
        <i class="fa-solid fa-camera me-2"></i>Pickup Package
      </button>
    </div>
  </div>

  <?php if ($flash): ?>
    <div class="alert alert-success"><?= h($flash) ?></div>
  <?php endif; ?>

  <div class="row g-3">
    <div class="col-12 col-xl-7">
      <div class="so-card">
        <div class="so-card-h">
          <h5 class="so-card-title"><i class="fa-solid fa-circle-info me-2" style="color:#cf1f1f"></i>Repo Details</h5>
        </div>
        <div class="so-card-body">
          <div class="row g-3">
            <div class="col-12 col-md-6">
              <div class="text-muted small">Status</div>
              <div class="fw-bold"><?= h(strtoupper(str_replace('_',' ', (string)$repo['status']))) ?></div>
            </div>
            <div class="col-12 col-md-6">
              <div class="text-muted small">Priority</div>
              <div class="fw-bold"><?= h((string)$repo['priority']) ?></div>
            </div>
            <div class="col-12 col-md-6">
              <div class="text-muted small">Assigned Driver</div>
              <div class="fw-bold"><?= h(trim((string)$repo['driver_name']) ?: '—') ?></div>
            </div>
            <div class="col-12 col-md-6">
              <div class="text-muted small">Picked Up</div>
              <div class="fw-bold"><?= h($repo['picked_up_at'] ? date('M j, Y g:i A', strtotime((string)$repo['picked_up_at'])) : '—') ?></div>
            </div>

            <div class="col-12">
              <div class="text-muted small">Pickup Address</div>
              <div class="fw-bold">
                <?= h((string)($repo['pickup_address1'] ?? '')) ?>
                <?= !empty($repo['pickup_address2']) ? (' ' . h((string)$repo['pickup_address2'])) : '' ?>
                <?= !empty($repo['pickup_city']) ? (', ' . h((string)$repo['pickup_city'])) : '' ?>
                <?= !empty($repo['pickup_state']) ? (' ' . h((string)$repo['pickup_state'])) : '' ?>
                <?= !empty($repo['pickup_zip']) ? (' ' . h((string)$repo['pickup_zip'])) : '' ?>
              </div>
            </div>

            <div class="col-12">
              <div class="text-muted small">Notes</div>
              <div><?= nl2br(h((string)($repo['notes'] ?? '—'))) ?></div>
            </div>
          </div>

          <?php if ($tenantType !== 'rto'): ?>
            <hr>
            <div class="row g-2 align-items-end">
              <div class="col-12">
                <label class="form-label fw-bold">Assign Driver</label>
                <form method="post" action="<?= app_path('repos/repo_assign_driver.php') ?>" class="d-flex gap-2 flex-wrap">
                  <input type="hidden" name="_csrf" value="<?= h(csrf_token()) ?>">
                  <input type="hidden" name="repo_id" value="<?= (int)$repo['id'] ?>">

                  <select name="driver_user_id" class="form-select" style="border-radius:12px; min-width: 260px;" required>
                    <option value="0">— Unassigned —</option>
                    <?php foreach ($drivers as $d): ?>
                      <?php $did = (int)$d['id']; ?>
                      <option value="<?= $did ?>" <?= ((int)$repo['assigned_driver_user_id'] === $did) ? 'selected' : '' ?>>
                        <?= h(trim((string)$d['name']) ?: ('User #'.$did)) ?>
                      </option>
                    <?php endforeach; ?>
                  </select>

                  <button class="btn btn-dark" style="border-radius:12px;" type="submit">
                    <i class="fa-solid fa-user-check me-2"></i>Save Driver
                  </button>
                </form>
                <div class="text-muted small mt-1">Assignment updates are logged in the timeline.</div>
              </div>
            </div>
          <?php else: ?>
            <hr>
            <div class="text-muted small">Driver assignment is managed by the manufacturer.</div>
          <?php endif; ?>

        </div>
      </div>

      <div class="so-card mt-3">
        <div class="so-card-h">
          <h5 class="so-card-title"><i class="fa-solid fa-images me-2" style="color:#cf1f1f"></i>Photos</h5>
        </div>
        <div class="so-card-body">
          <?php if (!$media): ?>
            <div class="text-muted">No photos uploaded yet.</div>
          <?php else: ?>
            <div class="row g-2">
              <?php foreach ($media as $m): ?>
                <div class="col-6 col-md-4 col-xl-3">
                  <a href="<?= h((string)$m['public_url']) ?>" target="_blank" style="text-decoration:none;">
                    <div class="so-card" style="box-shadow:none;">
                      <div style="border-radius:14px; overflow:hidden; border:1px solid rgba(18,20,24,0.12);">
                        <img src="<?= h((string)$m['public_url']) ?>" alt="" style="width:100%; height:150px; object-fit:cover;">
                      </div>
                      <div class="text-muted small mt-2"><?= h(str_replace('_',' ', (string)$m['media_type'])) ?></div>
                    </div>
                  </a>
                </div>
              <?php endforeach; ?>
            </div>
          <?php endif; ?>
        </div>
      </div>
    </div>

    <div class="col-12 col-xl-5">
      <div class="so-card">
        <div class="so-card-h">
          <h5 class="so-card-title"><i class="fa-solid fa-truck-fast me-2" style="color:#cf1f1f"></i>Pickup Summary</h5>
        </div>
        <div class="so-card-body">
          <?php if (!$pickup): ?>
            <div class="text-muted">Pickup package not submitted yet.</div>
          <?php else: ?>
            <div class="mb-2"><span class="text-muted small">Picked Up At</span><div class="fw-bold"><?= h(date('M j, Y g:i A', strtotime((string)$pickup['picked_up_at']))) ?></div></div>
            <div class="mb-2"><span class="text-muted small">Miles</span><div class="fw-bold"><?= h((string)($pickup['miles_pickup_to_dropoff'] ?? '—')) ?></div></div>
            <div class="mb-2"><span class="text-muted small">Repairs Needed</span><div class="fw-bold"><?= ((int)$pickup['repairs_needed'] === 1) ? 'Yes' : 'No' ?></div></div>
          <?php endif; ?>
          <hr>
          <div class="mb-2"><span class="text-muted small">Drop-off Location</span><div class="fw-bold"><?= h($dropoff['location_name'] ?? '—') ?></div></div>
        </div>
      </div>

      <div class="so-card mt-3">
        <div class="so-card-h">
          <h5 class="so-card-title"><i class="fa-solid fa-receipt me-2" style="color:#cf1f1f"></i>Repo Invoice</h5>
        </div>
        <div class="so-card-body">
          <?php if (!$invoice): ?>
            <div class="text-muted">No invoice generated yet.</div>
          <?php else: ?>
            <div class="d-flex justify-content-between">
              <div>
                <div class="text-muted small">Status</div>
                <div class="fw-bold"><?= h((string)$invoice['status']) ?></div>
              </div>
              <div class="text-end">
                <div class="text-muted small">Total</div>
                <div class="fw-bold">$<?= number_format((float)$invoice['total'], 2) ?></div>
              </div>
            </div>
            <div class="text-muted small mt-2">Billed To: <strong><?= h((string)$invoice['billed_to']) ?></strong></div>

            <?php if ((int)$invoice['overridden'] === 1): ?>
              <div class="alert alert-warning mt-3 mb-0">
                Override requested/applied.<br>
                <strong>Reason:</strong> <?= h((string)$invoice['override_reason']) ?>
              </div>
            <?php endif; ?>

            <?php if ($invoiceItems): ?>
              <hr>
              <div class="table-responsive">
                <table class="table so-table align-middle">
                  <thead>
                    <tr>
                      <th>Description</th>
                      <th class="text-end">Amount</th>
                    </tr>
                  </thead>
                  <tbody>
                    <?php foreach ($invoiceItems as $it): ?>
                      <tr>
                        <td><?= h((string)$it['description']) ?></td>
                        <td class="text-end">$<?= number_format((float)$it['amount'], 2) ?></td>
                      </tr>
                    <?php endforeach; ?>
                  </tbody>
                </table>
              </div>
            <?php endif; ?>
          <?php endif; ?>
        </div>
      </div>

      <?php if ($tenantType !== 'rto' && ($isCompanyAdmin || $isManager) && $invoice && (string)$invoice['status'] === 'submitted'): ?>
        <div class="so-card mt-3">
          <div class="so-card-h">
            <h5 class="so-card-title"><i class="fa-solid fa-circle-check me-2" style="color:#cf1f1f"></i>Invoice Approval Required</h5>
          </div>
          <div class="so-card-body">
            <div class="alert alert-warning">
              This invoice was submitted by a driver with an override and requires approval.
            </div>

            <form method="post" action="<?= app_path('repos/invoice_approve.php') ?>">
              <input type="hidden" name="_csrf" value="<?= h(csrf_token()) ?>">
              <input type="hidden" name="repo_id" value="<?= (int)$repoId ?>">
              <input type="hidden" name="invoice_id" value="<?= (int)$invoice['id'] ?>">

              <label class="form-label fw-bold">Approval Reason (required)</label>
              <input type="text" name="approval_reason" class="form-control" style="border-radius:12px;" maxlength="500" required placeholder="Why is this override approved?">

              <button class="btn btn-dark mt-3" style="border-radius:12px;" type="submit">
                <i class="fa-solid fa-check me-2"></i>Approve Invoice
              </button>
            </form>
          </div>
        </div>
      <?php endif; ?>

      <div class="so-card mt-3">
        <div class="so-card-h">
          <h5 class="so-card-title"><i class="fa-solid fa-clock-rotate-left me-2" style="color:#cf1f1f"></i>Timeline</h5>
        </div>
        <div class="so-card-body">
          <?php if (!$events): ?>
            <div class="text-muted">No events yet.</div>
          <?php else: ?>
            <div class="list-group">
              <?php foreach ($events as $e): ?>
                <div class="list-group-item" style="border-radius:12px; margin-bottom:10px; border:1px solid rgba(18,20,24,0.10);">
                  <div class="d-flex justify-content-between">
                    <div class="fw-bold"><?= h((string)$e['event_type']) ?></div>
                    <div class="text-muted small"><?= h(date('M j, Y g:i A', strtotime((string)$e['created_at']))) ?></div>
                  </div>
                  <?php if (!empty($e['meta_json'])): ?>
                    <div class="text-muted small mt-1"><?= h((string)$e['meta_json']) ?></div>
                  <?php endif; ?>
                </div>
              <?php endforeach; ?>
            </div>
          <?php endif; ?>
        </div>
      </div>

    </div>
  </div>

</div>

<!-- PICKUP MODAL (unchanged from add-on version; calls repo_confirm_pickup.php) -->
<div class="modal fade" id="pickupModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-xl modal-dialog-centered">
    <div class="modal-content" style="border-radius:16px; overflow:hidden;">
      <div class="modal-header" style="background:#111318; color:#fff;">
        <h5 class="modal-title">
          <i class="fa-solid fa-camera me-2" style="color:#cf1f1f"></i>Pickup Package
        </h5>
        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>

      <form method="post" action="<?= app_path('repos/repo_confirm_pickup.php') ?>" enctype="multipart/form-data">
        <div class="modal-body">
          <input type="hidden" name="_csrf" value="<?= h(csrf_token()) ?>">
          <input type="hidden" name="repo_id" value="<?= (int)$repoId ?>">

          <div class="row g-3">
            <div class="col-12 col-md-6">
              <label class="form-label fw-bold">Drop-off Location (required)</label>
              <select name="dropoff_location_id" class="form-select" style="border-radius:12px;" required>
                <option value="">— Select location —</option>
                <?php foreach ($locations as $l): ?>
                  <option value="<?= (int)$l['id'] ?>" <?= (!empty($dropoff['dropoff_location_id']) && (int)$dropoff['dropoff_location_id']===(int)$l['id']) ? 'selected' : '' ?>>
                    <?= h((string)$l['name']) ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>

            <div class="col-12 col-md-6">
              <label class="form-label fw-bold">Miles Pickup → Drop-off</label>
              <input type="number" step="0.01" min="0" name="miles_pickup_to_dropoff" class="form-control" style="border-radius:12px;" value="<?= h((string)($pickup['miles_pickup_to_dropoff'] ?? '')) ?>" placeholder="e.g. 72.5">
            </div>

            <div class="col-12 col-md-6">
              <label class="form-label fw-bold">Building Photo (required)</label>
              <input type="file" name="building_photo" class="form-control" style="border-radius:12px;" accept="image/*" required>
            </div>

            <div class="col-12 col-md-6">
              <label class="form-label fw-bold">Serial Number Photo (required)</label>
              <input type="file" name="serial_photo" class="form-control" style="border-radius:12px;" accept="image/*" required>
            </div>

            <div class="col-12">
              <label class="form-label fw-bold">Corner Photos (optional)</label>
              <input type="file" name="corner_photos[]" class="form-control" style="border-radius:12px;" accept="image/*" multiple>
            </div>

            <div class="col-12 col-md-3">
              <label class="form-label fw-bold">Repairs Needed?</label>
              <select name="repairs_needed" class="form-select" style="border-radius:12px;">
                <option value="0">No</option>
                <option value="1">Yes</option>
              </select>
            </div>

            <div class="col-12 col-md-9">
              <label class="form-label fw-bold">Repairs Notes</label>
              <input type="text" name="repairs_notes" class="form-control" style="border-radius:12px;" maxlength="2000">
            </div>

            <div class="col-12">
              <label class="form-label fw-bold">Condition Notes</label>
              <textarea name="condition_notes" class="form-control" style="border-radius:12px;" rows="3" maxlength="2000"></textarea>
            </div>

            <div class="col-12">
              <hr>
              <div class="fw-bold mb-2">Add-on Fees (optional)</div>

              <?php if (!$addonTypes): ?>
                <div class="text-muted">No add-on fee types configured.</div>
              <?php else: ?>
                <div class="table-responsive">
                  <table class="table so-table align-middle">
                    <thead>
                      <tr>
                        <th style="width:52px;"></th>
                        <th>Fee</th>
                        <th class="text-end" style="width:180px;">Amount</th>
                      </tr>
                    </thead>
                    <tbody>
                      <?php foreach ($addonTypes as $t): ?>
                        <?php
                          $aid = (int)$t['id'];
                          $defaultAmt = (float)$t['default_amount'];
                          $maxAmt = (float)$t['max_amount'];
                          $driverCanOverrideAmt = (int)$t['driver_can_override_amount'];

                          $amountEditable = false;

                          if ($isCompanyAdmin || $isManager) {
                            $amountEditable = ($maxAmt > 0);
                          } elseif ($isDriver) {
                            $amountEditable = ($driverCanOverrideAmt === 1 && $maxAmt > 0);
                          }

                          $amountHelp = $maxAmt > 0 ? ('Max $' . number_format($maxAmt,2)) : 'No override';
                        ?>
                        <tr>
                          <td>
                            <input class="form-check-input" type="checkbox" name="addon_ids[]" value="<?= $aid ?>">
                          </td>
                          <td>
                            <div class="fw-bold"><?= h((string)$t['name']) ?></div>
                            <?php if (!empty($t['description'])): ?>
                              <div class="text-muted small"><?= h((string)$t['description']) ?></div>
                            <?php endif; ?>
                            <div class="text-muted small">Default: $<?= number_format($defaultAmt,2) ?> · <?= h($amountHelp) ?></div>
                          </td>
                          <td class="text-end">
                            <input
                              type="number"
                              step="0.01"
                              min="0"
                              name="addon_amount[<?= $aid ?>]"
                              class="form-control"
                              style="border-radius:12px; text-align:right;"
                              value="<?= h((string)$defaultAmt) ?>"
                              <?= $amountEditable ? '' : 'readonly' ?>
                            >
                          </td>
                        </tr>
                      <?php endforeach; ?>
                    </tbody>
                  </table>
                </div>
              <?php endif; ?>
            </div>

            <?php if ($showOverrideFields): ?>
              <div class="col-12">
                <hr>
                <div class="fw-bold mb-2">Invoice Total Override (requires reason)</div>
              </div>

              <div class="col-12 col-md-4">
                <label class="form-label fw-bold">Override Total ($)</label>
                <input type="number" step="0.01" min="0" name="override_total" class="form-control" style="border-radius:12px;" placeholder="Leave blank for standard fee">
              </div>

              <div class="col-12 col-md-8">
                <label class="form-label fw-bold">Override Reason</label>
                <input type="text" name="override_reason" class="form-control" style="border-radius:12px;" maxlength="500" placeholder="Required if override total is set">
              </div>
            <?php endif; ?>

          </div>
        </div>

        <div class="modal-footer">
          <button class="btn btn-outline-dark" style="border-radius:12px;" type="button" data-bs-dismiss="modal">Cancel</button>
          <button class="btn btn-dark" style="border-radius:12px;" type="submit">
            <i class="fa-solid fa-paper-plane me-2"></i>Submit Pickup Package
          </button>
        </div>
      </form>
    </div>
  </div>
</div>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
