<?php
// /app/repos/repo_update_status.php
declare(strict_types=1);

require_once __DIR__ . '/../includes/auth.php';
require_company_context_if_rto();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  header('Location: ' . app_path('repos/repos_list.php'));
  exit;
}

$token = (string)($_POST['_csrf'] ?? '');
if (!csrf_validate($token)) {
  http_response_code(403);
  echo 'CSRF validation failed.';
  exit;
}

$repoId = (int)($_POST['repo_id'] ?? 0);
$newStatus = trim((string)($_POST['status'] ?? ''));
$eventNote = trim((string)($_POST['event_note'] ?? ''));

$validStatuses = ['new','assigned','en_route','picked_up','at_yard','closed','canceled'];
if ($repoId < 1 || !in_array($newStatus, $validStatuses, true)) {
  header('Location: ' . app_path('repos/repos_list.php'));
  exit;
}

$pdo = db();
$userId = (int) current_user_id();

$tenantType = current_tenant_type();
$rtoTenantId = (int) current_tenant_id();
$companyTenantId = ($tenantType === 'rto')
  ? (int) active_company_tenant_id()
  : (int) current_tenant_id();

try {
  $pdo->beginTransaction();

  // Lock the repo row to avoid double-processing
  $params = [
    ':company_tid' => $companyTenantId,
    ':repo_id' => $repoId
  ];

  $where = "tenant_id = :company_tid AND id = :repo_id";
  if ($tenantType === 'rto') {
    $where .= " AND rto_tenant_id = :rto_tid";
    $params[':rto_tid'] = $rtoTenantId;
  }

  $stmt = $pdo->prepare("SELECT * FROM repos WHERE $where FOR UPDATE");
  $stmt->execute($params);
  $repo = $stmt->fetch();

  if (!$repo) {
    $pdo->rollBack();
    header('Location: ' . app_path('repos/repos_list.php'));
    exit;
  }

  $oldStatus = (string)$repo['status'];

  // Update repo status
  $pickedUpAt = $repo['picked_up_at'];
  $setPickedUpAt = null;

  if ($newStatus === 'picked_up' && empty($pickedUpAt)) {
    $setPickedUpAt = date('Y-m-d H:i:s');
  }

  $stmt = $pdo->prepare("
    UPDATE repos
    SET status = :status,
        picked_up_at = COALESCE(picked_up_at, :picked_up_at),
        updated_at = CURRENT_TIMESTAMP
    WHERE tenant_id = :company_tid AND id = :repo_id
  ");
  $stmt->execute([
    ':status' => $newStatus,
    ':picked_up_at' => $setPickedUpAt,
    ':company_tid' => $companyTenantId,
    ':repo_id' => $repoId,
  ]);

  // Log event
  $meta = [
    'from' => $oldStatus,
    'to' => $newStatus,
  ];
  if ($eventNote !== '') $meta['note'] = $eventNote;

  $stmt = $pdo->prepare("
    INSERT INTO repo_events (tenant_id, repo_id, event_type, meta_json, created_by_user_id, created_at)
    VALUES (:tid, :rid, :etype, :meta, :uid, CURRENT_TIMESTAMP)
  ");
  $stmt->execute([
    ':tid' => $companyTenantId,
    ':rid' => $repoId,
    ':etype' => 'status_change',
    ':meta' => json_encode($meta, JSON_UNESCAPED_SLASHES),
    ':uid' => $userId,
  ]);

  // Auto-create repair order if picked up
  if ($newStatus === 'picked_up') {
    // Create repair only if it doesn't already exist for this repo (unique constraint will also enforce)
    $stmt = $pdo->prepare("
      SELECT id FROM repairs
      WHERE tenant_id = :tid AND repo_id = :rid
      LIMIT 1
    ");
    $stmt->execute([':tid' => $companyTenantId, ':rid' => $repoId]);
    $existingRepair = $stmt->fetchColumn();

    if (!$existingRepair) {
      $stmt = $pdo->prepare("
        INSERT INTO repairs
          (tenant_id, rto_tenant_id, asset_id, repo_id, location_id, status, priority, source, created_by_user_id, created_at, updated_at)
        VALUES
          (:tid, :rto, :asset_id, :repo_id, :loc_id, 'new', 'medium', 'auto_from_repo_pickup', :uid, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
      ");
      $stmt->execute([
        ':tid' => $companyTenantId,
        ':rto' => $repo['rto_tenant_id'] ?? null,
        ':asset_id' => (int)$repo['asset_id'],
        ':repo_id' => $repoId,
        ':loc_id' => $repo['location_id'] ?? null,
        ':uid' => $userId,
      ]);
    }
  }

  $pdo->commit();

  header('Location: ' . app_path('repos/repo_view.php?id=' . $repoId));
  exit;

} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  http_response_code(500);
  echo 'Failed to update repo. ' . htmlspecialchars($e->getMessage());
  exit;
}
