<?php
// /app/repos/repo_pickup.php
declare(strict_types=1);

require_once __DIR__ . '/../includes/auth.php';
require_company_context_if_rto();

require_once __DIR__ . '/../includes/acl.php';
require_once __DIR__ . '/../includes/scope.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  header('Location: ' . app_path('repos/repos_list.php'));
  exit;
}

if (!csrf_validate((string)($_POST['_csrf'] ?? ''))) {
  http_response_code(403);
  echo 'CSRF validation failed.';
  exit;
}

$repoId = (int)($_POST['repo_id'] ?? 0);
$note = trim((string)($_POST['note'] ?? ''));

if ($repoId < 1) {
  header('Location: ' . app_path('repos/repos_list.php'));
  exit;
}

$pdo = db();
acl_bootstrap($pdo);

$tenantType = current_tenant_type(); // company|rto
$companyTenantId = company_context_id();
$rtoTenantId = (int) current_tenant_id();
$userId = (int) current_user_id();

// RTO can confirm pickup only on their own repos; company users can confirm
try {
  $pdo->beginTransaction();

  // Lock repo (tenant safe)
  $where = "tenant_id = :tid1 AND id = :rid1";
  $params = [':tid1' => $companyTenantId, ':rid1' => $repoId];

  if ($tenantType === 'rto') {
    $where .= " AND rto_tenant_id = :rto1";
    $params[':rto1'] = $rtoTenantId;
  }

  $stmt = $pdo->prepare("SELECT * FROM repos WHERE $where FOR UPDATE");
  $stmt->execute($params);
  $repo = $stmt->fetch();

  if (!$repo) {
    $pdo->rollBack();
    http_response_code(404);
    echo "Repo not found.";
    exit;
  }

  // If already picked up, no-op
  if ((string)$repo['status'] === 'picked_up') {
    $pdo->commit();
    header('Location: ' . app_path('repos/repo_view.php?id=' . $repoId . '&ok=1'));
    exit;
  }

  // Update repo
  $pickedAt = date('Y-m-d H:i:s');
  $stmt = $pdo->prepare("
    UPDATE repos
    SET status='picked_up',
        picked_up_at = COALESCE(picked_up_at, :p1),
        updated_at = CURRENT_TIMESTAMP
    WHERE tenant_id = :tid1 AND id = :rid1
  ");
  $stmt->execute([':p1' => $pickedAt, ':tid1' => $companyTenantId, ':rid1' => $repoId]);

  // Update asset status
  $stmt = $pdo->prepare("
    UPDATE assets
    SET status='repo_recovered',
        updated_at=CURRENT_TIMESTAMP
    WHERE tenant_id=:tid1 AND id=:aid1
  ");
  $stmt->execute([':tid1' => $companyTenantId, ':aid1' => (int)$repo['asset_id']]);

  // Repo event
  $meta = ['action' => 'pickup_confirmed'];
  if ($note !== '') $meta['note'] = $note;

  $stmt = $pdo->prepare("
    INSERT INTO repo_events (tenant_id, repo_id, event_type, meta_json, created_by_user_id, created_at)
    VALUES (:tid1, :rid1, 'pickup_confirmed', :m1, :uid1, CURRENT_TIMESTAMP)
  ");
  $stmt->execute([
    ':tid1' => $companyTenantId,
    ':rid1' => $repoId,
    ':m1' => json_encode($meta, JSON_UNESCAPED_SLASHES),
    ':uid1' => $userId
  ]);

  // Auto-create repair (one-time)
  $stmt = $pdo->prepare("
    SELECT id FROM repairs
    WHERE tenant_id=:tid1 AND repo_id=:rid1
    LIMIT 1
  ");
  $stmt->execute([':tid1' => $companyTenantId, ':rid1' => $repoId]);
  $existingRepair = (int)($stmt->fetchColumn() ?: 0);

  if (!$existingRepair) {
    $stmt = $pdo->prepare("
      INSERT INTO repairs
        (tenant_id, rto_tenant_id, asset_id, repo_id, location_id, status, priority, source, created_by_user_id, created_at, updated_at)
      VALUES
        (:tid1, :rto1, :aid1, :rid1, :loc1, 'new', 'medium', 'auto_from_repo_pickup', :uid1, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
    ");
    $stmt->execute([
      ':tid1' => $companyTenantId,
      ':rto1' => $repo['rto_tenant_id'] ?? null,
      ':aid1' => (int)$repo['asset_id'],
      ':rid1' => $repoId,
      ':loc1' => $repo['location_id'] ?? null,
      ':uid1' => $userId,
    ]);
  }

  $pdo->commit();

  header('Location: ' . app_path('repos/repo_view.php?id=' . $repoId . '&ok=1'));
  exit;

} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  http_response_code(500);
  echo "Pickup confirm failed: " . htmlspecialchars($e->getMessage());
  exit;
}
