<?php
// /app/repos/repo_create.php
declare(strict_types=1);

require_once __DIR__ . '/../includes/auth.php';
require_company_context_if_rto();

require_once __DIR__ . '/../includes/acl.php';
require_once __DIR__ . '/../includes/scope.php';

$pageTitle = 'Create Repo';
$activeNav = 'repos';

$pdo = db();
acl_bootstrap($pdo);

$user = current_user();
$tenantType = current_tenant_type(); // company|rto
$companyTenantId = company_context_id();
$rtoTenantId = (int) current_tenant_id();

$isCompanyAdmin = user_has_role('company_admin');
$isManager = user_has_role('manager');
$canAssignDriver = ($tenantType !== 'rto') && ($isCompanyAdmin || $isManager || user_can('repos.assign_driver') || user_can('repos.view_all'));

// RTO authorization: ensure this RTO is linked to the active company context
if ($tenantType === 'rto') {
  $stmt = $pdo->prepare("
    SELECT id
    FROM rto_company_links
    WHERE rto_tenant_id = :rto1
      AND company_tenant_id = :co1
      AND status = 'active'
    LIMIT 1
  ");
  $stmt->execute([':rto1' => $rtoTenantId, ':co1' => $companyTenantId]);
  if (!$stmt->fetchColumn()) {
    http_response_code(403);
    echo "Forbidden: RTO is not linked to this company.";
    exit;
  }
}

// Assets for selection (company tenant only)
$stmt = $pdo->prepare("
  SELECT id, stock_number, asset_type, width_ft, length_ft, height_ft, status
  FROM assets
  WHERE tenant_id = :tid1
  ORDER BY stock_number ASC
  LIMIT 2000
");
$stmt->execute([':tid1' => $companyTenantId]);
$assets = $stmt->fetchAll();

// Drivers list (role-filtered) if company and allowed
$drivers = [];
if ($canAssignDriver) {
  $stmt = $pdo->prepare("
    SELECT u.id, CONCAT(COALESCE(u.first_name,''), ' ', COALESCE(u.last_name,'')) AS name
    FROM users u
    JOIN user_roles ur ON ur.user_id = u.id
    JOIN roles r ON r.id = ur.role_id
    WHERE u.tenant_id = :tid1
      AND u.status = 'active'
      AND r.tenant_id = :tid2
      AND r.name = 'driver'
    ORDER BY name ASC
    LIMIT 300
  ");
  $stmt->execute([':tid1' => $companyTenantId, ':tid2' => $companyTenantId]);
  $drivers = $stmt->fetchAll();
}

$err = '';
// Default form values
$form = [
  'asset_id' => (int)($_GET['asset_id'] ?? 0),
  'priority' => 'medium',
  'driver_user_id' => 0,
  'pickup_address1' => '',
  'pickup_address2' => '',
  'pickup_city' => '',
  'pickup_state' => '',
  'pickup_zip' => '',
  'notes' => '',
];

require_once __DIR__ . '/../includes/header.php';
?>
<div class="container-fluid">

  <div class="d-flex flex-wrap justify-content-between align-items-end gap-2 mb-3">
    <div>
      <div class="so-page-title">Create Repo</div>
      <div class="so-page-sub">
        <?= h($tenantType === 'rto' ? ('RTO View: ' . ($user['active_company_name'] ?? '')) : 'Company View') ?>
      </div>
    </div>
    <div class="d-flex gap-2">
      <a class="btn so-btn" href="<?= app_path('repos/repos_list.php') ?>"><i class="fa-solid fa-arrow-left me-2"></i>Back</a>
    </div>
  </div>

  <?php if ($err): ?>
    <div class="alert alert-danger"><?= h($err) ?></div>
  <?php endif; ?>

  <div class="row g-3">
    <div class="col-12 col-xl-8">
      <div class="so-card">
        <div class="so-card-h">
          <h5 class="so-card-title"><i class="fa-solid fa-truck-ramp-box me-2" style="color:#cf1f1f"></i>Repo Information</h5>
        </div>
        <div class="so-card-body">

          <form method="post" action="<?= app_path('repos/repo_store.php') ?>" class="row g-3">
            <input type="hidden" name="_csrf" value="<?= h(csrf_token()) ?>">

            <div class="col-12">
              <label class="form-label fw-bold">Asset</label>
              <select name="asset_id" class="form-select" style="border-radius:12px;" required>
                <option value="">— Select an asset —</option>
                <?php foreach ($assets as $a): ?>
                  <?php
                    $aid = (int)$a['id'];
                    $size = trim(($a['width_ft'] ? $a['width_ft'].'x' : '') . ($a['length_ft'] ?? ''));
                    $size = $size ?: strtoupper((string)$a['asset_type']);
                    $label = (string)$a['stock_number'] . ' — ' . $size . ' — ' . (string)$a['status'];
                  ?>
                  <option value="<?= $aid ?>" <?= ($form['asset_id'] === $aid) ? 'selected' : '' ?>>
                    <?= h($label) ?>
                  </option>
                <?php endforeach; ?>
              </select>
              <div class="text-muted small mt-1">
                Recommended: choose an asset that is not currently in an active repo.
              </div>
            </div>

            <div class="col-12 col-md-4">
              <label class="form-label fw-bold">Priority</label>
              <select name="priority" class="form-select" style="border-radius:12px;" required>
                <?php foreach (['low','medium','high'] as $p): ?>
                  <option value="<?= h($p) ?>" <?= ($form['priority'] === $p) ? 'selected' : '' ?>>
                    <?= h(strtoupper($p)) ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>

            <div class="col-12 col-md-8">
              <label class="form-label fw-bold">Assign Driver (optional)</label>
              <select name="driver_user_id" class="form-select" style="border-radius:12px;" <?= $canAssignDriver ? '' : 'disabled' ?>>
                <option value="0">— Unassigned —</option>
                <?php foreach ($drivers as $d): ?>
                  <?php $did = (int)$d['id']; ?>
                  <option value="<?= $did ?>" <?= ((int)$form['driver_user_id'] === $did) ? 'selected' : '' ?>>
                    <?= h(trim((string)$d['name']) ?: ('User #'.$did)) ?>
                  </option>
                <?php endforeach; ?>
              </select>
              <?php if (!$canAssignDriver): ?>
                <div class="text-muted small mt-1">Driver assignment is managed by the manufacturer/admin.</div>
              <?php endif; ?>
            </div>

            <div class="col-12">
              <div class="so-card" style="box-shadow:none;">
                <div class="so-card-h">
                  <h6 class="so-card-title" style="font-size:16px;">
                    <i class="fa-solid fa-location-dot me-2" style="color:#cf1f1f"></i>Pickup Address
                  </h6>
                </div>
                <div class="so-card-body">
                  <div class="row g-2">
                    <div class="col-12 col-md-6">
                      <label class="form-label fw-bold">Address 1</label>
                      <input type="text" name="pickup_address1" class="form-control" style="border-radius:12px;" required maxlength="190">
                    </div>
                    <div class="col-12 col-md-6">
                      <label class="form-label fw-bold">Address 2</label>
                      <input type="text" name="pickup_address2" class="form-control" style="border-radius:12px;" maxlength="190">
                    </div>
                    <div class="col-12 col-md-4">
                      <label class="form-label fw-bold">City</label>
                      <input type="text" name="pickup_city" class="form-control" style="border-radius:12px;" required maxlength="100">
                    </div>
                    <div class="col-12 col-md-4">
                      <label class="form-label fw-bold">State</label>
                      <input type="text" name="pickup_state" class="form-control" style="border-radius:12px;" required maxlength="50" placeholder="KY">
                    </div>
                    <div class="col-12 col-md-4">
                      <label class="form-label fw-bold">Zip</label>
                      <input type="text" name="pickup_zip" class="form-control" style="border-radius:12px;" required maxlength="20">
                    </div>
                  </div>
                </div>
              </div>
            </div>

            <div class="col-12">
              <label class="form-label fw-bold">Notes (optional)</label>
              <textarea name="notes" class="form-control" style="border-radius:12px;" rows="4" maxlength="2000" placeholder="Any pickup instructions, gate codes, contact notes..."></textarea>
            </div>

            <div class="col-12">
              <button class="btn btn-dark" style="border-radius:12px;" type="submit">
                <i class="fa-solid fa-plus me-2"></i>Create Repo
              </button>
            </div>

          </form>

        </div>
      </div>
    </div>

    <div class="col-12 col-xl-4">
      <div class="so-card">
        <div class="so-card-h">
          <h5 class="so-card-title"><i class="fa-solid fa-shield-halved me-2" style="color:#cf1f1f"></i>Validation Rules</h5>
        </div>
        <div class="so-card-body text-muted">
          <ul class="mb-0">
            <li>Asset must belong to the selected company.</li>
            <li>RTO users can only create repos for linked companies.</li>
            <li>RTO repos are tagged to the RTO tenant automatically.</li>
            <li>Only one active repo per asset is allowed (enforced on save).</li>
          </ul>
        </div>
      </div>
    </div>

  </div>

</div>
<?php require_once __DIR__ . '/../includes/footer.php'; ?>
