<?php
// /app/repos/repo_assign_driver.php
declare(strict_types=1);

require_once __DIR__ . '/../includes/auth.php';
require_company_context_if_rto();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  header('Location: ' . app_path('repos/repos_list.php'));
  exit;
}

if (!csrf_validate((string)($_POST['_csrf'] ?? ''))) {
  http_response_code(403);
  echo 'CSRF validation failed.';
  exit;
}

$repoId = (int)($_POST['repo_id'] ?? 0);
$driverUserId = (int)($_POST['driver_user_id'] ?? 0);

if ($repoId < 1) {
  header('Location: ' . app_path('repos/repos_list.php'));
  exit;
}

// Business rule: RTO cannot assign drivers
if (current_tenant_type() === 'rto') {
  http_response_code(403);
  echo 'RTO accounts cannot assign drivers.';
  exit;
}

$pdo = db();
$companyTenantId = (int) current_tenant_id();
$actorId = (int) current_user_id();

try {
  $pdo->beginTransaction();

  // Lock repo row
  $stmt = $pdo->prepare("
    SELECT id, assigned_driver_user_id
    FROM repos
    WHERE tenant_id = :tid AND id = :rid
    FOR UPDATE
  ");
  $stmt->execute([':tid' => $companyTenantId, ':rid' => $repoId]);
  $repo = $stmt->fetch();

  if (!$repo) {
    $pdo->rollBack();
    header('Location: ' . app_path('repos/repos_list.php'));
    exit;
  }

  $oldDriver = (int)($repo['assigned_driver_user_id'] ?? 0);

  // Validate driver user belongs to same company tenant (or allow unassign with 0)
  if ($driverUserId > 0) {
    $stmt = $pdo->prepare("
      SELECT id
      FROM users
      WHERE tenant_id = :tid AND id = :uid AND status='active'
      LIMIT 1
    ");
    $stmt->execute([':tid' => $companyTenantId, ':uid' => $driverUserId]);
    if (!$stmt->fetchColumn()) {
      $pdo->rollBack();
      http_response_code(400);
      echo 'Invalid driver selection.';
      exit;
    }
  }

  $stmt = $pdo->prepare("
    UPDATE repos
    SET assigned_driver_user_id = :duid,
        updated_at = CURRENT_TIMESTAMP
    WHERE tenant_id = :tid AND id = :rid
  ");
  $stmt->execute([
    ':duid' => ($driverUserId > 0 ? $driverUserId : null),
    ':tid' => $companyTenantId,
    ':rid' => $repoId,
  ]);

  // Log event
  $meta = [
    'from_driver_user_id' => $oldDriver ?: null,
    'to_driver_user_id' => $driverUserId ?: null,
  ];

  $stmt = $pdo->prepare("
    INSERT INTO repo_events (tenant_id, repo_id, event_type, meta_json, created_by_user_id, created_at)
    VALUES (:tid, :rid, 'driver_assigned', :meta, :uid, CURRENT_TIMESTAMP)
  ");
  $stmt->execute([
    ':tid' => $companyTenantId,
    ':rid' => $repoId,
    ':meta' => json_encode($meta, JSON_UNESCAPED_SLASHES),
    ':uid' => $actorId,
  ]);

  $pdo->commit();

  header('Location: ' . app_path('repos/repo_view.php?id=' . $repoId . '&ok=1'));
  exit;

} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  http_response_code(500);
  echo 'Failed to assign driver. ' . htmlspecialchars($e->getMessage());
}
