<?php
// /app/repos/invoices_list.php
declare(strict_types=1);

require_once __DIR__ . '/../includes/auth.php';
require_company_context_if_rto();

require_once __DIR__ . '/../includes/acl.php';
require_once __DIR__ . '/../includes/scope.php';

$pageTitle = 'Repo Invoices';
$activeNav = 'repos';

$pdo = db();
acl_bootstrap($pdo);

$user = current_user();

$tenantType = current_tenant_type(); // company|rto
$companyTenantId = company_context_id();
$rtoTenantId = (int) current_tenant_id();

// ---- schema detection (prevents fatal when columns are missing)
function table_has_column(PDO $pdo, string $table, string $column): bool {
  $stmt = $pdo->prepare("
    SELECT COUNT(*)
    FROM information_schema.COLUMNS
    WHERE TABLE_SCHEMA = :db
      AND TABLE_NAME = :tbl
      AND COLUMN_NAME = :col
    LIMIT 1
  ");
  $stmt->execute([':db' => DB_NAME, ':tbl' => $table, ':col' => $column]);
  return ((int)$stmt->fetchColumn() === 1);
}

$hasBilledTo = table_has_column($pdo, 'repo_invoices', 'billed_to');
$hasSubmittedBy = table_has_column($pdo, 'repo_invoices', 'submitted_by_user_id');
$hasApprovedBy = table_has_column($pdo, 'repo_invoices', 'approved_by_user_id');

// Filters
$status = trim((string)($_GET['status'] ?? 'submitted'));
$billedTo = trim((string)($_GET['billed_to'] ?? ''));
$q = trim((string)($_GET['q'] ?? ''));

$validStatuses = ['draft','submitted','approved','final','void'];
if ($status !== '' && !in_array($status, $validStatuses, true)) $status = 'submitted';

$validBilledTo = ['company','rto'];
if ($billedTo !== '' && !in_array($billedTo, $validBilledTo, true)) $billedTo = '';

// WHERE
$where = [];
$params = [];

$where[] = "inv.tenant_id = :tid1";
$params[':tid1'] = $companyTenantId;

if ($tenantType === 'rto') {
  $where[] = "r.rto_tenant_id = :rto1";
  $params[':rto1'] = $rtoTenantId;
}

if ($status !== '') {
  $where[] = "inv.status = :st1";
  $params[':st1'] = $status;
}

if ($hasBilledTo && $billedTo !== '') {
  $where[] = "inv.billed_to = :bt1";
  $params[':bt1'] = $billedTo;
}

if ($q !== '') {
  $where[] = "(
    CAST(inv.id AS CHAR) LIKE :q1
    OR CAST(inv.repo_id AS CHAR) LIKE :q1
    OR a.stock_number LIKE :q1
    OR t_rto.name LIKE :q1
  )";
  $params[':q1'] = '%' . $q . '%';
}

$whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

// SELECT pieces based on schema
$selectBilledTo = $hasBilledTo ? "inv.billed_to," : "'rto' AS billed_to,";
$selectSubmitted = $hasSubmittedBy ? "CONCAT(COALESCE(u_sub.first_name,''),' ',COALESCE(u_sub.last_name,'')) AS submitted_by_name," : "NULL AS submitted_by_name,";
$selectApproved = $hasApprovedBy ? "CONCAT(COALESCE(u_app.first_name,''),' ',COALESCE(u_app.last_name,'')) AS approved_by_name," : "NULL AS approved_by_name,";

$joinSubmitted = $hasSubmittedBy ? "LEFT JOIN users u_sub ON u_sub.id = inv.submitted_by_user_id" : "";
$joinApproved = $hasApprovedBy ? "LEFT JOIN users u_app ON u_app.id = inv.approved_by_user_id" : "";

$sql = "
  SELECT
    inv.id AS invoice_id,
    inv.repo_id,
    inv.status,
    $selectBilledTo
    inv.total,
    inv.subtotal,
    inv.overridden,
    inv.override_reason,
    inv.created_at,
    inv.updated_at,

    r.status AS repo_status,
    r.priority AS repo_priority,

    a.stock_number,
    a.width_ft,
    a.length_ft,
    a.asset_type,

    t_rto.name AS rto_name,

    $selectSubmitted
    $selectApproved
    1 AS _end
  FROM repo_invoices inv
  JOIN repos r ON r.id = inv.repo_id
  JOIN assets a ON a.id = r.asset_id
  LEFT JOIN tenants t_rto ON t_rto.id = r.rto_tenant_id
  $joinSubmitted
  $joinApproved
  $whereSql
  ORDER BY
    FIELD(inv.status,'submitted','draft','approved','final','void'),
    inv.updated_at DESC
  LIMIT 300
";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$rows = $stmt->fetchAll();

function pill_status(string $s): string {
  return match ($s) {
    'submitted' => 'orange',
    'approved' => 'blue',
    'final' => 'gray',
    'draft' => 'blue',
    'void' => 'gray',
    default => 'gray'
  };
}

require_once __DIR__ . '/../includes/header.php';
?>
<div class="container-fluid">

  <div class="d-flex flex-wrap justify-content-between align-items-end gap-2 mb-3">
    <div>
      <div class="so-page-title">Repo Invoices</div>
      <div class="so-page-sub">
        <?= h($tenantType === 'rto' ? ('RTO View: ' . ($user['active_company_name'] ?? '')) : 'Company View') ?>
      </div>
    </div>
    <div class="d-flex gap-2">
      <a class="btn so-btn" href="<?= app_path('repos/invoices_list.php') ?>"><i class="fa-solid fa-rotate me-2"></i>Refresh</a>
      <a class="btn btn-outline-dark" style="border-radius:12px;" href="<?= app_path('repos/repos_list.php') ?>">
        <i class="fa-solid fa-truck-ramp-box me-2"></i>Repos
      </a>
    </div>
  </div>

  <div class="so-card mb-3">
    <div class="so-card-body">
      <form class="row g-2 align-items-end" method="get">

        <div class="col-12 col-md-3">
          <label class="form-label fw-bold">Status</label>
          <select name="status" class="form-select" style="border-radius:12px;">
            <?php foreach (['submitted','draft','approved','final','void'] as $s): ?>
              <option value="<?= h($s) ?>" <?= $status===$s?'selected':'' ?>><?= h(strtoupper($s)) ?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div class="col-12 col-md-2">
          <label class="form-label fw-bold">Billed To</label>
          <select name="billed_to" class="form-select" style="border-radius:12px;" <?= $hasBilledTo ? '' : 'disabled' ?>>
            <option value="">All</option>
            <option value="rto" <?= $billedTo==='rto'?'selected':'' ?>>RTO</option>
            <option value="company" <?= $billedTo==='company'?'selected':'' ?>>Company</option>
          </select>
          <?php if (!$hasBilledTo): ?>
            <div class="text-muted small mt-1">Column billed_to not in DB yet.</div>
          <?php endif; ?>
        </div>

        <div class="col-12 col-md-5">
          <label class="form-label fw-bold">Search</label>
          <input type="text" name="q" value="<?= h($q) ?>" class="form-control" style="border-radius:12px;" placeholder="Invoice ID, Repo ID, Stock #, RTO name...">
        </div>

        <div class="col-12 col-md-2">
          <button class="btn btn-dark w-100" style="border-radius:12px;" type="submit">
            <i class="fa-solid fa-filter me-2"></i>Filter
          </button>
        </div>

        <div class="col-12 mt-2">
          <a class="btn btn-outline-dark" style="border-radius:12px;" href="<?= app_path('repos/invoices_list.php') ?>">Clear</a>
        </div>

      </form>
    </div>
  </div>

  <div class="so-card">
    <div class="so-card-h">
      <h5 class="so-card-title">
        <i class="fa-solid fa-receipt me-2" style="color:#cf1f1f"></i>
        Invoice Queue
      </h5>
      <div class="text-muted small">Showing up to 300 results</div>
    </div>

    <div class="so-card-body">
      <div class="table-responsive">
        <table class="table so-table align-middle">
          <thead>
            <tr>
              <th>Invoice</th>
              <th>Repo</th>
              <th>Asset</th>
              <th>RTO</th>
              <th>Status</th>
              <th class="text-end">Total</th>
              <th>Override</th>
              <th class="text-end">Updated</th>
              <th class="text-end">Action</th>
            </tr>
          </thead>
          <tbody>
          <?php if (!$rows): ?>
            <tr><td colspan="9" class="text-muted">No invoices found.</td></tr>
          <?php else: ?>
            <?php foreach ($rows as $r): ?>
              <?php
                $size = trim(((int)$r['width_ft'] ? (int)$r['width_ft'].'x' : '') . ((int)$r['length_ft'] ?? ''));
                $size = $size ?: strtoupper((string)$r['asset_type']);
                $updated = (string)($r['updated_at'] ?: $r['created_at']);
              ?>
              <tr>
                <td>
                  <strong>#<?= (int)$r['invoice_id'] ?></strong>
                  <div class="text-muted small"><?= h((string)$r['billed_to']) ?></div>
                </td>
                <td>
                  <strong>#<?= (int)$r['repo_id'] ?></strong>
                  <div class="text-muted small"><?= h((string)$r['repo_status']) ?></div>
                </td>
                <td>
                  <strong><?= h((string)$r['stock_number']) ?></strong>
                  <div class="text-muted small"><?= h($size) ?></div>
                </td>
                <td><strong><?= h($r['rto_name'] ?? '—') ?></strong></td>
                <td>
                  <span class="so-pill <?= h(pill_status((string)$r['status'])) ?>">
                    <?= h(strtoupper((string)$r['status'])) ?>
                  </span>
                </td>
                <td class="text-end fw-bold">$<?= number_format((float)$r['total'], 2) ?></td>
                <td>
                  <?php if ((int)$r['overridden'] === 1): ?>
                    <span class="so-pill orange">Yes</span>
                    <div class="text-muted small"><?= h(mb_substr((string)$r['override_reason'], 0, 40)) ?></div>
                  <?php else: ?>
                    <span class="so-pill gray">No</span>
                  <?php endif; ?>
                </td>
                <td class="text-end text-muted small"><?= h(date('M j, Y g:i A', strtotime($updated))) ?></td>
                <td class="text-end">
                  <a class="btn btn-sm btn-dark" style="border-radius:12px;" href="<?= app_path('repos/repo_view.php?id='.(int)$r['repo_id']) ?>">
                    Open <i class="fa-solid fa-arrow-right ms-2"></i>
                  </a>
                </td>
              </tr>
            <?php endforeach; ?>
          <?php endif; ?>
          </tbody>
        </table>
      </div>
    </div>
  </div>

</div>
<?php require_once __DIR__ . '/../includes/footer.php'; ?>
