<?php
// /app/repairs/repair_view.php
declare(strict_types=1);

require_once __DIR__ . '/../includes/auth.php';
require_company_context_if_rto();

$pageTitle = 'Repair Details';
$activeNav = 'repairs';

$pdo = db();

$tenantType = current_tenant_type(); // company|rto
$rtoTenantId = (int) current_tenant_id();
$companyTenantId = ($tenantType === 'rto')
  ? (int) active_company_tenant_id()
  : (int) current_tenant_id();

$repairId = (int)($_GET['id'] ?? 0);
if ($repairId < 1) {
  header('Location: ' . app_path('repairs/repairs_list.php'));
  exit;
}

$where = "rp.tenant_id = :company_tid AND rp.id = :rid";
$params = [':company_tid' => $companyTenantId, ':rid' => $repairId];

if ($tenantType === 'rto') {
  $where .= " AND rp.rto_tenant_id = :rto_tid";
  $params[':rto_tid'] = $rtoTenantId;
}

$sql = "
  SELECT
    rp.*,
    a.stock_number, a.asset_type, a.length_ft, a.width_ft, a.height_ft, a.serial_number,
    t_rto.name AS rto_name,
    CONCAT(COALESCE(u.first_name,''), ' ', COALESCE(u.last_name,'')) AS tech_name
  FROM repairs rp
  JOIN assets a ON a.id = rp.asset_id
  LEFT JOIN tenants t_rto ON t_rto.id = rp.rto_tenant_id
  LEFT JOIN users u ON u.id = rp.assigned_tech_user_id
  WHERE $where
  LIMIT 1
";
$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$repair = $stmt->fetch();

if (!$repair) {
  header('Location: ' . app_path('repairs/repairs_list.php'));
  exit;
}

/**
 * Technicians list (company users only) — ROLE FILTERED
 * IMPORTANT: Use unique placeholder names (PDO emulate_prepares is OFF).
 */
$techs = [];
if ($tenantType !== 'rto') {
  $stmt = $pdo->prepare("
    SELECT u.id, CONCAT(COALESCE(u.first_name,''), ' ', COALESCE(u.last_name,'')) AS name
    FROM users u
    JOIN user_roles ur ON ur.user_id = u.id
    JOIN roles r ON r.id = ur.role_id
    WHERE u.tenant_id = :tid1
      AND u.status = 'active'
      AND r.tenant_id = :tid2
      AND r.name = 'repair_tech'
    ORDER BY name ASC
    LIMIT 300
  ");
  $stmt->execute([
    ':tid1' => $companyTenantId,
    ':tid2' => $companyTenantId,
  ]);
  $techs = $stmt->fetchAll();
}

// Load repair items
$stmt = $pdo->prepare("
  SELECT id, category, description, estimated_cost, actual_cost, item_status, created_at
  FROM repair_items
  WHERE tenant_id = :tid AND repair_id = :rid
  ORDER BY
    FIELD(item_status,'open','done'),
    id DESC
");
$stmt->execute([':tid' => $companyTenantId, ':rid' => $repairId]);
$items = $stmt->fetchAll();

// Totals
$estTotal = 0.0;
$actTotal = 0.0;
foreach ($items as $it) {
  $estTotal += (float)($it['estimated_cost'] ?? 0);
  $actTotal += (float)($it['actual_cost'] ?? 0);
}

$statuses = ['new','assigned','in_progress','waiting_parts','completed','canceled'];
$priorities = ['high','medium','low'];

$flash = '';
if (!empty($_GET['ok'])) $flash = 'Saved successfully.';

require_once __DIR__ . '/../includes/header.php';
?>
<div class="container-fluid">

  <div class="d-flex flex-wrap justify-content-between align-items-end gap-2 mb-3">
    <div>
      <div class="so-page-title">Repair #<?= (int)$repair['id'] ?></div>
      <div class="so-page-sub">
        Asset: <strong><?= h($repair['stock_number']) ?></strong>
        <?php if (!empty($repair['repo_id'])): ?>
          · Repo: <strong>#<?= (int)$repair['repo_id'] ?></strong>
        <?php endif; ?>
        <?php if (!empty($repair['rto_name'])): ?>
          · RTO: <strong><?= h((string)$repair['rto_name']) ?></strong>
        <?php endif; ?>
      </div>
    </div>
    <div class="d-flex gap-2">
      <a class="btn so-btn" href="<?= app_path('repairs/repairs_list.php') ?>"><i class="fa-solid fa-arrow-left me-2"></i>Back</a>
      <?php if (!empty($repair['repo_id'])): ?>
        <a class="btn btn-outline-dark" style="border-radius:12px;" href="<?= app_path('repos/repo_view.php?id='.(int)$repair['repo_id']) ?>">
          <i class="fa-solid fa-link me-2"></i>View Repo
        </a>
      <?php endif; ?>
    </div>
  </div>

  <?php if ($flash): ?>
    <div class="alert alert-success"><?= h($flash) ?></div>
  <?php endif; ?>

  <div class="row g-3">
    <!-- LEFT: REPAIR DETAILS + UPDATE -->
    <div class="col-12 col-xl-7">
      <div class="so-card">
        <div class="so-card-h">
          <h5 class="so-card-title"><i class="fa-solid fa-circle-info me-2" style="color:#cf1f1f"></i>Repair Details</h5>
        </div>
        <div class="so-card-body">

          <div class="row g-3">
            <div class="col-12 col-md-6">
              <div class="text-muted small">Status</div>
              <div class="fw-bold"><?= h(strtoupper(str_replace('_',' ', (string)$repair['status']))) ?></div>
            </div>
            <div class="col-12 col-md-6">
              <div class="text-muted small">Priority</div>
              <div class="fw-bold"><?= h(strtoupper((string)$repair['priority'])) ?></div>
            </div>

            <div class="col-12 col-md-6">
              <div class="text-muted small">Assigned Tech</div>
              <div class="fw-bold"><?= h(trim((string)$repair['tech_name']) ?: '—') ?></div>
            </div>
            <div class="col-12 col-md-6">
              <div class="text-muted small">Source</div>
              <div class="fw-bold"><?= h(str_replace('_',' ', (string)$repair['source'])) ?></div>
            </div>
          </div>

          <hr>

          <?php if ($tenantType !== 'rto'): ?>
            <div class="row g-2 align-items-end">
              <div class="col-12">
                <label class="form-label fw-bold">Assign Technician</label>
                <form method="post" action="<?= app_path('repairs/repair_assign_tech.php') ?>" class="d-flex gap-2 flex-wrap">
                  <input type="hidden" name="_csrf" value="<?= h(csrf_token()) ?>">
                  <input type="hidden" name="repair_id" value="<?= (int)$repair['id'] ?>">

                  <select name="tech_user_id" class="form-select" style="border-radius:12px; min-width: 260px;" required>
                    <option value="0">— Unassigned —</option>
                    <?php foreach ($techs as $t): ?>
                      <?php $tid = (int)$t['id']; ?>
                      <option value="<?= $tid ?>" <?= ((int)$repair['assigned_tech_user_id'] === $tid) ? 'selected' : '' ?>>
                        <?= h(trim((string)$t['name']) ?: ('User #'.$tid)) ?>
                      </option>
                    <?php endforeach; ?>
                  </select>

                  <button class="btn btn-dark" style="border-radius:12px;" type="submit">
                    <i class="fa-solid fa-user-check me-2"></i>Save Tech
                  </button>
                </form>
                <div class="text-muted small mt-1">
                  Technician list is role-filtered to users with role <strong>repair_tech</strong>.
                </div>
              </div>
            </div>
            <hr>
          <?php else: ?>
            <div class="text-muted small">
              Technician assignment is managed by the manufacturer.
            </div>
            <hr>
          <?php endif; ?>

          <form method="post" action="<?= app_path('repairs/repair_update_status.php') ?>" class="row g-2 align-items-end">
            <input type="hidden" name="_csrf" value="<?= h(csrf_token()) ?>">
            <input type="hidden" name="repair_id" value="<?= (int)$repair['id'] ?>">

            <div class="col-12 col-md-4">
              <label class="form-label fw-bold">Update Status</label>
              <select name="status" class="form-select" style="border-radius:12px;" required>
                <?php foreach ($statuses as $s): ?>
                  <option value="<?= h($s) ?>" <?= ((string)$repair['status'] === $s) ? 'selected' : '' ?>>
                    <?= h(strtoupper(str_replace('_',' ', $s))) ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>

            <div class="col-12 col-md-4">
              <label class="form-label fw-bold">Priority</label>
              <select name="priority" class="form-select" style="border-radius:12px;" required>
                <?php foreach ($priorities as $p): ?>
                  <option value="<?= h($p) ?>" <?= ((string)$repair['priority'] === $p) ? 'selected' : '' ?>>
                    <?= h(strtoupper($p)) ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>

            <div class="col-12 col-md-4">
              <label class="form-label fw-bold">Event Note (optional)</label>
              <input type="text" name="event_note" class="form-control" style="border-radius:12px;" maxlength="240" placeholder="Short note...">
            </div>

            <div class="col-12">
              <button class="btn btn-dark" style="border-radius:12px;" type="submit">
                <i class="fa-solid fa-floppy-disk me-2"></i>Save Changes
              </button>
            </div>
          </form>

        </div>
      </div>

      <!-- REPAIR ITEMS -->
      <div class="so-card mt-3">
        <div class="so-card-h">
          <div>
            <h5 class="so-card-title"><i class="fa-solid fa-list-check me-2" style="color:#cf1f1f"></i>Repair Items</h5>
            <div class="text-muted small mt-1">
              Estimated: <strong>$<?= number_format($estTotal, 2) ?></strong>
              · Actual: <strong>$<?= number_format($actTotal, 2) ?></strong>
            </div>
          </div>
          <button class="btn so-btn btn-sm" type="button" data-bs-toggle="collapse" data-bs-target="#addItem">
            <i class="fa-solid fa-plus me-2"></i>Add Item
          </button>
        </div>

        <div class="so-card-body">
          <div class="collapse mb-3" id="addItem">
            <form method="post" action="<?= app_path('repairs/repair_item_add.php') ?>" class="row g-2">
              <input type="hidden" name="_csrf" value="<?= h(csrf_token()) ?>">
              <input type="hidden" name="repair_id" value="<?= (int)$repair['id'] ?>">

              <div class="col-12 col-md-3">
                <label class="form-label fw-bold">Category</label>
                <select name="category" class="form-select" style="border-radius:12px;">
                  <option value="">—</option>
                  <?php foreach (['door','roof','siding','trim','floor','electrical','paint','hardware','other'] as $c): ?>
                    <option value="<?= h($c) ?>"><?= h(ucfirst($c)) ?></option>
                  <?php endforeach; ?>
                </select>
              </div>

              <div class="col-12 col-md-5">
                <label class="form-label fw-bold">Description</label>
                <input type="text" name="description" class="form-control" style="border-radius:12px;" required maxlength="500" placeholder="e.g., Replace damaged door trim">
              </div>

              <div class="col-6 col-md-2">
                <label class="form-label fw-bold">Est. $</label>
                <input type="number" name="estimated_cost" class="form-control" style="border-radius:12px;" step="0.01" min="0" placeholder="0.00">
              </div>

              <div class="col-6 col-md-2">
                <label class="form-label fw-bold">Act. $</label>
                <input type="number" name="actual_cost" class="form-control" style="border-radius:12px;" step="0.01" min="0" placeholder="0.00">
              </div>

              <div class="col-12">
                <button class="btn btn-dark" style="border-radius:12px;" type="submit">
                  <i class="fa-solid fa-plus me-2"></i>Add Item
                </button>
              </div>
            </form>
            <hr>
          </div>

          <div class="table-responsive">
            <table class="table so-table align-middle">
              <thead>
                <tr>
                  <th>Status</th>
                  <th>Category</th>
                  <th>Description</th>
                  <th class="text-end">Est.</th>
                  <th class="text-end">Act.</th>
                  <th class="text-end">Actions</th>
                </tr>
              </thead>
              <tbody>
                <?php if (!$items): ?>
                  <tr><td colspan="6" class="text-muted">No repair items yet.</td></tr>
                <?php else: ?>
                  <?php foreach ($items as $it): ?>
                    <?php
                      $isDone = ((string)$it['item_status'] === 'done');
                      $cat = (string)($it['category'] ?? '');
                      $est = $it['estimated_cost'] !== null ? (float)$it['estimated_cost'] : null;
                      $act = $it['actual_cost'] !== null ? (float)$it['actual_cost'] : null;
                    ?>
                    <tr>
                      <td>
                        <span class="so-pill <?= $isDone ? 'gray' : 'blue' ?>">
                          <?= $isDone ? 'DONE' : 'OPEN' ?>
                        </span>
                      </td>
                      <td><?= h($cat ?: '—') ?></td>
                      <td>
                        <div class="fw-bold"><?= h((string)$it['description']) ?></div>
                        <div class="text-muted small">Added: <?= h(date('M j, Y', strtotime((string)$it['created_at']))) ?></div>
                      </td>
                      <td class="text-end"><?= $est === null ? '—' : ('$' . number_format($est, 2)) ?></td>
                      <td class="text-end"><?= $act === null ? '—' : ('$' . number_format($act, 2)) ?></td>
                      <td class="text-end">
                        <form method="post" action="<?= app_path('repairs/repair_item_toggle.php') ?>" class="d-inline">
                          <input type="hidden" name="_csrf" value="<?= h(csrf_token()) ?>">
                          <input type="hidden" name="repair_id" value="<?= (int)$repair['id'] ?>">
                          <input type="hidden" name="item_id" value="<?= (int)$it['id'] ?>">
                          <button class="btn btn-sm btn-outline-dark" style="border-radius:12px;" type="submit">
                            <?= $isDone ? 'Reopen' : 'Mark Done' ?>
                          </button>
                        </form>

                        <button
                          type="button"
                          class="btn btn-sm btn-outline-dark ms-1"
                          style="border-radius:12px;"
                          data-bs-toggle="modal"
                          data-bs-target="#editItemModal"
                          data-item-id="<?= (int)$it['id'] ?>"
                          data-item-category="<?= h($cat) ?>"
                          data-item-description="<?= h((string)$it['description']) ?>"
                          data-item-est="<?= $est === null ? '' : h((string)$est) ?>"
                          data-item-act="<?= $act === null ? '' : h((string)$act) ?>"
                        >
                          Edit
                        </button>

                        <form method="post" action="<?= app_path('repairs/repair_item_delete.php') ?>" class="d-inline"
                              onsubmit="return confirm('Delete this repair item?');">
                          <input type="hidden" name="_csrf" value="<?= h(csrf_token()) ?>">
                          <input type="hidden" name="repair_id" value="<?= (int)$repair['id'] ?>">
                          <input type="hidden" name="item_id" value="<?= (int)$it['id'] ?>">
                          <button class="btn btn-sm btn-outline-danger ms-1" style="border-radius:12px;" type="submit">
                            Delete
                          </button>
                        </form>
                      </td>
                    </tr>
                  <?php endforeach; ?>
                <?php endif; ?>
              </tbody>
            </table>
          </div>

        </div>
      </div>
    </div>

    <!-- RIGHT: ASSET SUMMARY -->
    <div class="col-12 col-xl-5">
      <div class="so-card">
        <div class="so-card-h">
          <h5 class="so-card-title"><i class="fa-solid fa-box me-2" style="color:#cf1f1f"></i>Asset</h5>
        </div>
        <div class="so-card-body">
          <div><span class="text-muted small">Stock #</span><div class="fw-bold"><?= h((string)$repair['stock_number']) ?></div></div>
          <div class="mt-2"><span class="text-muted small">Type</span><div class="fw-bold"><?= h((string)$repair['asset_type']) ?></div></div>
          <div class="mt-2"><span class="text-muted small">Size</span>
            <div class="fw-bold">
              <?= h(($repair['width_ft'] ?? '') . 'x' . ($repair['length_ft'] ?? '')) ?><?= $repair['height_ft'] ? (' x ' . h((string)$repair['height_ft'])) : '' ?>
            </div>
          </div>
          <div class="mt-2"><span class="text-muted small">Serial</span><div class="fw-bold"><?= h((string)($repair['serial_number'] ?? '—')) ?></div></div>

          <?php if (!empty($repair['repo_id'])): ?>
            <hr>
            <div class="text-muted small">Linked Repo</div>
            <a class="btn so-btn mt-2" href="<?= app_path('repos/repo_view.php?id='.(int)$repair['repo_id']) ?>">
              View Repo #<?= (int)$repair['repo_id'] ?> <i class="fa-solid fa-arrow-right ms-2"></i>
            </a>
          <?php endif; ?>
        </div>
      </div>
    </div>
  </div>

</div>

<!-- Edit Item Modal -->
<div class="modal fade" id="editItemModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-centered">
    <div class="modal-content" style="border-radius:16px; overflow:hidden;">
      <div class="modal-header" style="background:#111318; color:#fff;">
        <h5 class="modal-title"><i class="fa-solid fa-pen-to-square me-2" style="color:#cf1f1f"></i>Edit Repair Item</h5>
        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <form method="post" action="<?= app_path('repairs/repair_item_update.php') ?>">
        <div class="modal-body">
          <input type="hidden" name="_csrf" value="<?= h(csrf_token()) ?>">
          <input type="hidden" name="repair_id" value="<?= (int)$repair['id'] ?>">
          <input type="hidden" name="item_id" id="edit_item_id" value="">

          <div class="row g-2">
            <div class="col-12 col-md-3">
              <label class="form-label fw-bold">Category</label>
              <select name="category" id="edit_category" class="form-select" style="border-radius:12px;">
                <option value="">—</option>
                <?php foreach (['door','roof','siding','trim','floor','electrical','paint','hardware','other'] as $c): ?>
                  <option value="<?= h($c) ?>"><?= h(ucfirst($c)) ?></option>
                <?php endforeach; ?>
              </select>
            </div>
            <div class="col-12 col-md-9">
              <label class="form-label fw-bold">Description</label>
              <input type="text" name="description" id="edit_description" class="form-control" style="border-radius:12px;" required maxlength="500">
            </div>
            <div class="col-6 col-md-3">
              <label class="form-label fw-bold">Est. $</label>
              <input type="number" name="estimated_cost" id="edit_est" class="form-control" style="border-radius:12px;" step="0.01" min="0">
            </div>
            <div class="col-6 col-md-3">
              <label class="form-label fw-bold">Act. $</label>
              <input type="number" name="actual_cost" id="edit_act" class="form-control" style="border-radius:12px;" step="0.01" min="0">
            </div>
          </div>
        </div>
        <div class="modal-footer">
          <button class="btn btn-outline-dark" style="border-radius:12px;" type="button" data-bs-dismiss="modal">Cancel</button>
          <button class="btn btn-dark" style="border-radius:12px;" type="submit">
            <i class="fa-solid fa-floppy-disk me-2"></i>Save Item
          </button>
        </div>
      </form>
    </div>
  </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function(){
  const modal = document.getElementById('editItemModal');
  if (!modal) return;

  modal.addEventListener('show.bs.modal', function(event){
    const btn = event.relatedTarget;
    document.getElementById('edit_item_id').value = btn.getAttribute('data-item-id') || '';
    document.getElementById('edit_category').value = btn.getAttribute('data-item-category') || '';
    document.getElementById('edit_description').value = btn.getAttribute('data-item-description') || '';
    document.getElementById('edit_est').value = btn.getAttribute('data-item-est') || '';
    document.getElementById('edit_act').value = btn.getAttribute('data-item-act') || '';
  });
});
</script>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
