<?php
// /app/operations/workorders_list.php
declare(strict_types=1);

require_once __DIR__ . '/_lib.php';
if (!ops_can_view()) { http_response_code(403); exit('Forbidden'); }

$pageTitle = 'Operations · Work Orders';
$activeNav = 'operations';
$activeOps = 'workorders';

$pdo = db_safe();
$companyId = ops_require_company_scope();

// Filters
$q         = trim((string)($_GET['q'] ?? ''));
$status    = trim((string)($_GET['status'] ?? ''));
$serviceId = (int)($_GET['service_id'] ?? 0);

$page = (int)($_GET['page'] ?? 1);
if ($page < 1) $page = 1;
$perPage = 25;
$offset  = ($page - 1) * $perPage;

$statuses = [
  ''            => 'All Statuses',
  'new'         => 'New',
  'scheduled'   => 'Scheduled',
  'in_progress' => 'In Progress',
  'blocked'     => 'Blocked',
  'completed'   => 'Completed',
  'canceled'    => 'Canceled',
];

function ops_qs(array $override = []): string {
  $base = $_GET;
  foreach ($override as $k => $v) {
    if ($v === null || $v === '') unset($base[$k]);
    else $base[$k] = $v;
  }
  return '?' . http_build_query($base);
}

try {
  $servicesStmt = $pdo->prepare("
    SELECT id, service_name
    FROM ops_services
    WHERE company_id = :cid AND is_active = 1
    ORDER BY service_name ASC
  ");
  $servicesStmt->execute([':cid' => $companyId]);
  $services = $servicesStmt->fetchAll(PDO::FETCH_ASSOC);

  $where = ["wo.company_id = :cid"];
  $params = [':cid' => $companyId];

  if ($status !== '' && isset($statuses[$status])) {
    $where[] = "wo.status = :status";
    $params[':status'] = $status;
  }
  if ($serviceId > 0) {
    $where[] = "wo.service_id = :sid";
    $params[':sid'] = $serviceId;
  }
  if ($q !== '') {
    $where[] = "(
      wo.customer_name LIKE :q
      OR wo.external_order_id LIKE :q
      OR wo.external_order_number LIKE :q
      OR wo.summary LIKE :q
      OR wo.serial_number LIKE :q
    )";
    $params[':q'] = '%' . $q . '%';
  }

  $whereSql = "WHERE " . implode(" AND ", $where);

  $countStmt = $pdo->prepare("SELECT COUNT(*) FROM ops_work_orders wo $whereSql");
  $countStmt->execute($params);
  $total = (int)$countStmt->fetchColumn();

  $sql = "
    SELECT
      wo.id,
      wo.status,
      wo.priority,
      wo.customer_name,
      wo.external_order_id,
      wo.external_order_number,
      wo.serial_number,
      wo.scheduled_date,
      wo.due_date,
      wo.updated_at,
      s.service_name
    FROM ops_work_orders wo
    LEFT JOIN ops_services s ON s.id = wo.service_id
    $whereSql
    ORDER BY wo.created_at DESC
    LIMIT :limit OFFSET :offset
  ";
  $stmt = $pdo->prepare($sql);
  foreach ($params as $k => $v) $stmt->bindValue($k, $v);
  $stmt->bindValue(':limit', $perPage, PDO::PARAM_INT);
  $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
  $stmt->execute();
  $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

  $totalPages = max(1, (int)ceil($total / $perPage));

} catch (Throwable $e) {
  error_log("[operations] workorders_list.php error: {$e->getMessage()} in {$e->getFile()}:{$e->getLine()}");
  http_response_code(500);
  require_once __DIR__ . '/../includes/header.php';
  require_once __DIR__ . '/_subnav.php';
  echo "<div class='container-fluid'><div class='so-card'><div class='so-card-body'>Work Orders failed to load. Check logs.</div></div></div>";
  require_once __DIR__ . '/../includes/footer.php';
  exit;
}

require_once __DIR__ . '/../includes/header.php';
require_once __DIR__ . '/_subnav.php';

function pill_class(string $st): string {
  return match ($st) {
    'blocked' => 'red',
    'in_progress' => 'orange',
    'scheduled' => 'blue',
    'completed' => 'blue',
    default => 'gray',
  };
}
?>
<div class="container-fluid">
  <div class="row">
    <div class="col-12">
      <div class="so-page-title">Work Orders</div>
      <div class="so-page-sub">Search, filter, and manage service installs.</div>
    </div>
  </div>

  <div class="row g-3 mb-3">
    <div class="col-12">
      <div class="so-card">
        <div class="so-card-h">
          <h5 class="so-card-title">
            <i class="fa-solid fa-filter me-2" style="color:#cf1f1f"></i>Filters
          </h5>
          <a class="so-btn btn btn-sm" href="<?= h(app_path('operations/workorder_view.php?id=new')) ?>">
            <i class="fa-solid fa-plus me-2"></i>New
          </a>
        </div>
        <div class="so-card-body">
          <form method="get" class="row g-2 align-items-end">
            <div class="col-12 col-lg-5">
              <label class="form-label small text-muted mb-1">Search</label>
              <input class="form-control" type="text" name="q" value="<?= h($q) ?>" placeholder="Customer, order #, serial, summary...">
            </div>

            <div class="col-12 col-sm-6 col-lg-3">
              <label class="form-label small text-muted mb-1">Status</label>
              <select class="form-select" name="status">
                <?php foreach ($statuses as $val => $label): ?>
                  <option value="<?= h($val) ?>" <?= $status === $val ? 'selected' : '' ?>><?= h($label) ?></option>
                <?php endforeach; ?>
              </select>
            </div>

            <div class="col-12 col-sm-6 col-lg-3">
              <label class="form-label small text-muted mb-1">Service</label>
              <select class="form-select" name="service_id">
                <option value="0">All Services</option>
                <?php foreach ($services as $s): ?>
                  <option value="<?= (int)$s['id'] ?>" <?= $serviceId === (int)$s['id'] ? 'selected' : '' ?>>
                    <?= h((string)$s['service_name']) ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>

            <div class="col-12 col-lg-1 d-grid">
              <button class="so-btn btn btn-sm" type="submit">Go</button>
            </div>
          </form>
        </div>
      </div>
    </div>
  </div>

  <div class="row g-3">
    <div class="col-12">
      <div class="so-card">
        <div class="so-card-h">
          <h5 class="so-card-title">
            <i class="fa-solid fa-clipboard-list me-2" style="color:#cf1f1f"></i>Queue
          </h5>
          <div class="text-muted small">Total: <?= (int)$total ?></div>
        </div>
        <div class="so-card-body">
          <div class="table-responsive">
            <table class="table so-table align-middle">
              <thead>
                <tr>
                  <th>ID</th>
                  <th>Service</th>
                  <th>Status</th>
                  <th class="text-end">Priority</th>
                  <th>Customer</th>
                  <th>Order</th>
                  <th>Serial</th>
                  <th>Scheduled</th>
                  <th>Due</th>
                  <th></th>
                </tr>
              </thead>
              <tbody>
              <?php if (!$rows): ?>
                <tr><td colspan="10" class="text-center text-muted py-4">No work orders found.</td></tr>
              <?php else: ?>
                <?php foreach ($rows as $r): ?>
                  <tr>
                    <td><strong><?= (int)$r['id'] ?></strong></td>
                    <td><?= h((string)($r['service_name'] ?? '')) ?></td>
                    <td><span class="so-pill <?= h(pill_class((string)$r['status'])) ?>"><?= h(ucwords(str_replace('_',' ', (string)$r['status']))) ?></span></td>
                    <td class="text-end"><?= (int)$r['priority'] ?></td>
                    <td><?= h((string)($r['customer_name'] ?? '')) ?></td>
                    <td><div class="text-muted small"><?= h((string)($r['external_order_number'] ?: $r['external_order_id'] ?: '')) ?></div></td>
                    <td><div class="text-muted small"><?= h((string)($r['serial_number'] ?? '')) ?></div></td>
                    <td><?= h((string)($r['scheduled_date'] ?? '')) ?></td>
                    <td><?= h((string)($r['due_date'] ?? '')) ?></td>
                    <td class="text-end">
                      <a href="<?= h(app_path('operations/workorder_view.php?id='.(int)$r['id'])) ?>" class="text-muted">
                        <i class="fa-solid fa-chevron-right"></i>
                      </a>
                    </td>
                  </tr>
                <?php endforeach; ?>
              <?php endif; ?>
              </tbody>
            </table>
          </div>

          <?php if ($totalPages > 1): ?>
            <div class="d-flex justify-content-between align-items-center mt-3">
              <div class="text-muted small">
                Showing <?= h((string)min($total, $offset + 1)) ?>–<?= h((string)min($total, $offset + $perPage)) ?> of <?= h((string)$total) ?>
              </div>
              <div class="d-flex gap-2">
                <a class="so-btn btn btn-sm <?= $page <= 1 ? 'disabled' : '' ?>" href="<?= $page <= 1 ? '#' : ops_qs(['page' => $page - 1]) ?>">Prev</a>
                <a class="so-btn btn btn-sm <?= $page >= $totalPages ? 'disabled' : '' ?>" href="<?= $page >= $totalPages ? '#' : ops_qs(['page' => $page + 1]) ?>">Next</a>
              </div>
            </div>
          <?php endif; ?>

        </div>
      </div>
    </div>
  </div>
</div>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
