<?php
// /app/operations/workorder_view.php
declare(strict_types=1);

require_once __DIR__ . '/_lib.php';

$pageTitle = 'Work Order';
$activeNav = 'operations';
$activeOps = 'workorders';

$pdo = db_safe();
$companyId = ops_require_company_scope();

if (!ops_can_view()) { http_response_code(403); exit('Forbidden'); }
$canManage = ops_can_manage();

$idRaw = (string)($_GET['id'] ?? '0');
$isNew = ($idRaw === 'new');
$id = $isNew ? 0 : (int)$idRaw;
if (!$isNew && $id <= 0) { http_response_code(400); exit('Invalid id'); }

$svcStmt = $pdo->prepare("
  SELECT id, service_name
  FROM ops_services
  WHERE company_id=:cid AND is_active=1
  ORDER BY service_name
");
$svcStmt->execute([':cid'=>$companyId]);
$services = $svcStmt->fetchAll(PDO::FETCH_ASSOC);

if (!$isNew) {
  $stmt = $pdo->prepare("
    SELECT wo.*, s.service_name
    FROM ops_work_orders wo
    LEFT JOIN ops_services s ON s.id = wo.service_id
    WHERE wo.id=:id AND wo.company_id=:cid
    LIMIT 1
  ");
  $stmt->execute([':id'=>$id, ':cid'=>$companyId]);
  $wo = $stmt->fetch(PDO::FETCH_ASSOC);
  if (!$wo) { http_response_code(404); exit('Work order not found'); }
} else {
  $wo = [
    'id'=>0,'service_id'=>0,'status'=>'new','priority'=>3,'source'=>'manual',
    'external_order_id'=>'','external_order_number'=>'','external_customer_id'=>'',
    'serial_number'=>'','customer_name'=>'','customer_phone'=>'','customer_email'=>'',
    'install_location'=>'','address_line1'=>'','address_line2'=>'','city'=>'','state'=>'','postal_code'=>'',
    'scheduled_date'=>null,'due_date'=>null,'completed_at'=>null,
    'summary'=>'','internal_notes'=>'','customer_notes'=>'','source_payload_json'=>null,
    'created_at'=>null,'updated_at'=>null,'service_name'=>'',
  ];
}

function pill_class(string $st): string {
  return match ($st) {
    'blocked' => 'red',
    'in_progress' => 'orange',
    'scheduled' => 'blue',
    'completed' => 'blue',
    default => 'gray',
  };
}

$title = $isNew ? 'New Work Order' : ('Work Order #' . (int)$wo['id']);

require_once __DIR__ . '/../includes/header.php';
require_once __DIR__ . '/_subnav.php';
?>
<div class="container-fluid">
  <div class="row">
    <div class="col-12">
      <div class="so-page-title"><?= h($title) ?></div>
      <div class="so-page-sub">
        <?php if (!$isNew): ?>
          Created <?= h(date('M j, Y g:i A', strtotime((string)$wo['created_at']))) ?>
          · Updated <?= h(date('M j, Y g:i A', strtotime((string)$wo['updated_at']))) ?>
          <?php if (!empty($wo['external_order_number']) || !empty($wo['external_order_id'])): ?>
            · External Order: <?= h((string)($wo['external_order_number'] ?: $wo['external_order_id'])) ?>
          <?php endif; ?>
        <?php else: ?>
          Create a new work order for any service.
        <?php endif; ?>
      </div>
    </div>
  </div>

  <form method="post" action="/app/operations/workorder_save.php" class="row g-3">
    <input type="hidden" name="csrf" value="<?=h(csrf_token())?>">
    <input type="hidden" name="id" value="<?= (int)$wo['id'] ?>">

    <div class="col-12 col-xl-8">
      <div class="so-card">
        <div class="so-card-h">
          <h5 class="so-card-title">
            <i class="fa-solid fa-file-signature me-2" style="color:#cf1f1f"></i>Details
          </h5>
          <span class="so-pill <?= h(pill_class((string)$wo['status'])) ?>"><?= h(ucwords(str_replace('_',' ', (string)$wo['status']))) ?></span>
        </div>

        <div class="so-card-body">
          <div class="row g-2">
            <div class="col-12 col-md-6">
              <label class="form-label small text-muted mb-1">Service</label>
              <select class="form-select" name="service_id" <?= $canManage ? '' : 'disabled' ?> required>
                <option value="">— Select Service —</option>
                <?php foreach ($services as $s): ?>
                  <option value="<?= (int)$s['id'] ?>" <?= (int)$wo['service_id']===(int)$s['id']?'selected':'' ?>><?=h((string)$s['service_name'])?></option>
                <?php endforeach; ?>
              </select>
              <?php if (!$canManage): ?><input type="hidden" name="service_id" value="<?= (int)$wo['service_id'] ?>"><?php endif; ?>
            </div>

            <div class="col-6 col-md-3">
              <label class="form-label small text-muted mb-1">Status</label>
              <select class="form-select" name="status" <?= $canManage ? '' : 'disabled' ?>>
                <?php foreach (['new','scheduled','in_progress','blocked','completed','canceled'] as $st): ?>
                  <option value="<?=$st?>" <?=$st===$wo['status']?'selected':''?>><?=h(ucwords(str_replace('_',' ', $st)))?></option>
                <?php endforeach; ?>
              </select>
              <?php if (!$canManage): ?><input type="hidden" name="status" value="<?=h((string)$wo['status'])?>"><?php endif; ?>
            </div>

            <div class="col-6 col-md-3">
              <label class="form-label small text-muted mb-1">Priority</label>
              <select class="form-select" name="priority" <?= $canManage ? '' : 'disabled' ?>>
                <?php for ($p=1;$p<=5;$p++): ?>
                  <option value="<?=$p?>" <?=$p==(int)$wo['priority']?'selected':''?>><?=$p?></option>
                <?php endfor; ?>
              </select>
              <?php if (!$canManage): ?><input type="hidden" name="priority" value="<?= (int)$wo['priority'] ?>"><?php endif; ?>
            </div>

            <div class="col-12">
              <label class="form-label small text-muted mb-1">Summary</label>
              <input class="form-control" name="summary" value="<?=h((string)$wo['summary'])?>" placeholder="Short description of the work order">
            </div>

            <div class="col-6 col-md-4">
              <label class="form-label small text-muted mb-1">Serial #</label>
              <input class="form-control" name="serial_number" value="<?=h((string)$wo['serial_number'])?>">
            </div>

            <div class="col-6 col-md-4">
              <label class="form-label small text-muted mb-1">Install Location</label>
              <input class="form-control" name="install_location" value="<?=h((string)$wo['install_location'])?>">
            </div>

            <div class="col-6 col-md-2">
              <label class="form-label small text-muted mb-1">Scheduled</label>
              <input type="date" class="form-control" name="scheduled_date" value="<?=h((string)($wo['scheduled_date'] ?? ''))?>" <?= $canManage ? '' : 'disabled' ?>>
              <?php if (!$canManage): ?><input type="hidden" name="scheduled_date" value="<?=h((string)($wo['scheduled_date'] ?? ''))?>"><?php endif; ?>
            </div>

            <div class="col-6 col-md-2">
              <label class="form-label small text-muted mb-1">Due</label>
              <input type="date" class="form-control" name="due_date" value="<?=h((string)($wo['due_date'] ?? ''))?>" <?= $canManage ? '' : 'disabled' ?>>
              <?php if (!$canManage): ?><input type="hidden" name="due_date" value="<?=h((string)($wo['due_date'] ?? ''))?>"><?php endif; ?>
            </div>

            <hr class="my-2">

            <div class="col-12 col-md-6">
              <label class="form-label small text-muted mb-1">Customer Name</label>
              <input class="form-control" name="customer_name" value="<?=h((string)$wo['customer_name'])?>">
            </div>
            <div class="col-6 col-md-3">
              <label class="form-label small text-muted mb-1">Phone</label>
              <input class="form-control" name="customer_phone" value="<?=h((string)$wo['customer_phone'])?>">
            </div>
            <div class="col-6 col-md-3">
              <label class="form-label small text-muted mb-1">Email</label>
              <input class="form-control" name="customer_email" value="<?=h((string)$wo['customer_email'])?>">
            </div>

            <div class="col-12">
              <label class="form-label small text-muted mb-1">Address</label>
              <input class="form-control mb-2" name="address_line1" value="<?=h((string)$wo['address_line1'])?>" placeholder="Street address">
              <input class="form-control" name="address_line2" value="<?=h((string)$wo['address_line2'])?>" placeholder="Apt, suite, etc (optional)">
            </div>

            <div class="col-6 col-md-4">
              <label class="form-label small text-muted mb-1">City</label>
              <input class="form-control" name="city" value="<?=h((string)$wo['city'])?>">
            </div>
            <div class="col-3 col-md-4">
              <label class="form-label small text-muted mb-1">State</label>
              <input class="form-control" name="state" value="<?=h((string)$wo['state'])?>">
            </div>
            <div class="col-3 col-md-4">
              <label class="form-label small text-muted mb-1">ZIP</label>
              <input class="form-control" name="postal_code" value="<?=h((string)$wo['postal_code'])?>">
            </div>

            <div class="col-12">
              <label class="form-label small text-muted mb-1">Internal Notes</label>
              <textarea class="form-control" name="internal_notes" rows="3"><?=h((string)$wo['internal_notes'])?></textarea>
            </div>

            <div class="col-12">
              <label class="form-label small text-muted mb-1">Customer Notes</label>
              <textarea class="form-control" name="customer_notes" rows="3"><?=h((string)$wo['customer_notes'])?></textarea>
            </div>
          </div>

          <div class="d-flex gap-2 mt-3">
            <button class="so-btn btn btn-sm" type="submit"><i class="fa-solid fa-floppy-disk me-2"></i><?= $isNew ? 'Create' : 'Save' ?></button>
            <a class="so-btn btn btn-sm" href="<?= h(app_path('operations/workorders_list.php')) ?>" style="border-color:rgba(18,20,24,0.10);background:#fff;color:#111827;">
              <i class="fa-solid fa-arrow-left me-2"></i>Back
            </a>
          </div>

        </div>
      </div>
    </div>

    <div class="col-12 col-xl-4">
      <div class="so-card">
        <div class="so-card-h">
          <h5 class="so-card-title">
            <i class="fa-solid fa-user-gear me-2" style="color:#cf1f1f"></i>Assignment
          </h5>
        </div>
        <div class="so-card-body">
          <div class="text-muted">Assignment UI will be enabled once assignee tables/actions are finalized.</div>
        </div>
      </div>

      <div class="so-card mt-3">
        <div class="so-card-h">
          <h5 class="so-card-title">
            <i class="fa-solid fa-clock-rotate-left me-2" style="color:#cf1f1f"></i>Activity
          </h5>
        </div>
        <div class="so-card-body">
          <div class="text-muted">Activity feed will render once events are enabled.</div>
        </div>
      </div>
    </div>
  </form>
</div>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
