-- 2026_01_21_operations.sql
-- ShedOffice Operations Module: multi-tenant work orders + services + addon mapping + webhook inbox

SET NAMES utf8mb4;
SET time_zone = '+00:00';

-- ---------------------------------------------------------------------------
-- Webhook keys per company (multi-tenant)
-- ---------------------------------------------------------------------------
CREATE TABLE IF NOT EXISTS ops_webhook_keys (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  company_id BIGINT UNSIGNED NOT NULL,
  label VARCHAR(100) NOT NULL,
  api_key CHAR(40) NOT NULL,
  is_enabled TINYINT(1) NOT NULL DEFAULT 1,
  created_by BIGINT UNSIGNED NULL,
  created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  last_used_at DATETIME NULL,
  last_used_ip VARCHAR(45) NULL,
  PRIMARY KEY (id),
  UNIQUE KEY uq_ops_webhook_keys_key (api_key),
  KEY idx_ops_webhook_keys_company (company_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ---------------------------------------------------------------------------
-- Services (Trades) per company
-- ---------------------------------------------------------------------------
CREATE TABLE IF NOT EXISTS ops_services (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  company_id BIGINT UNSIGNED NOT NULL,
  service_code VARCHAR(50) NOT NULL,
  service_name VARCHAR(120) NOT NULL,
  description TEXT NULL,
  default_priority TINYINT UNSIGNED NOT NULL DEFAULT 3, -- 1=urgent..5=low
  default_due_days SMALLINT UNSIGNED NOT NULL DEFAULT 7,
  default_assignee_user_id BIGINT UNSIGNED NULL,
  is_active TINYINT(1) NOT NULL DEFAULT 1,
  created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (id),
  UNIQUE KEY uq_ops_services_company_code (company_id, service_code),
  KEY idx_ops_services_company_active (company_id, is_active),
  KEY idx_ops_services_default_assignee (default_assignee_user_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ---------------------------------------------------------------------------
-- Add-on -> Service mapping (matches ShedSuite add-on names to services)
-- ---------------------------------------------------------------------------
CREATE TABLE IF NOT EXISTS ops_addon_service_map (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  company_id BIGINT UNSIGNED NOT NULL,
  service_id BIGINT UNSIGNED NOT NULL,
  match_type ENUM('exact','contains','regex') NOT NULL DEFAULT 'contains',
  match_value VARCHAR(255) NOT NULL,
  priority SMALLINT UNSIGNED NOT NULL DEFAULT 100, -- lower runs first
  is_active TINYINT(1) NOT NULL DEFAULT 1,
  created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (id),
  KEY idx_ops_map_company_active (company_id, is_active, priority),
  KEY idx_ops_map_service (service_id),
  CONSTRAINT fk_ops_map_service FOREIGN KEY (service_id) REFERENCES ops_services(id)
    ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ---------------------------------------------------------------------------
-- Webhook inbox (raw payload storage; processed by cron)
-- ---------------------------------------------------------------------------
CREATE TABLE IF NOT EXISTS ops_webhook_inbox (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  company_id BIGINT UNSIGNED NOT NULL,
  provider VARCHAR(50) NOT NULL DEFAULT 'shedsuite',
  external_event_id VARCHAR(120) NULL,
  external_order_id VARCHAR(120) NULL,
  received_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  source_ip VARCHAR(45) NULL,
  headers_json JSON NULL,
  payload_json JSON NOT NULL,
  status ENUM('pending','processed','failed') NOT NULL DEFAULT 'pending',
  processed_at DATETIME NULL,
  error_message VARCHAR(255) NULL,
  PRIMARY KEY (id),
  KEY idx_ops_inbox_company_status (company_id, status, received_at),
  KEY idx_ops_inbox_external_order (company_id, external_order_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ---------------------------------------------------------------------------
-- Work Orders
-- ---------------------------------------------------------------------------
CREATE TABLE IF NOT EXISTS ops_work_orders (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  company_id BIGINT UNSIGNED NOT NULL,

  service_id BIGINT UNSIGNED NOT NULL,
  status ENUM('new','scheduled','in_progress','blocked','completed','canceled') NOT NULL DEFAULT 'new',
  priority TINYINT UNSIGNED NOT NULL DEFAULT 3, -- 1..5
  source VARCHAR(50) NOT NULL DEFAULT 'shedsuite',
  external_order_id VARCHAR(120) NULL,
  external_order_number VARCHAR(120) NULL,
  external_customer_id VARCHAR(120) NULL,

  serial_number VARCHAR(60) NULL,
  customer_name VARCHAR(140) NULL,
  customer_phone VARCHAR(50) NULL,
  customer_email VARCHAR(120) NULL,

  install_location VARCHAR(100) NULL,
  address_line1 VARCHAR(200) NULL,
  address_line2 VARCHAR(200) NULL,
  city VARCHAR(80) NULL,
  state VARCHAR(30) NULL,
  postal_code VARCHAR(20) NULL,

  scheduled_date DATE NULL,
  due_date DATE NULL,
  completed_at DATETIME NULL,

  summary VARCHAR(255) NULL,
  internal_notes MEDIUMTEXT NULL,
  customer_notes MEDIUMTEXT NULL,

  source_payload_json JSON NULL,

  created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

  PRIMARY KEY (id),
  -- idempotency for webhooks: at most one work order per service per external order
  UNIQUE KEY uq_ops_wo_company_external_service (company_id, external_order_id, service_id),
  KEY idx_ops_wo_company_status (company_id, status, created_at),
  KEY idx_ops_wo_company_service (company_id, service_id, created_at),
  KEY idx_ops_wo_assigned_date (company_id, scheduled_date),
  CONSTRAINT fk_ops_wo_service FOREIGN KEY (service_id) REFERENCES ops_services(id)
    ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ---------------------------------------------------------------------------
-- Work order assignees (supports multiple installers/teams)
-- ---------------------------------------------------------------------------
CREATE TABLE IF NOT EXISTS ops_work_order_assignees (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  work_order_id BIGINT UNSIGNED NOT NULL,
  user_id BIGINT UNSIGNED NOT NULL,
  role ENUM('lead','tech','helper') NOT NULL DEFAULT 'tech',
  assigned_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (id),
  UNIQUE KEY uq_ops_wo_assignee (work_order_id, user_id),
  KEY idx_ops_woa_user (user_id),
  CONSTRAINT fk_ops_woa_workorder FOREIGN KEY (work_order_id) REFERENCES ops_work_orders(id)
    ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ---------------------------------------------------------------------------
-- Work order tasks / checklist
-- ---------------------------------------------------------------------------
CREATE TABLE IF NOT EXISTS ops_work_order_tasks (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  work_order_id BIGINT UNSIGNED NOT NULL,
  title VARCHAR(180) NOT NULL,
  status ENUM('open','done','blocked') NOT NULL DEFAULT 'open',
  sort_order SMALLINT UNSIGNED NOT NULL DEFAULT 100,
  assigned_to BIGINT UNSIGNED NULL,
  due_date DATE NULL,
  completed_at DATETIME NULL,
  created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (id),
  KEY idx_ops_tasks_workorder (work_order_id, sort_order),
  CONSTRAINT fk_ops_tasks_workorder FOREIGN KEY (work_order_id) REFERENCES ops_work_orders(id)
    ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ---------------------------------------------------------------------------
-- Audit/events
-- ---------------------------------------------------------------------------
CREATE TABLE IF NOT EXISTS ops_work_order_events (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  work_order_id BIGINT UNSIGNED NOT NULL,
  actor_user_id BIGINT UNSIGNED NULL,
  event_type VARCHAR(60) NOT NULL,
  message VARCHAR(255) NULL,
  meta_json JSON NULL,
  created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (id),
  KEY idx_ops_events_workorder (work_order_id, created_at),
  CONSTRAINT fk_ops_events_workorder FOREIGN KEY (work_order_id) REFERENCES ops_work_orders(id)
    ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ---------------------------------------------------------------------------
-- Seed default services + mappings for EXISTING companies.
-- This is safe to run multiple times; it inserts only when missing.
-- Assumes you have a `companies` table with id. If not, ignore/adjust.
-- ---------------------------------------------------------------------------
-- You can comment out this whole block if your company table differs.

-- Create defaults for every company found in `companies`.
INSERT INTO ops_services (company_id, service_code, service_name, description, default_priority, default_due_days, is_active)
SELECT c.id, 'electrical', 'Electrical Installation', 'Electrical package installation work order.', 3, 7, 1
  FROM companies c
 WHERE NOT EXISTS (SELECT 1 FROM ops_services s WHERE s.company_id=c.id AND s.service_code='electrical');

INSERT INTO ops_services (company_id, service_code, service_name, description, default_priority, default_due_days, is_active)
SELECT c.id, 'minisplit', 'Mini-Split Installation', 'Mini-split install work order.', 3, 7, 1
  FROM companies c
 WHERE NOT EXISTS (SELECT 1 FROM ops_services s WHERE s.company_id=c.id AND s.service_code='minisplit');

INSERT INTO ops_services (company_id, service_code, service_name, description, default_priority, default_due_days, is_active)
SELECT c.id, 'spray_foam', 'Spray Foam Installation', 'Spray foam install work order.', 3, 10, 1
  FROM companies c
 WHERE NOT EXISTS (SELECT 1 FROM ops_services s WHERE s.company_id=c.id AND s.service_code='spray_foam');

INSERT INTO ops_services (company_id, service_code, service_name, description, default_priority, default_due_days, is_active)
SELECT c.id, 'interior_finish', 'Interior Finish Out', 'Interior finish-out work order.', 4, 14, 1
  FROM companies c
 WHERE NOT EXISTS (SELECT 1 FROM ops_services s WHERE s.company_id=c.id AND s.service_code='interior_finish');

-- Mappings (contains match by default)
INSERT INTO ops_addon_service_map (company_id, service_id, match_type, match_value, priority, is_active)
SELECT s.company_id, s.id, 'contains', 'Electrical Package', 10, 1
  FROM ops_services s
 WHERE s.service_code='electrical'
   AND NOT EXISTS (
     SELECT 1 FROM ops_addon_service_map m
      WHERE m.company_id=s.company_id AND m.service_id=s.id AND m.match_value='Electrical Package'
   );

INSERT INTO ops_addon_service_map (company_id, service_id, match_type, match_value, priority, is_active)
SELECT s.company_id, s.id, 'contains', 'Mini Split', 10, 1
  FROM ops_services s
 WHERE s.service_code='minisplit'
   AND NOT EXISTS (
     SELECT 1 FROM ops_addon_service_map m
      WHERE m.company_id=s.company_id AND m.service_id=s.id AND m.match_value='Mini Split'
   );

INSERT INTO ops_addon_service_map (company_id, service_id, match_type, match_value, priority, is_active)
SELECT s.company_id, s.id, 'contains', 'Spray Foam', 10, 1
  FROM ops_services s
 WHERE s.service_code='spray_foam'
   AND NOT EXISTS (
     SELECT 1 FROM ops_addon_service_map m
      WHERE m.company_id=s.company_id AND m.service_id=s.id AND m.match_value='Spray Foam'
   );

INSERT INTO ops_addon_service_map (company_id, service_id, match_type, match_value, priority, is_active)
SELECT s.company_id, s.id, 'contains', 'Interior Finish', 10, 1
  FROM ops_services s
 WHERE s.service_code='interior_finish'
   AND NOT EXISTS (
     SELECT 1 FROM ops_addon_service_map m
      WHERE m.company_id=s.company_id AND m.service_id=s.id AND m.match_value='Interior Finish'
   );
