<?php
// /app/operations/includes/bootstrap.php
declare(strict_types=1);

/**
 * Operations module bootstrap
 * - Loads core app config from /app/includes/config.php
 * - Logs PHP errors/exceptions to /app/operations/logs/php_errors.log
 *
 * Notes:
 * - display_errors stays OFF (no leakage to end users)
 * - Ensure /app/operations/logs is writable by PHP.
 */

$coreConfig = __DIR__ . '/../../includes/config.php';
if (!is_file($coreConfig)) {
  http_response_code(500);
  echo 'Missing required config.';
  exit;
}
require_once $coreConfig;

$logDir = __DIR__ . '/../logs';
if (!is_dir($logDir)) @mkdir($logDir, 0775, true);
$logFile = $logDir . '/php_errors.log';

@ini_set('display_errors', '0');
@ini_set('log_errors', '1');
@ini_set('error_log', $logFile);
error_reporting(E_ALL);

set_error_handler(static function (int $severity, string $message, string $file, int $line): bool {
  if (!(error_reporting() & $severity)) return false;
  throw new ErrorException($message, 0, $severity, $file, $line);
});

set_exception_handler(static function (Throwable $e) use ($logFile): void {
  $ts  = gmdate('c');
  $uri = $_SERVER['REQUEST_URI'] ?? '(cli)';
  $ip  = $_SERVER['REMOTE_ADDR'] ?? '(unknown)';
  $msg = "[{$ts}] [operations] {$uri} {$ip}
"
       . get_class($e) . ': ' . $e->getMessage() . "
"
       . 'in ' . $e->getFile() . ':' . $e->getLine() . "
"
       . $e->getTraceAsString() . "

";
  @file_put_contents($logFile, $msg, FILE_APPEND);
  @error_log($msg);
  http_response_code(500);
  echo 'Internal Server Error';
  exit;
});
