<?php
// /app/operations/_lib.php
declare(strict_types=1);

/**
 * Operations module shared helpers.
 *
 * IMPORTANT:
 * - Do NOT redeclare helpers like h() that are already defined in /app/includes/config.php
 * - Always rely on core ShedOffice framework includes for auth/acl/db.
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/acl.php';
require_once __DIR__ . '/../includes/db.php';

require_login();

/** @return PDO */
function db_safe(): PDO {
  /** @var PDO $pdo */
  $pdo = db();
  return $pdo;
}

// Ensure ACL is loaded (roles + permissions cached in session)
if (function_exists('acl_bootstrap')) {
  acl_bootstrap(db_safe());
}

/**
 * Determine active company scope for the current session.
 * - In company tenant mode: company_id = current_tenant_id()
 * - In platform/admin mode with active company: company_id = active_company_tenant_id()
 */
function ops_company_id(): int {
  $type = function_exists('current_tenant_type') ? (string) current_tenant_type() : 'company';

  if ($type === 'company' && function_exists('current_tenant_id')) {
    return (int) current_tenant_id();
  }

  if (function_exists('active_company_tenant_id')) {
    $cid = (int) (active_company_tenant_id() ?: 0);
    if ($cid > 0) return $cid;
  }

  // Last resort fallback
  if (function_exists('current_tenant_id')) return (int) current_tenant_id();
  return 0;
}

function ops_require_company_scope(): int {
  $cid = ops_company_id();
  if ($cid <= 0) {
    http_response_code(403);
    echo "<div style='padding:16px;font-family:system-ui'>No active company context. Please select a company tenant.</div>";
    exit;
  }
  return $cid;
}

/** Permission gates */
function ops_can_view(): bool {
  return user_has_role('company_admin')
      || user_has_role('manager')
      || user_can('operations.view')
      || user_can('workorders.view');
}

function ops_can_manage(): bool {
  return user_has_role('company_admin')
      || user_has_role('manager')
      || user_can('workorders.edit')
      || user_can('operations.manage');
}

function ops_can_configure(): bool {
  return user_has_role('company_admin')
      || user_can('operations.configure')
      || user_can('services.manage');
}

function ops_can_webhooks(): bool {
  return user_has_role('company_admin')
      || user_can('webhooks.manage')
      || user_can('operations.configure');
}

/**
 * Simple JSON response helper for AJAX endpoints.
 */
function ops_json(array $payload, int $status = 200): void {
  http_response_code($status);
  header('Content-Type: application/json; charset=utf-8');
  echo json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
  exit;
}
