<?php
// /app/operations/_bootstrap.php
declare(strict_types=1);

/**
 * Minimal, early error logging for Operations module.
 * This file must be required as the FIRST thing in every Operations entrypoint.
 *
 * Logs to: /app/operations/logs/php_errors.log
 */

$opsLogDir = __DIR__ . '/logs';
if (!is_dir($opsLogDir)) {
  @mkdir($opsLogDir, 0775, true);
}
$opsLogFile = $opsLogDir . '/php_errors.log';

@ini_set('display_errors', '0');
@ini_set('log_errors', '1');
@ini_set('error_log', $opsLogFile);
error_reporting(E_ALL);

// Log fatal errors (parse errors won't reach here, but most fatals will)
register_shutdown_function(static function () use ($opsLogFile): void {
  $e = error_get_last();
  if (!$e) return;
  $fatalTypes = [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR, E_USER_ERROR];
  if (!in_array($e['type'] ?? 0, $fatalTypes, true)) return;

  $ts = gmdate('c');
  $msg = "[{$ts}] [operations] FATAL: {$e['message']} in {$e['file']}:{$e['line']}
";
  @file_put_contents($opsLogFile, $msg, FILE_APPEND);
});

// Uncaught exceptions
set_exception_handler(static function (Throwable $ex) use ($opsLogFile): void {
  $ts = gmdate('c');
  $uri = $_SERVER['REQUEST_URI'] ?? '(cli)';
  $msg = "[{$ts}] [operations] EXCEPTION at {$uri}
"
       . get_class($ex) . ": " . $ex->getMessage() . "
"
       . "in " . $ex->getFile() . ":" . $ex->getLine() . "
"
       . $ex->getTraceAsString() . "

";
  @file_put_contents($opsLogFile, $msg, FILE_APPEND);
  http_response_code(500);
  echo "Internal Server Error";
  exit;
});

// Ensure core app config is loaded (DB constants, paths, helpers)
$coreConfig = __DIR__ . '/../includes/config.php';
if (is_file($coreConfig)) {
  require_once $coreConfig;
}
