<?php
// /app/includes/scope.php
declare(strict_types=1);

require_once __DIR__ . '/auth.php';
require_once __DIR__ . '/acl.php';

/**
 * Returns the company tenant context ID for data queries.
 * - Company users: their tenant_id
 * - RTO users: active_company_tenant_id (must be selected)
 */
function company_context_id(): int {
  $tt = current_tenant_type();
  if ($tt === 'rto') return (int)active_company_tenant_id();
  return (int)current_tenant_id();
}

/**
 * Adds repo visibility constraints for current user.
 * - Always: r.tenant_id = company_context_id()
 * - If RTO: r.rto_tenant_id = current tenant_id
 * - If driver role and no repos.view_all: assigned only
 *
 * NOTE: expects repos table alias "r"
 */
function apply_repo_scope(array &$where, array &$params): void {
  $tenantType = current_tenant_type();
  $companyTid = company_context_id();

  $where[] = "r.tenant_id = :repo_company_tid";
  $params[':repo_company_tid'] = $companyTid;

  if ($tenantType === 'rto') {
    $where[] = "r.rto_tenant_id = :repo_rto_tid";
    $params[':repo_rto_tid'] = (int)current_tenant_id();
    return;
  }

  if (user_has_role('driver') && !user_can('repos.view_all')) {
    $where[] = "r.assigned_driver_user_id = :repo_me";
    $params[':repo_me'] = (int)current_user_id();
  }
}

/**
 * Adds repair visibility constraints for current user.
 * - Always: rp.tenant_id = company_context_id()
 * - If RTO: rp.rto_tenant_id = current tenant_id
 * - If repair_tech role and no repairs.view_all: assigned only
 *
 * NOTE: expects repairs table alias "rp"
 */
function apply_repair_scope(array &$where, array &$params): void {
  $tenantType = current_tenant_type();
  $companyTid = company_context_id();

  $where[] = "rp.tenant_id = :rep_company_tid";
  $params[':rep_company_tid'] = $companyTid;

  if ($tenantType === 'rto') {
    $where[] = "rp.rto_tenant_id = :rep_rto_tid";
    $params[':rep_rto_tid'] = (int)current_tenant_id();
    return;
  }

  if (user_has_role('repair_tech') && !user_can('repairs.view_all')) {
    $where[] = "rp.assigned_tech_user_id = :rep_me";
    $params[':rep_me'] = (int)current_user_id();
  }
}
