<?php
// /app/includes/bunny.php
declare(strict_types=1);

require_once __DIR__ . '/config.php';

/**
 * Bunny Storage upload via HTTP PUT.
 * Returns array: [storage_key, public_url, content_type, bytes]
 */
function bunny_upload_file(string $storageKey, string $localTmpPath, ?string $contentType = null): array {
  if (!is_file($localTmpPath)) {
    throw new RuntimeException("Local file not found for upload.");
  }

  $contentType = $contentType ?: bunny_guess_mime($storageKey);

  $url = 'https://' . BUNNY_STORAGE_HOST . '/' . rawurlencode(BUNNY_STORAGE_ZONE) . '/' . bunny_encode_path($storageKey);

  $fh = fopen($localTmpPath, 'rb');
  if ($fh === false) {
    throw new RuntimeException("Failed to open local file.");
  }

  $size = filesize($localTmpPath);
  if ($size === false) $size = 0;

  $ch = curl_init($url);
  if ($ch === false) {
    fclose($fh);
    throw new RuntimeException("Failed to initialize cURL.");
  }

  curl_setopt_array($ch, [
    CURLOPT_CUSTOMREQUEST => 'PUT',
    CURLOPT_UPLOAD => true,
    CURLOPT_INFILE => $fh,
    CURLOPT_INFILESIZE => $size,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_HEADER => true,
    CURLOPT_HTTPHEADER => [
      'AccessKey: ' . BUNNY_STORAGE_ACCESS_KEY,
      'Content-Type: ' . $contentType,
      'Accept: */*',
    ],
    CURLOPT_CONNECTTIMEOUT => 20,
    CURLOPT_TIMEOUT => 120,
  ]);

  $resp = curl_exec($ch);
  $err  = curl_error($ch);
  $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);

  curl_close($ch);
  fclose($fh);

  if ($resp === false) {
    throw new RuntimeException("Bunny upload failed: {$err}");
  }

  // Bunny returns 201/200 on success typically
  if ($code < 200 || $code >= 300) {
    // Include a small snippet for debugging
    $snippet = substr($resp, 0, 300);
    throw new RuntimeException("Bunny upload HTTP {$code}. Response: {$snippet}");
  }

  $publicUrl = rtrim(BUNNY_PULL_BASE_URL, '/') . '/' . ltrim($storageKey, '/');

  return [
    'storage_key' => $storageKey,
    'public_url' => $publicUrl,
    'content_type' => $contentType,
    'bytes' => $size,
  ];
}

function bunny_guess_mime(string $path): string {
  $ext = strtolower(pathinfo($path, PATHINFO_EXTENSION));
  return match ($ext) {
    'jpg', 'jpeg' => 'image/jpeg',
    'png' => 'image/png',
    'webp' => 'image/webp',
    'gif' => 'image/gif',
    'pdf' => 'application/pdf',
    default => 'application/octet-stream',
  };
}

/**
 * Encode each path segment safely (preserve slashes).
 */
function bunny_encode_path(string $path): string {
  $path = ltrim($path, '/');
  $parts = explode('/', $path);
  $parts = array_map(static fn($p) => rawurlencode($p), $parts);
  return implode('/', $parts);
}

/**
 * Validate image upload and return normalized info.
 * Returns: ['tmp' => ..., 'ext' => ..., 'size' => ...]
 */
function bunny_require_uploaded_image(string $inputName, array $allowedExts = ['jpg','jpeg','png','webp']): array {
  if (empty($_FILES[$inputName]) || !isset($_FILES[$inputName]['tmp_name'])) {
    throw new RuntimeException("Missing required upload: {$inputName}");
  }

  $f = $_FILES[$inputName];

  if ((int)$f['error'] !== UPLOAD_ERR_OK) {
    throw new RuntimeException("Upload error for {$inputName}: " . (string)$f['error']);
  }

  $tmp = (string)$f['tmp_name'];
  if (!is_uploaded_file($tmp)) {
    throw new RuntimeException("Invalid upload for {$inputName}");
  }

  $name = (string)$f['name'];
  $ext = strtolower(pathinfo($name, PATHINFO_EXTENSION));
  if (!in_array($ext, $allowedExts, true)) {
    throw new RuntimeException("Invalid file type for {$inputName} (.$ext)");
  }

  $size = (int)($f['size'] ?? 0);

  return ['tmp' => $tmp, 'ext' => $ext, 'size' => $size];
}

/**
 * Handle multiple image uploads (input name like corner_photos[]).
 * Returns array of tmp paths/exts.
 */
function bunny_optional_multi_images(string $inputName, array $allowedExts = ['jpg','jpeg','png','webp'], int $maxFiles = 10): array {
  if (empty($_FILES[$inputName]) || !is_array($_FILES[$inputName]['name'])) return [];

  $out = [];
  $count = count($_FILES[$inputName]['name']);
  $count = min($count, $maxFiles);

  for ($i=0; $i<$count; $i++) {
    if ((int)$_FILES[$inputName]['error'][$i] !== UPLOAD_ERR_OK) continue;

    $tmp = (string)$_FILES[$inputName]['tmp_name'][$i];
    if (!is_uploaded_file($tmp)) continue;

    $name = (string)$_FILES[$inputName]['name'][$i];
    $ext = strtolower(pathinfo($name, PATHINFO_EXTENSION));
    if (!in_array($ext, $allowedExts, true)) continue;

    $out[] = ['tmp' => $tmp, 'ext' => $ext];
  }

  return $out;
}

/**
 * Generate a safe storage key for repo uploads.
 * Example: repos/1/123/building_ab12cd34ef.jpg
 */
function bunny_repo_storage_key(int $companyTenantId, int $repoId, string $typePrefix, string $ext): string {
  $rand = bin2hex(random_bytes(10));
  $ext = strtolower($ext);
  return "repos/{$companyTenantId}/{$repoId}/{$typePrefix}_{$rand}.{$ext}";
}
