<?php
// /app/gps/gps_list.php
declare(strict_types=1);

require_once __DIR__ . '/../includes/auth.php';
require_company_context_if_rto();

require_once __DIR__ . '/../includes/acl.php';
require_once __DIR__ . '/../includes/scope.php';

$pageTitle = 'GPS';
$activeNav = 'gps';

$pdo = db();
acl_bootstrap($pdo);

$user = current_user();

// Permission gate (matches header toggles)
$isCompanyAdmin = user_has_role('company_admin');
$isManager      = user_has_role('manager');
if (!($isCompanyAdmin || $isManager || user_can('gps.view'))) {
  http_response_code(403);
  exit('Forbidden');
}

$tenantId = company_context_id();

/**
 * Latest ping join + current assignment.
 * - current assignment: gps_assignments.unassigned_at IS NULL
 * - latest ping: max(pinged_at) per device
 */
$sql = "
SELECT
  d.id,
  d.device_name,
  d.provider,
  d.provider_device_id,
  d.status,
  d.created_at,
  d.updated_at,

  a.asset_id,
  a.assigned_at,

  p.lat,
  p.lng,
  p.speed_mph,
  p.heading_deg,
  p.pinged_at

FROM gps_devices d
LEFT JOIN gps_assignments a
  ON a.tenant_id = d.tenant_id
 AND a.gps_device_id = d.id
 AND a.unassigned_at IS NULL
LEFT JOIN gps_pings p
  ON p.tenant_id = d.tenant_id
 AND p.gps_device_id = d.id
 AND p.pinged_at = (
    SELECT MAX(p2.pinged_at)
    FROM gps_pings p2
    WHERE p2.tenant_id = d.tenant_id
      AND p2.gps_device_id = d.id
 )
WHERE d.tenant_id = :tid
ORDER BY d.id DESC
";
$stmt = $pdo->prepare($sql);
$stmt->execute([':tid' => $tenantId]);
$rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Metrics
$totalDevices = count($rows);
$assignedCnt = 0;
$ping24Cnt = 0;
$offlineCnt = 0;

$now = time();
foreach ($rows as $r) {
  if (!empty($r['asset_id'])) $assignedCnt++;
  $pingedAt = !empty($r['pinged_at']) ? strtotime((string)$r['pinged_at']) : 0;
  if ($pingedAt > 0 && ($now - $pingedAt) <= 86400) $ping24Cnt++;
  // Offline heuristic: status != active OR last ping older than 24h
  $isOffline = ((string)($r['status'] ?? '') !== 'active') || ($pingedAt > 0 && ($now - $pingedAt) > 86400) || ($pingedAt === 0);
  if ($isOffline) $offlineCnt++;
}

function so_status_pill(string $status): array {
  $s = strtolower(trim($status));
  return match ($s) {
    'active' => ['blue', 'Active'],
    'inactive' => ['gray', 'Inactive'],
    'suspended' => ['orange', 'Suspended'],
    'lost' => ['red', 'Lost'],
    default => ['gray', $status ?: 'Unknown'],
  };
}

require_once __DIR__ . '/../includes/header.php';
?>
<div class="container-fluid">

  <div class="row">
    <div class="col-12">
      <div class="so-page-title">GPS</div>
      <div class="so-page-sub">Manage GPS devices, assignments, and live locations.</div>
    </div>
  </div>

  <!-- TOP STAT CARDS -->
  <div class="row g-3 mb-3">
    <div class="col-12 col-md-6 col-xl-3">
      <div class="so-card so-stat">
        <div class="so-stat-ico"><i class="fa-solid fa-satellite-dish"></i></div>
        <div class="flex-grow-1">
          <h6 class="so-stat-title mb-0">Devices</h6>
          <div class="so-stat-line">Total devices</div>
          <div class="so-stat-value"><?= (int)$totalDevices ?></div>
          <div class="so-stat-meta">Tenant ID: <?= (int)$tenantId ?></div>
        </div>
      </div>
    </div>

    <div class="col-12 col-md-6 col-xl-3">
      <div class="so-card so-stat">
        <div class="so-stat-ico"><i class="fa-solid fa-link"></i></div>
        <div class="flex-grow-1">
          <h6 class="so-stat-title mb-0">Assigned</h6>
          <div class="so-stat-line">Attached to assets</div>
          <div class="so-stat-value"><?= (int)$assignedCnt ?></div>
          <div class="so-stat-meta"><?= (int)max(0, $totalDevices - $assignedCnt) ?> unassigned</div>
        </div>
      </div>
    </div>

    <div class="col-12 col-md-6 col-xl-3">
      <div class="so-card so-stat">
        <div class="so-stat-ico"><i class="fa-solid fa-signal"></i></div>
        <div class="flex-grow-1">
          <h6 class="so-stat-title mb-0">Pinged (24h)</h6>
          <div class="so-stat-line">Recent activity</div>
          <div class="so-stat-value"><?= (int)$ping24Cnt ?></div>
          <div class="so-stat-meta">Last 24 hours</div>
        </div>
      </div>
    </div>

    <div class="col-12 col-md-6 col-xl-3">
      <div class="so-card so-stat">
        <div class="so-stat-ico"><i class="fa-solid fa-triangle-exclamation"></i></div>
        <div class="flex-grow-1">
          <h6 class="so-stat-title mb-0">Offline</h6>
          <div class="so-stat-line">Needs attention</div>
          <div class="so-stat-value"><?= (int)$offlineCnt ?></div>
          <div class="so-stat-meta">Status or ping aged</div>
        </div>
      </div>
    </div>
  </div>

  <!-- DEVICES TABLE -->
  <div class="row g-3 mb-3">
    <div class="col-12 col-xl-7">
      <div class="so-card">
        <div class="so-card-h">
          <h5 class="so-card-title">
            <i class="fa-solid fa-satellite me-2" style="color:#cf1f1f"></i>Devices
          </h5>
          <a class="so-btn btn btn-sm" href="<?= app_path('gps/gps_add.php') ?>">
            <i class="fa-solid fa-plus me-2"></i>Add Device
          </a>
        </div>
        <div class="so-card-body">
          <div class="table-responsive">
            <table class="table so-table align-middle">
              <thead>
                <tr>
                  <th>Device</th>
                  <th>Provider</th>
                  <th>Assignment</th>
                  <th>Last Ping</th>
                  <th class="text-end">Action</th>
                </tr>
              </thead>
              <tbody>
              <?php if (!$rows): ?>
                <tr><td colspan="5" class="text-center text-muted py-4">No GPS devices found.</td></tr>
              <?php else: ?>
                <?php foreach ($rows as $r): ?>
                  <?php [$pillCls,$pillLbl] = so_status_pill((string)($r['status'] ?? '')); ?>
                  <tr>
                    <td>
                      <strong><?= h($r['device_name'] ?? ('Device #' . (int)$r['id'])) ?></strong>
                      <div class="text-muted small">
                        ID: <?= (int)$r['id'] ?> · <span class="so-pill <?= h($pillCls) ?>"><?= h($pillLbl) ?></span>
                      </div>
                    </td>
                    <td>
                      <strong><?= h((string)($r['provider'] ?? '—')) ?></strong>
                      <div class="text-muted small"><?= h((string)($r['provider_device_id'] ?? '')) ?></div>
                    </td>
                    <td>
                      <?php if (!empty($r['asset_id'])): ?>
                        <strong>Asset #<?= h((string)$r['asset_id']) ?></strong>
                        <div class="text-muted small">Assigned <?= h((string)($r['assigned_at'] ?? '')) ?></div>
                      <?php else: ?>
                        <span class="text-muted">— Unassigned —</span>
                      <?php endif; ?>
                    </td>
                    <td>
                      <?php if (!empty($r['pinged_at'])): ?>
                        <strong><?= h((string)$r['pinged_at']) ?></strong>
                        <div class="text-muted small">
                          <?= h((string)$r['lat']) ?>, <?= h((string)$r['lng']) ?>
                          <?php if ($r['speed_mph'] !== null): ?> · <?= h((string)$r['speed_mph']) ?> mph<?php endif; ?>
                        </div>
                      <?php else: ?>
                        <span class="text-muted">No pings</span>
                      <?php endif; ?>
                    </td>
                    <td class="text-end">
                      <a href="<?= app_path('gps/gps_edit.php?id=' . (int)$r['id']) ?>" class="text-muted" title="View / Edit">
                        <i class="fa-solid fa-chevron-right"></i>
                      </a>
                    </td>
                  </tr>
                <?php endforeach; ?>
              <?php endif; ?>
              </tbody>
            </table>
          </div>
        </div>
      </div>
    </div>

    <!-- MAP -->
    <div class="col-12 col-xl-5">
      <div class="so-card">
        <div class="so-card-h">
          <div>
            <h5 class="so-card-title mb-0">
              <i class="fa-solid fa-map-location-dot me-2" style="color:#cf1f1f"></i>Live Map
            </h5>
            <div class="text-muted small mt-1">Latest ping per device.</div>
          </div>
          <a class="so-btn btn btn-sm" href="javascript:void(0)" id="gpsRefreshBtn">
            <i class="fa-solid fa-rotate me-2"></i>Refresh
          </a>
        </div>
        <div class="so-card-body">
          <div id="gpsMap" class="so-map"></div>
        </div>
      </div>
    </div>
  </div>

</div>

<script>
(function(){
  const mapEl = document.getElementById('gpsMap');
  if (!mapEl) return;

  // Default center (US-ish); we recenter after loading pings
  const map = L.map('gpsMap').setView([37.8, -96], 4);
  L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
    maxZoom: 19,
    attribution: '&copy; OpenStreetMap'
  }).addTo(map);

  let markers = [];

  function clearMarkers(){
    markers.forEach(m => map.removeLayer(m));
    markers = [];
  }

  async function load(){
    const res = await fetch('<?= app_path('gps/gps_lookup.php') ?>');
    const data = await res.json();
    clearMarkers();

    if (!data.ok) return;

    const pts = data.items || [];
    const bounds = [];

    pts.forEach(item => {
      if (!item.lat || !item.lng) return;
      const lat = parseFloat(item.lat);
      const lng = parseFloat(item.lng);
      if (Number.isNaN(lat) || Number.isNaN(lng)) return;

      const label = item.device_name + (item.asset_id ? (' · Asset #' + item.asset_id) : '');
      const sub = (item.pinged_at ? item.pinged_at : 'No ping');

      const m = L.marker([lat,lng]).addTo(map).bindPopup(
        `<strong>${label}</strong><br><span style="color:#6b7280;font-size:12px">${sub}</span>`
      );
      markers.push(m);
      bounds.push([lat,lng]);
    });

    if (bounds.length) {
      map.fitBounds(bounds, {padding:[20,20]});
    }
  }

  document.getElementById('gpsRefreshBtn')?.addEventListener('click', load);

  load();
})();
</script>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
