<?php
// /app/gps/gps_edit.php
declare(strict_types=1);

require_once __DIR__ . '/../includes/auth.php';
require_company_context_if_rto();

require_once __DIR__ . '/../includes/acl.php';
require_once __DIR__ . '/../includes/scope.php';

$pageTitle = 'GPS · Device';
$activeNav = 'gps';

$pdo = db();
acl_bootstrap($pdo);

$isCompanyAdmin = user_has_role('company_admin');
$isManager      = user_has_role('manager');
$canManage = ($isCompanyAdmin || $isManager || user_can('gps.manage'));
$canView   = ($isCompanyAdmin || $isManager || user_can('gps.view'));

if (!$canView) { http_response_code(403); exit('Forbidden'); }

$tenantId = company_context_id();

$id = (int)($_GET['id'] ?? 0);
if ($id <= 0) { http_response_code(400); exit('Missing id'); }

// Device
$d = $pdo->prepare("SELECT * FROM gps_devices WHERE id=:id AND tenant_id=:tid LIMIT 1");
$d->execute([':id'=>$id, ':tid'=>$tenantId]);
$device = $d->fetch(PDO::FETCH_ASSOC);
if (!$device) { http_response_code(404); exit('Device not found'); }

// Current assignment
$a = $pdo->prepare("
  SELECT * FROM gps_assignments
  WHERE tenant_id=:tid AND gps_device_id=:id AND unassigned_at IS NULL
  ORDER BY assigned_at DESC
  LIMIT 1
");
$a->execute([':tid'=>$tenantId, ':id'=>$id]);
$assign = $a->fetch(PDO::FETCH_ASSOC);

// Recent pings
$p = $pdo->prepare("
  SELECT *
  FROM gps_pings
  WHERE tenant_id=:tid AND gps_device_id=:id
  ORDER BY pinged_at DESC
  LIMIT 50
");
$p->execute([':tid'=>$tenantId, ':id'=>$id]);
$pings = $p->fetchAll(PDO::FETCH_ASSOC);

require_once __DIR__ . '/../includes/header.php';
?>
<div class="container-fluid">

  <div class="row">
    <div class="col-12">
      <div class="so-page-title"><?= h($device['device_name'] ?? ('Device #' . $id)) ?></div>
      <div class="so-page-sub">Provider: <?= h((string)($device['provider'] ?? '—')) ?> · ID: <?= (int)$id ?></div>
    </div>
  </div>

  <div class="row g-3 mb-3">
    <div class="col-12 col-xl-7">
      <div class="so-card">
        <div class="so-card-h">
          <h5 class="so-card-title"><i class="fa-solid fa-pen-to-square me-2" style="color:#cf1f1f"></i>Edit Device</h5>
          <a class="so-btn btn btn-sm" href="<?= app_path('gps/gps_list.php') ?>"><i class="fa-solid fa-arrow-left me-2"></i>Back</a>
        </div>
        <div class="so-card-body">
          <?php if (!$canManage): ?>
            <div class="text-muted">You have view-only access.</div>
          <?php endif; ?>

          <form method="post" action="<?= app_path('gps/gps_save.php') ?>" class="row g-2">
            <input type="hidden" name="_csrf" value="<?= h(csrf_token()) ?>">
            <input type="hidden" name="id" value="<?= (int)$device['id'] ?>">

            <div class="col-12 col-md-6">
              <label class="form-label small text-muted mb-1">Device Name</label>
              <input class="form-control" name="device_name" value="<?= h((string)$device['device_name']) ?>" <?= $canManage ? 'required' : 'disabled' ?>>
            </div>

            <div class="col-12 col-md-3">
              <label class="form-label small text-muted mb-1">Provider</label>
              <input class="form-control" name="provider" value="<?= h((string)$device['provider']) ?>" <?= $canManage ? '' : 'disabled' ?>>
            </div>

            <div class="col-12 col-md-3">
              <label class="form-label small text-muted mb-1">Provider Device ID</label>
              <input class="form-control" name="provider_device_id" value="<?= h((string)$device['provider_device_id']) ?>" <?= $canManage ? '' : 'disabled' ?>>
            </div>

            <div class="col-12 col-md-4">
              <label class="form-label small text-muted mb-1">Status</label>
              <select class="form-select" name="status" <?= $canManage ? '' : 'disabled' ?>>
                <?php foreach (['active','inactive','suspended','lost'] as $st): ?>
                  <option value="<?= h($st) ?>" <?= ($device['status'] === $st) ? 'selected' : '' ?>><?= h($st) ?></option>
                <?php endforeach; ?>
              </select>
            </div>

            <?php if ($canManage): ?>
              <div class="col-12">
                <button class="so-btn btn btn-sm" type="submit"><i class="fa-solid fa-floppy-disk me-2"></i>Save</button>
              </div>
            <?php endif; ?>
          </form>
        </div>
      </div>

      <div class="so-card mt-3">
        <div class="so-card-h">
          <h5 class="so-card-title"><i class="fa-solid fa-location-dot me-2" style="color:#cf1f1f"></i>Recent Pings</h5>
        </div>
        <div class="so-card-body">
          <div class="table-responsive">
            <table class="table so-table align-middle">
              <thead>
                <tr>
                  <th>Time</th>
                  <th>Lat/Lng</th>
                  <th class="text-end">Speed</th>
                  <th class="text-end">Heading</th>
                </tr>
              </thead>
              <tbody>
              <?php if (!$pings): ?>
                <tr><td colspan="4" class="text-center text-muted py-4">No pings for this device.</td></tr>
              <?php else: ?>
                <?php foreach ($pings as $pp): ?>
                  <tr>
                    <td><strong><?= h((string)$pp['pinged_at']) ?></strong></td>
                    <td class="text-muted small"><?= h((string)$pp['lat']) ?>, <?= h((string)$pp['lng']) ?></td>
                    <td class="text-end"><?= $pp['speed_mph'] === null ? '—' : h((string)$pp['speed_mph']) . ' mph' ?></td>
                    <td class="text-end"><?= $pp['heading_deg'] === null ? '—' : h((string)$pp['heading_deg']) ?></td>
                  </tr>
                <?php endforeach; ?>
              <?php endif; ?>
              </tbody>
            </table>
          </div>
        </div>
      </div>

    </div>

    <div class="col-12 col-xl-5">
      <div class="so-card">
        <div class="so-card-h">
          <h5 class="so-card-title"><i class="fa-solid fa-link me-2" style="color:#cf1f1f"></i>Assignment</h5>
        </div>
        <div class="so-card-body">
          <?php if ($assign): ?>
            <div class="mb-2">
              <strong>Asset #<?= h((string)$assign['asset_id']) ?></strong>
              <div class="text-muted small">Assigned at <?= h((string)$assign['assigned_at']) ?></div>
            </div>

            <?php if ($canManage): ?>
              <form method="post" action="<?= app_path('gps/gps_unassign.php') ?>">
                <input type="hidden" name="_csrf" value="<?= h(csrf_token()) ?>">
                <input type="hidden" name="gps_device_id" value="<?= (int)$device['id'] ?>">
                <button class="so-btn btn btn-sm" type="submit" style="border-color:rgba(220,38,38,0.35);background:rgba(220,38,38,0.10);color:#b91c1c;">
                  <i class="fa-solid fa-link-slash me-2"></i>Unassign
                </button>
              </form>
            <?php endif; ?>

          <?php else: ?>
            <div class="text-muted">No active assignment.</div>

            <?php if ($canManage): ?>
              <hr>
              <form method="post" action="<?= app_path('gps/gps_assign.php') ?>" class="row g-2">
                <input type="hidden" name="_csrf" value="<?= h(csrf_token()) ?>">
                <input type="hidden" name="gps_device_id" value="<?= (int)$device['id'] ?>">

                <div class="col-12">
                  <label class="form-label small text-muted mb-1">Assign to Asset ID</label>
                  <input class="form-control" name="asset_id" required placeholder="Enter your internal asset id (numeric or string)">
                </div>

                <div class="col-12">
                  <button class="so-btn btn btn-sm" type="submit">
                    <i class="fa-solid fa-link me-2"></i>Assign
                  </button>
                </div>
              </form>
            <?php endif; ?>
          <?php endif; ?>
        </div>
      </div>

      <div class="so-card mt-3">
        <div class="so-card-h">
          <div>
            <h5 class="so-card-title mb-0"><i class="fa-solid fa-map-location-dot me-2" style="color:#cf1f1f"></i>Map</h5>
            <div class="text-muted small mt-1">Latest ping (if available).</div>
          </div>
        </div>
        <div class="so-card-body">
          <div id="deviceMap" class="so-map"></div>
        </div>
      </div>

    </div>
  </div>
</div>

<script>
(function(){
  const pings = <?= json_encode($pings, JSON_UNESCAPED_SLASHES) ?>;
  const map = L.map('deviceMap').setView([37.8, -96], 4);
  L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', { maxZoom: 19 }).addTo(map);

  if (!pings || !pings.length) return;

  const latest = pings[0];
  const lat = parseFloat(latest.lat);
  const lng = parseFloat(latest.lng);
  if (Number.isNaN(lat) || Number.isNaN(lng)) return;

  const marker = L.marker([lat,lng]).addTo(map);
  marker.bindPopup(`<strong><?= h((string)($device['device_name'] ?? 'Device')) ?></strong><br><?= h((string)($device['provider'] ?? '')) ?><br><span style="color:#6b7280;font-size:12px">${latest.pinged_at}</span>`).openPopup();
  map.setView([lat,lng], 12);
})();
</script>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
