<?php
// /app/driver/repos.php
declare(strict_types=1);

require_once __DIR__ . '/../includes/auth.php';
require_login();

require_once __DIR__ . '/../includes/acl.php';
require_once __DIR__ . '/../includes/scope.php';

$pdo = db();
acl_bootstrap($pdo);

if (current_tenant_type() !== 'company') {
  http_response_code(403);
  echo "Driver mode is for company accounts only.";
  exit;
}

if (!user_has_role('driver')) {
  http_response_code(403);
  echo "Forbidden: driver role required.";
  exit;
}

$pageTitle = 'My Repos';
$activeNav = 'repos';

$companyTenantId = company_context_id();
$driverUserId = (int) current_user_id();

/**
 * Assigned repos + final state indicators:
 * - pickup submitted? (repo_pickups)
 * - repairs needed? (repo_pickups.repairs_needed)
 * - invoice status (repo_invoices.status)
 * - photo count (repo_media)
 */
$stmt = $pdo->prepare("
  SELECT
    r.id,
    r.status,
    r.priority,
    r.pickup_city,
    r.pickup_state,
    r.pickup_address1,
    r.pickup_zip,
    r.updated_at,
    r.created_at,
    a.stock_number,
    a.width_ft,
    a.length_ft,
    a.asset_type,
    t_rto.name AS rto_name,

    pk.picked_up_at,
    pk.repairs_needed,

    inv.status AS invoice_status,
    inv.overridden AS invoice_overridden,
    inv.total AS invoice_total,

    COALESCE(m.photo_count, 0) AS photo_count
  FROM repos r
  JOIN assets a ON a.id = r.asset_id
  LEFT JOIN tenants t_rto ON t_rto.id = r.rto_tenant_id

  LEFT JOIN repo_pickups pk
    ON pk.tenant_id = r.tenant_id AND pk.repo_id = r.id

  LEFT JOIN repo_invoices inv
    ON inv.tenant_id = r.tenant_id AND inv.repo_id = r.id

  LEFT JOIN (
    SELECT tenant_id, repo_id, COUNT(*) AS photo_count
    FROM repo_media
    GROUP BY tenant_id, repo_id
  ) m ON m.tenant_id = r.tenant_id AND m.repo_id = r.id

  WHERE r.tenant_id = :tid1
    AND r.assigned_driver_user_id = :uid1
    AND r.status NOT IN ('closed','canceled')
  ORDER BY
    FIELD(r.status,'new','assigned','en_route','picked_up','at_yard'),
    r.updated_at DESC
  LIMIT 200
");
$stmt->execute([':tid1' => $companyTenantId, ':uid1' => $driverUserId]);
$repos = $stmt->fetchAll();

function pillClass(string $status): string {
  return match ($status) {
    'new' => 'blue',
    'assigned','en_route' => 'orange',
    'picked_up','at_yard' => 'red',
    default => 'gray'
  };
}

require_once __DIR__ . '/../includes/header.php';
?>
<div class="container-fluid">

  <div class="d-flex flex-wrap justify-content-between align-items-end gap-2 mb-3">
    <div>
      <div class="so-page-title">My Assigned Repos</div>
      <div class="so-page-sub">Tap Pickup Package to submit photos, drop-off, miles, add-ons/override (if allowed).</div>
    </div>
    <div class="d-flex gap-2">
      <a class="btn so-btn" href="<?= app_path('driver/repos.php') ?>"><i class="fa-solid fa-rotate me-2"></i>Refresh</a>
    </div>
  </div>

  <?php if (!$repos): ?>
    <div class="so-card">
      <div class="so-card-body">
        <div class="text-muted">No assigned repos found.</div>
      </div>
    </div>
  <?php else: ?>

    <div class="row g-3">
      <?php foreach ($repos as $r): ?>
        <?php
          $size = trim(((int)$r['width_ft'] ? (int)$r['width_ft'].'x' : '') . ((int)$r['length_ft'] ?? ''));
          $size = $size ?: strtoupper((string)$r['asset_type']);

          $status = (string)$r['status'];
          $repoUrl = app_path('driver/repo_view.php?id='.(int)$r['id']);
          $pickupUrl = $repoUrl . '&pickup=1';

          $pickupSubmitted = !empty($r['picked_up_at']);
          $needsRepairs = ((int)($r['repairs_needed'] ?? 0) === 1);
          $photoCount = (int)($r['photo_count'] ?? 0);

          $invStatus = (string)($r['invoice_status'] ?? '');
          $invOver = ((int)($r['invoice_overridden'] ?? 0) === 1);

          $stateLabel = 'Not submitted';
          $statePill = 'gray';

          if ($pickupSubmitted) {
            if ($invStatus === 'submitted') { $stateLabel = 'Override pending approval'; $statePill = 'orange'; }
            elseif ($invStatus === 'approved') { $stateLabel = 'Approved'; $statePill = 'blue'; }
            else { $stateLabel = 'Submitted'; $statePill = 'blue'; }
          }
        ?>
        <div class="col-12 col-lg-6 col-xxl-4">
          <div class="so-card h-100">
            <div class="so-card-body">
              <div class="d-flex justify-content-between">
                <div>
                  <div class="fw-bold" style="font-size:18px;">
                    #<?= (int)$r['id'] ?> · <?= h((string)$r['stock_number']) ?>
                  </div>
                  <div class="text-muted small"><?= h($size) ?></div>
                </div>
                <div class="text-end">
                  <span class="so-pill <?= h(pillClass($status)) ?>">
                    <?= h(strtoupper(str_replace('_',' ', $status))) ?>
                  </span>
                </div>
              </div>

              <div class="mt-2 d-flex flex-wrap gap-2">
                <span class="so-pill <?= h($statePill) ?>"><?= h($stateLabel) ?></span>
                <?php if ($needsRepairs): ?><span class="so-pill red">Repairs Needed</span><?php endif; ?>
                <span class="so-pill gray">Photos: <?= $photoCount ?></span>
                <?php if ($invOver): ?><span class="so-pill orange">Override</span><?php endif; ?>
              </div>

              <hr>

              <div class="text-muted small">Pickup</div>
              <div class="fw-bold">
                <?= h((string)($r['pickup_address1'] ?? '—')) ?>
              </div>
              <div class="text-muted small">
                <?= h((string)($r['pickup_city'] ?? '')) ?> <?= h((string)($r['pickup_state'] ?? '')) ?> <?= h((string)($r['pickup_zip'] ?? '')) ?>
              </div>

              <?php if (!empty($r['rto_name'])): ?>
                <div class="mt-2 text-muted small">RTO</div>
                <div class="fw-bold"><?= h((string)$r['rto_name']) ?></div>
              <?php endif; ?>

              <div class="d-flex gap-2 mt-3">
                <a class="btn btn-outline-dark w-50" style="border-radius:12px;" href="<?= $repoUrl ?>">
                  View
                </a>
                <a class="btn btn-dark w-50" style="border-radius:12px;" href="<?= $pickupUrl ?>">
                  <i class="fa-solid fa-camera me-2"></i>Pickup Package
                </a>
              </div>

            </div>
          </div>
        </div>
      <?php endforeach; ?>
    </div>

  <?php endif; ?>

</div>
<?php require_once __DIR__ . '/../includes/footer.php'; ?>
