<?php
// /app/dev/seed.php
declare(strict_types=1);

/**
 * One-time Seeder for ShedOffice demo data
 *
 * SAFETY:
 * - This script is intended for dev/testing only.
 * - It locks after first successful run by writing /app/dev/.seed_ran
 * - Delete /app/dev/seed.php after running.
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';

$lockFile = __DIR__ . '/.seed_ran';

if (file_exists($lockFile)) {
  http_response_code(403);
  echo "Seeder already ran. Delete {$lockFile} if you intentionally want to re-run (NOT recommended).\n";
  exit;
}

function now(): string { return date('Y-m-d H:i:s'); }

function ensureDir(string $path): void {
  if (!is_dir($path)) @mkdir($path, 0755, true);
}

function out(string $msg): void {
  echo $msg . "<br>\n";
}

$pdo = db();

$companyName = 'Demo Manufacturer Co';
$rtoName     = 'Demo RTO Funding';

$companyAdminEmail = 'admin@demomanufacturer.com';
$rtoAdminEmail     = 'admin@demorto.com';

// Demo passwords (change later)
$companyAdminPass = 'Password123!';
$rtoAdminPass     = 'Password123!';

// Optional: ensure tables exist (won't validate schema; assumes you ran the SQL already)
$requiredTables = ['tenants','users','locations','rto_company_links','assets','repos','repairs','repo_events'];

$stmt = $pdo->prepare("
  SELECT COUNT(*) 
  FROM information_schema.TABLES 
  WHERE TABLE_SCHEMA = :db
    AND TABLE_NAME = :tbl
  LIMIT 1
");

foreach ($requiredTables as $t) {
  $stmt->execute([':db' => DB_NAME, ':tbl' => $t]);
  $exists = (int)$stmt->fetchColumn();
  if ($exists !== 1) {
    http_response_code(500);
    echo "Missing table: {$t}. Run the schema SQL first.\n";
    exit;
  }
}

try {
  $pdo->beginTransaction();

  // ---------- TENANTS ----------
  // Company tenant
  $stmt = $pdo->prepare("SELECT id FROM tenants WHERE tenant_type='company' AND name=:n LIMIT 1");
  $stmt->execute([':n' => $companyName]);
  $companyTenantId = (int)($stmt->fetchColumn() ?: 0);

  if (!$companyTenantId) {
    $stmt = $pdo->prepare("
      INSERT INTO tenants (tenant_type, name, status, timezone, primary_contact_name, primary_contact_email, created_at, updated_at)
      VALUES ('company', :name, 'active', 'America/Indiana/Indianapolis', 'Demo Admin', :email, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
    ");
    $stmt->execute([':name' => $companyName, ':email' => $companyAdminEmail]);
    $companyTenantId = (int)$pdo->lastInsertId();
    out("Created company tenant #{$companyTenantId}");
  } else {
    out("Company tenant exists #{$companyTenantId}");
  }

  // RTO tenant
  $stmt = $pdo->prepare("SELECT id FROM tenants WHERE tenant_type='rto' AND name=:n LIMIT 1");
  $stmt->execute([':n' => $rtoName]);
  $rtoTenantId = (int)($stmt->fetchColumn() ?: 0);

  if (!$rtoTenantId) {
    $stmt = $pdo->prepare("
      INSERT INTO tenants (tenant_type, name, status, timezone, primary_contact_name, primary_contact_email, created_at, updated_at)
      VALUES ('rto', :name, 'active', 'America/Indiana/Indianapolis', 'Demo RTO Admin', :email, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
    ");
    $stmt->execute([':name' => $rtoName, ':email' => $rtoAdminEmail]);
    $rtoTenantId = (int)$pdo->lastInsertId();
    out("Created RTO tenant #{$rtoTenantId}");
  } else {
    out("RTO tenant exists #{$rtoTenantId}");
  }

  // ---------- LINK RTO -> COMPANY ----------
  $stmt = $pdo->prepare("
    SELECT id FROM rto_company_links
    WHERE rto_tenant_id=:rto AND company_tenant_id=:co LIMIT 1
  ");
  $stmt->execute([':rto' => $rtoTenantId, ':co' => $companyTenantId]);
  $linkId = (int)($stmt->fetchColumn() ?: 0);

  if (!$linkId) {
    $stmt = $pdo->prepare("
      INSERT INTO rto_company_links (rto_tenant_id, company_tenant_id, status, created_at)
      VALUES (:rto, :co, 'active', CURRENT_TIMESTAMP)
    ");
    $stmt->execute([':rto' => $rtoTenantId, ':co' => $companyTenantId]);
    $linkId = (int)$pdo->lastInsertId();
    out("Linked RTO #{$rtoTenantId} to Company #{$companyTenantId} (link #{$linkId})");
  } else {
    out("RTO link exists (link #{$linkId})");
  }

  // ---------- LOCATION ----------
  $stmt = $pdo->prepare("SELECT id FROM locations WHERE tenant_id=:tid AND name='Main Lot' LIMIT 1");
  $stmt->execute([':tid' => $companyTenantId]);
  $locationId = (int)($stmt->fetchColumn() ?: 0);

  if (!$locationId) {
    $stmt = $pdo->prepare("
      INSERT INTO locations
        (tenant_id, name, address1, city, state, zip, lat, lng, phone, status, created_at, updated_at)
      VALUES
        (:tid, 'Main Lot', '1971 US HWY 60 E', 'Salem', 'KY', '42078', 37.2645, -88.2427, '555-0100', 'active', CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
    ");
    $stmt->execute([':tid' => $companyTenantId]);
    $locationId = (int)$pdo->lastInsertId();
    out("Created location #{$locationId}");
  } else {
    out("Location exists #{$locationId}");
  }

  // ---------- USERS ----------
  // Company admin
  $stmt = $pdo->prepare("SELECT id FROM users WHERE tenant_id=:tid AND email=:e LIMIT 1");
  $stmt->execute([':tid' => $companyTenantId, ':e' => $companyAdminEmail]);
  $companyUserId = (int)($stmt->fetchColumn() ?: 0);

  if (!$companyUserId) {
    $stmt = $pdo->prepare("
      INSERT INTO users
        (tenant_id, home_location_id, email, password_hash, first_name, last_name, status, mfa_enabled, created_at, updated_at)
      VALUES
        (:tid, :loc, :email, :ph, 'Demo', 'Admin', 'active', 0, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
    ");
    $stmt->execute([
      ':tid' => $companyTenantId,
      ':loc' => $locationId,
      ':email' => $companyAdminEmail,
      ':ph' => password_hash($companyAdminPass, PASSWORD_DEFAULT),
    ]);
    $companyUserId = (int)$pdo->lastInsertId();
    out("Created company admin user #{$companyUserId}");
  } else {
    out("Company admin user exists #{$companyUserId}");
  }

  // RTO admin
  $stmt = $pdo->prepare("SELECT id FROM users WHERE tenant_id=:tid AND email=:e LIMIT 1");
  $stmt->execute([':tid' => $rtoTenantId, ':e' => $rtoAdminEmail]);
  $rtoUserId = (int)($stmt->fetchColumn() ?: 0);

  if (!$rtoUserId) {
    $stmt = $pdo->prepare("
      INSERT INTO users
        (tenant_id, home_location_id, email, password_hash, first_name, last_name, status, mfa_enabled, created_at, updated_at)
      VALUES
        (:tid, NULL, :email, :ph, 'Demo', 'RTO', 'active', 0, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
    ");
    $stmt->execute([
      ':tid' => $rtoTenantId,
      ':email' => $rtoAdminEmail,
      ':ph' => password_hash($rtoAdminPass, PASSWORD_DEFAULT),
    ]);
    $rtoUserId = (int)$pdo->lastInsertId();
    out("Created RTO admin user #{$rtoUserId}");
  } else {
    out("RTO admin user exists #{$rtoUserId}");
  }

  // ---------- ASSETS ----------
  $assetsToEnsure = [
    ['stock' => 'S100001', 'type' => 'shed', 'w' => 10, 'l' => 16, 'h' => null, 'serial' => 'SN-AAA-100001'],
    ['stock' => 'S100002', 'type' => 'shed', 'w' => 12, 'l' => 20, 'h' => null, 'serial' => 'SN-AAA-100002'],
    ['stock' => 'S100003', 'type' => 'cabin','w' => 14, 'l' => 24, 'h' => null, 'serial' => 'SN-AAA-100003'],
    ['stock' => 'S100004', 'type' => 'shed', 'w' => 10, 'l' => 12, 'h' => null, 'serial' => 'SN-AAA-100004'],
  ];

  $assetIds = [];
  foreach ($assetsToEnsure as $a) {
    $stmt = $pdo->prepare("SELECT id FROM assets WHERE tenant_id=:tid AND stock_number=:sn LIMIT 1");
    $stmt->execute([':tid' => $companyTenantId, ':sn' => $a['stock']]);
    $aid = (int)($stmt->fetchColumn() ?: 0);

    if (!$aid) {
      $stmt = $pdo->prepare("
        INSERT INTO assets
          (tenant_id, location_id, stock_number, asset_type, width_ft, length_ft, height_ft, serial_number, status, created_at, updated_at)
        VALUES
          (:tid, :loc, :sn, :atype, :w, :l, :h, :serial, 'repo_in_progress', CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
      ");
      $stmt->execute([
        ':tid' => $companyTenantId,
        ':loc' => $locationId,
        ':sn' => $a['stock'],
        ':atype' => $a['type'],
        ':w' => $a['w'],
        ':l' => $a['l'],
        ':h' => $a['h'],
        ':serial' => $a['serial'],
      ]);
      $aid = (int)$pdo->lastInsertId();
      out("Created asset {$a['stock']} (#{$aid})");
    } else {
      out("Asset exists {$a['stock']} (#{$aid})");
    }

    $assetIds[] = $aid;
  }

  // ---------- REPOS ----------
  // We'll create 4 repos linked to the RTO tenant under the company tenant.
  $repoTemplates = [
    [
      'asset_id' => $assetIds[0],
      'status' => 'assigned',
      'priority' => 'high',
      'pickup_city' => 'Paducah',
      'pickup_state' => 'KY',
      'assigned_driver_user_id' => $companyUserId, // placeholder for now
      'notes' => 'Customer defaulted. Coordinate pickup with RTO.'
    ],
    [
      'asset_id' => $assetIds[1],
      'status' => 'en_route',
      'priority' => 'medium',
      'pickup_city' => 'Jackson',
      'pickup_state' => 'TN',
      'assigned_driver_user_id' => $companyUserId,
      'notes' => 'Repo scheduled for today. Confirm gate access.'
    ],
    [
      'asset_id' => $assetIds[2],
      'status' => 'picked_up',
      'priority' => 'high',
      'pickup_city' => 'Evansville',
      'pickup_state' => 'IN',
      'assigned_driver_user_id' => $companyUserId,
      'notes' => 'Picked up. Likely needs repairs.'
    ],
    [
      'asset_id' => $assetIds[3],
      'status' => 'new',
      'priority' => 'low',
      'pickup_city' => 'Nashville',
      'pickup_state' => 'TN',
      'assigned_driver_user_id' => null,
      'notes' => 'New repo request from RTO.'
    ],
  ];

  $repoIds = [];

  foreach ($repoTemplates as $idx => $r) {
    // Ensure a repo for this asset_id + rto_tenant exists (demo uniqueness rule)
    $stmt = $pdo->prepare("
      SELECT id FROM repos
      WHERE tenant_id=:tid AND rto_tenant_id=:rto AND asset_id=:aid
      LIMIT 1
    ");
    $stmt->execute([
      ':tid' => $companyTenantId,
      ':rto' => $rtoTenantId,
      ':aid' => (int)$r['asset_id']
    ]);
    $rid = (int)($stmt->fetchColumn() ?: 0);

    if (!$rid) {
      $pickedUpAt = ($r['status'] === 'picked_up') ? now() : null;

      $stmt = $pdo->prepare("
        INSERT INTO repos
          (tenant_id, rto_tenant_id, asset_id, location_id, status, priority,
           assigned_driver_user_id,
           pickup_address1, pickup_city, pickup_state, pickup_zip,
           picked_up_at, notes, created_by_user_id, created_at, updated_at)
        VALUES
          (:tid, :rto, :aid, :loc, :status, :priority,
           :driver,
           '123 Demo St', :pcity, :pstate, '00000',
           :picked, :notes, :created_by, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
      ");
      $stmt->execute([
        ':tid' => $companyTenantId,
        ':rto' => $rtoTenantId,
        ':aid' => (int)$r['asset_id'],
        ':loc' => $locationId,
        ':status' => $r['status'],
        ':priority' => $r['priority'],
        ':driver' => $r['assigned_driver_user_id'],
        ':pcity' => $r['pickup_city'],
        ':pstate' => $r['pickup_state'],
        ':picked' => $pickedUpAt,
        ':notes' => $r['notes'],
        ':created_by' => $rtoUserId, // created by RTO user for demo
      ]);
      $rid = (int)$pdo->lastInsertId();
      out("Created repo #{$rid} (status={$r['status']})");
    } else {
      out("Repo exists for asset_id={$r['asset_id']} (#{$rid})");
    }

    // Add repo event
    $stmt = $pdo->prepare("
      INSERT INTO repo_events (tenant_id, repo_id, event_type, meta_json, created_by_user_id, created_at)
      VALUES (:tid, :rid, :etype, :meta, :uid, CURRENT_TIMESTAMP)
    ");
    $stmt->execute([
      ':tid' => $companyTenantId,
      ':rid' => $rid,
      ':etype' => 'seed_created',
      ':meta' => json_encode(['note' => 'Seed data created'], JSON_UNESCAPED_SLASHES),
      ':uid' => $companyUserId,
    ]);

    $repoIds[] = $rid;
  }

  // ---------- REPAIR (auto_from_repo_pickup) ----------
  // Create repair for the picked_up repo (3rd one)
  $pickedRepoId = $repoIds[2] ?? 0;
  if ($pickedRepoId) {
    $stmt = $pdo->prepare("SELECT id, asset_id, location_id FROM repos WHERE tenant_id=:tid AND id=:rid LIMIT 1");
    $stmt->execute([':tid' => $companyTenantId, ':rid' => $pickedRepoId]);
    $pr = $stmt->fetch();

    if ($pr) {
      $stmt = $pdo->prepare("SELECT id FROM repairs WHERE tenant_id=:tid AND repo_id=:rid LIMIT 1");
      $stmt->execute([':tid' => $companyTenantId, ':rid' => $pickedRepoId]);
      $repId = (int)($stmt->fetchColumn() ?: 0);

      if (!$repId) {
        $stmt = $pdo->prepare("
          INSERT INTO repairs
            (tenant_id, rto_tenant_id, asset_id, repo_id, location_id, status, priority, source, created_by_user_id, created_at, updated_at)
          VALUES
            (:tid, :rto, :aid, :rid, :loc, 'new', 'high', 'auto_from_repo_pickup', :uid, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
        ");
        $stmt->execute([
          ':tid' => $companyTenantId,
          ':rto' => $rtoTenantId,
          ':aid' => (int)$pr['asset_id'],
          ':rid' => $pickedRepoId,
          ':loc' => $pr['location_id'],
          ':uid' => $companyUserId,
        ]);
        $repId = (int)$pdo->lastInsertId();
        out("Created repair #{$repId} for picked_up repo #{$pickedRepoId}");
      } else {
        out("Repair already exists for picked_up repo #{$pickedRepoId} (repair #{$repId})");
      }
    }
  }

  $pdo->commit();

  // Write lock file so it cannot run again accidentally
  file_put_contents($lockFile, "seed_ran_at=" . now() . "\n");

  out("<hr>");
  out("<strong>Seed complete.</strong>");
  out("Company login: <code>{$companyAdminEmail}</code> / <code>{$companyAdminPass}</code>");
  out("RTO login: <code>{$rtoAdminEmail}</code> / <code>{$rtoAdminPass}</code>");
  out("<br>Go to: <a href=\"" . APP_BASE_URL . "/index.php\">" . APP_BASE_URL . "/index.php</a>");
  out("<br><strong>Delete this file:</strong> /app/dev/seed.php");

} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  http_response_code(500);
  echo "Seed failed: " . htmlspecialchars($e->getMessage());
}
