<?php
// /app/deliveries/deliveries.php
declare(strict_types=1);

require_once __DIR__ . '/_lib.php';

if (!d_can_view()) { http_response_code(403); exit('Forbidden'); }

$pageTitle = 'Deliveries';
$activeNav = 'deliveries';
$tenantId  = company_context_id();

$status = trim((string)($_GET['status'] ?? ''));
$q      = trim((string)($_GET['q'] ?? ''));
$view   = trim((string)($_GET['view'] ?? 'active')); // active|all|delivered

$where = ["o.tenant_id = :tid"];
$params = [':tid' => $tenantId];

if ($view === 'active') {
  $where[] = "o.status IN ('submitted','processed')";
} elseif ($view === 'delivered') {
  $where[] = "o.status = 'delivered'";
}

if ($status !== '') {
  $where[] = "o.status = :status";
  $params[':status'] = $status;
}

if ($q !== '') {
  $where[] = "(o.order_number LIKE :q OR o.customer_name LIKE :q OR o.serial_number LIKE :q OR o.delivery_city LIKE :q OR o.delivery_state LIKE :q)";
  $params[':q'] = '%'.$q.'%';
}

$whereSql = "WHERE " . implode(" AND ", $where);

$totalStmt = $pdo->prepare("SELECT COUNT(*) FROM orders o $whereSql");
$totalStmt->execute($params);
$total = (int)$totalStmt->fetchColumn();

/**
 * IMPORTANT: your users table does NOT have u.name.
 * Use first_name / last_name and email.
 */
$rowsStmt = $pdo->prepare("
  SELECT o.*,
         u.id AS driver_id,
         u.first_name AS driver_first_name,
         u.last_name  AS driver_last_name,
         u.email      AS driver_email
  FROM orders o
  LEFT JOIN users u ON u.id = o.driver_user_id
  $whereSql
  ORDER BY
    FIELD(o.status,'submitted','processed','delivered','canceled','unknown'),
    o.updated_at DESC
  LIMIT 200
");
$rowsStmt->execute($params);
$rows = $rowsStmt->fetchAll(PDO::FETCH_ASSOC);

$drivers = [];
try {
  // Some installs store drivers in users.role, others in a roles table.
  // Start simple: role='driver' if present; if not, show all users.
  $drivers = $pdo->query("
    SELECT id, first_name, last_name, email
    FROM users
    WHERE role='driver'
    ORDER BY first_name, last_name
  ")->fetchAll(PDO::FETCH_ASSOC);
} catch (Throwable $e) {
  $drivers = $pdo->query("
    SELECT id, first_name, last_name, email
    FROM users
    ORDER BY first_name, last_name
  ")->fetchAll(PDO::FETCH_ASSOC);
}

require_once __DIR__ . '/../includes/header.php';
?>
<div class="container-fluid">
  <div class="row">
    <div class="col-12">
      <div class="so-page-title">Deliveries</div>
      <div class="so-page-sub">Orders pulled from ShedSuite webhooks. Assign drivers and track delivery status.</div>
    </div>
  </div>

  <div class="row g-3 mb-3">
    <div class="col-12">
      <div class="so-card">
        <div class="so-card-h">
          <h5 class="so-card-title"><i class="fa-solid fa-truck me-2" style="color:#cf1f1f"></i>Delivery Queue</h5>
          <div class="d-flex gap-2 flex-wrap">
            <a class="so-btn btn btn-sm" href="<?= app_path('deliveries/deliveries.php?view=active') ?>">Active</a>
            <a class="so-btn btn btn-sm" href="<?= app_path('deliveries/deliveries.php?view=delivered') ?>" style="border-color:rgba(18,20,24,0.10);background:#fff;color:#111827;">Delivered</a>
            <a class="so-btn btn btn-sm" href="<?= app_path('deliveries/deliveries.php?view=all') ?>" style="border-color:rgba(18,20,24,0.10);background:#fff;color:#111827;">All</a>

            <a class="so-btn btn btn-sm" href="<?= app_path('deliveries/deliveries_map.php') ?>" style="border-color:rgba(18,20,24,0.10);background:#fff;color:#111827;">
              <i class="fa-solid fa-map-location-dot me-2"></i>Map
            </a>
            <a class="so-btn btn btn-sm" href="<?= app_path('deliveries/deliveries_optimize.php') ?>" style="border-color:rgba(18,20,24,0.10);background:#fff;color:#111827;">
              <i class="fa-solid fa-route me-2"></i>Optimize
            </a>
          </div>
        </div>

        <div class="so-card-body">
          <form class="row g-2 align-items-end" method="get">
            <input type="hidden" name="view" value="<?= h($view) ?>">
            <div class="col-12 col-lg-5">
              <label class="form-label small text-muted mb-1">Search</label>
              <input class="form-control" name="q" value="<?= h($q) ?>" placeholder="Order #, customer, serial, city...">
            </div>
            <div class="col-12 col-sm-6 col-lg-3">
              <label class="form-label small text-muted mb-1">Status</label>
              <select class="form-select" name="status">
                <option value="">All</option>
                <?php foreach (['submitted','processed','delivered','canceled'] as $st): ?>
                  <option value="<?= h($st) ?>" <?= $status===$st?'selected':'' ?>><?= h(ucfirst($st)) ?></option>
                <?php endforeach; ?>
              </select>
            </div>
            <div class="col-12 col-sm-6 col-lg-2 d-grid">
              <button class="so-btn btn btn-sm" type="submit">Apply</button>
            </div>
            <div class="col-12 col-lg-2 text-end text-muted small">
              Total: <strong><?= (int)$total ?></strong>
            </div>
          </form>
        </div>
      </div>
    </div>
  </div>

  <div class="row g-3">
    <div class="col-12">
      <div class="so-card">
        <div class="so-card-body">
          <div class="table-responsive">
            <table class="table so-table align-middle">
              <thead>
                <tr>
                  <th>Order</th>
                  <th>Customer</th>
                  <th>Location</th>
                  <th>Status</th>
                  <th>Driver</th>
                  <th class="text-end">Actions</th>
                </tr>
              </thead>
              <tbody>
              <?php if (!$rows): ?>
                <tr><td colspan="6" class="text-center text-muted py-4">No deliveries found.</td></tr>
              <?php else: ?>
                <?php foreach ($rows as $r): ?>
                  <?php
                    [$pillCls,$pillLbl] = d_status_pill((string)$r['status']);
                    $loc = trim(((string)($r['delivery_city'] ?? '')).', '.((string)($r['delivery_state'] ?? '')), ' ,');

                    $driverDisplay = '';
                    if (!empty($r['driver_id'])) {
                      $driverDisplay = trim(((string)($r['driver_first_name'] ?? '')).' '.((string)($r['driver_last_name'] ?? '')));
                      if ($driverDisplay === '') $driverDisplay = (string)($r['driver_email'] ?? '');
                    }
                  ?>
                  <tr>
                    <td>
                      <strong><?= h((string)$r['order_number']) ?></strong>
                      <?php if (!empty($r['serial_number'])): ?>
                        <div class="text-muted small">SN: <?= h((string)$r['serial_number']) ?></div>
                      <?php endif; ?>
                      <?php if (!empty($r['size']) || !empty($r['model'])): ?>
                        <div class="text-muted small"><?= h((string)($r['size'] ?? '')) ?> <?= h((string)($r['model'] ?? '')) ?></div>
                      <?php endif; ?>
                    </td>
                    <td><?= h((string)($r['customer_name'] ?? '')) ?></td>
                    <td>
                      <div><?= h($loc) ?></div>
                      <div class="text-muted small"><?= h((string)($r['delivery_street_line_one'] ?? '')) ?></div>
                    </td>
                    <td><span class="so-pill <?= h($pillCls) ?>"><?= h($pillLbl) ?></span></td>
                    <td>
                      <?php if (!d_can_manage()): ?>
                        <?= h($driverDisplay ?: '—') ?>
                      <?php else: ?>
                        <select class="form-select form-select-sm js-driver" data-id="<?= (int)$r['id'] ?>">
                          <option value="">— Unassigned —</option>
                          <?php foreach ($drivers as $d): ?>
                            <?php $dName = trim(((string)($d['first_name'] ?? '')).' '.((string)($d['last_name'] ?? ''))); ?>
                            <?php if ($dName === '') $dName = (string)($d['email'] ?? ('User #'.(int)$d['id'])); ?>
                            <option value="<?= (int)$d['id'] ?>" <?= ((int)$r['driver_user_id'] === (int)$d['id'])?'selected':'' ?>>
                              <?= h($dName) ?>
                            </option>
                          <?php endforeach; ?>
                        </select>
                      <?php endif; ?>
                    </td>
                    <td class="text-end">
                      <a class="text-muted" href="<?= app_path('deliveries/delivery_view.php?id='.(int)$r['id']) ?>">
                        <i class="fa-solid fa-chevron-right"></i>
                      </a>
                    </td>
                  </tr>
                <?php endforeach; ?>
              <?php endif; ?>
              </tbody>
            </table>
          </div>
          <?php if (d_can_manage()): ?>
            <div class="text-muted small mt-2">Tip: Assigning a driver saves instantly.</div>
          <?php endif; ?>
        </div>
      </div>
    </div>
  </div>
</div>

<?php if (d_can_manage()): ?>
<script>
document.addEventListener('change', async (e) => {
  const el = e.target;
  if (!el.classList.contains('js-driver')) return;
  const orderId = el.getAttribute('data-id');
  const driverId = el.value || '';
  const body = new URLSearchParams({ action:'driver', order_id: orderId, driver_user_id: driverId, _csrf: <?= json_encode(csrf_token()) ?> });

  const res = await fetch('<?= app_path('deliveries/delivery_ajax.php') ?>', {
    method: 'POST',
    headers: {'Content-Type':'application/x-www-form-urlencoded'},
    body: body.toString()
  });

  if (!res.ok) alert('Failed to update driver.');
});
</script>
<?php endif; ?>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
