<?php
// /app/admin/repo_fee_settings.php
declare(strict_types=1);

require_once __DIR__ . '/../includes/auth.php';
require_login();

require_once __DIR__ . '/../includes/acl.php';
require_once __DIR__ . '/../includes/scope.php';

$pageTitle = 'Repo Fee Settings';
$activeNav = 'settings';

$pdo = db();
acl_bootstrap($pdo);

$tenantType = current_tenant_type();
if ($tenantType === 'rto') {
  http_response_code(403);
  echo "RTO accounts cannot access fee settings.";
  exit;
}

$isCompanyAdmin = user_has_role('company_admin');
$isManager = user_has_role('manager');
if (!$isCompanyAdmin && !$isManager) {
  http_response_code(403);
  echo "Only admin/manager can access fee settings.";
  exit;
}

$companyTenantId = (int) current_tenant_id();
$user = current_user();

$ok = !empty($_GET['ok']);
$tab = trim((string)($_GET['tab'] ?? 'company'));
if (!in_array($tab, ['company','rto','addons'], true)) $tab = 'company';

// Load tenant settings
$stmt = $pdo->prepare("SELECT * FROM tenant_settings WHERE tenant_id=:tid1 LIMIT 1");
$stmt->execute([':tid1' => $companyTenantId]);
$settings = $stmt->fetch() ?: [
  'repo_invoice_bill_to' => 'rto',
  'driver_can_override_repo_invoice' => 0,
  'driver_can_submit_custom_invoice' => 0,
];

// Load schedules
$stmt = $pdo->prepare("
  SELECT *
  FROM repo_fee_schedules
  WHERE tenant_id=:tid1
  ORDER BY (applies_to='company') DESC, (is_active=1) DESC, id DESC
");
$stmt->execute([':tid1' => $companyTenantId]);
$schedules = $stmt->fetchAll();

// Load width rules (group by schedule)
$stmt = $pdo->prepare("
  SELECT *
  FROM repo_fee_width_rules
  WHERE schedule_id IN (SELECT id FROM repo_fee_schedules WHERE tenant_id=:tid1)
  ORDER BY schedule_id DESC, width_ft ASC
");
$stmt->execute([':tid1' => $companyTenantId]);
$widthRules = $stmt->fetchAll();

$rulesBySchedule = [];
foreach ($widthRules as $r) {
  $sid = (int)$r['schedule_id'];
  if (!isset($rulesBySchedule[$sid])) $rulesBySchedule[$sid] = [];
  $rulesBySchedule[$sid][] = $r;
}

// Load add-on fee types
$stmt = $pdo->prepare("
  SELECT *
  FROM repo_addon_fee_types
  WHERE tenant_id=:tid1
  ORDER BY (applies_to='company') DESC, is_active DESC, name ASC
");
$stmt->execute([':tid1' => $companyTenantId]);
$addons = $stmt->fetchAll();

// Load available RTO tenants linked to this company (for RTO-specific schedules/addons)
$stmt = $pdo->prepare("
  SELECT t.id, t.name
  FROM rto_company_links l
  JOIN tenants t ON t.id = l.rto_tenant_id
  WHERE l.company_tenant_id = :tid1 AND l.status='active'
  ORDER BY t.name ASC
");
$stmt->execute([':tid1' => $companyTenantId]);
$linkedRTOs = $stmt->fetchAll();

require_once __DIR__ . '/../includes/header.php';
?>
<div class="container-fluid">

  <div class="d-flex flex-wrap justify-content-between align-items-end gap-2 mb-3">
    <div>
      <div class="so-page-title">Repo Fee Settings</div>
      <div class="so-page-sub">Configure billing rules, schedules, width rules, mileage, and add-ons.</div>
    </div>
    <div class="d-flex gap-2">
      <a class="btn so-btn" href="<?= app_path('admin/repo_fee_settings.php') ?>"><i class="fa-solid fa-rotate me-2"></i>Refresh</a>
    </div>
  </div>

  <?php if ($ok): ?>
    <div class="alert alert-success">Saved successfully.</div>
  <?php endif; ?>

  <div class="so-card mb-3">
    <div class="so-card-body">
      <ul class="nav nav-pills gap-2">
        <li class="nav-item">
          <a class="nav-link <?= $tab==='company'?'active':'' ?>" href="<?= app_path('admin/repo_fee_settings.php?tab=company') ?>" style="border-radius:12px;">
            Company Defaults
          </a>
        </li>
        <li class="nav-item">
          <a class="nav-link <?= $tab==='rto'?'active':'' ?>" href="<?= app_path('admin/repo_fee_settings.php?tab=rto') ?>" style="border-radius:12px;">
            RTO Overrides
          </a>
        </li>
        <li class="nav-item">
          <a class="nav-link <?= $tab==='addons'?'active':'' ?>" href="<?= app_path('admin/repo_fee_settings.php?tab=addons') ?>" style="border-radius:12px;">
            Add-on Fees
          </a>
        </li>
      </ul>
    </div>
  </div>

  <?php if ($tab === 'company'): ?>
    <!-- COMPANY DEFAULTS -->
    <div class="row g-3">
      <div class="col-12 col-xl-5">
        <div class="so-card">
          <div class="so-card-h">
            <h5 class="so-card-title"><i class="fa-solid fa-gear me-2" style="color:#cf1f1f"></i>Company Billing Defaults</h5>
          </div>
          <div class="so-card-body">
            <form method="post" action="<?= app_path('admin/repo_fee_settings_save.php') ?>">
              <input type="hidden" name="_csrf" value="<?= h(csrf_token()) ?>">

              <div class="mb-3">
                <label class="form-label fw-bold">Repo Invoice Billed To</label>
                <select name="repo_invoice_bill_to" class="form-select" style="border-radius:12px;">
                  <option value="rto" <?= (($settings['repo_invoice_bill_to'] ?? 'rto')==='rto')?'selected':'' ?>>RTO</option>
                  <option value="company" <?= (($settings['repo_invoice_bill_to'] ?? 'rto')==='company')?'selected':'' ?>>Company</option>
                </select>
                <div class="text-muted small mt-1">If billed_to is RTO but the repo has no RTO tenant, it falls back to Company.</div>
              </div>

              <div class="form-check mb-2">
                <input class="form-check-input" type="checkbox" value="1" id="driver_override" name="driver_can_override_repo_invoice"
                  <?= ((int)($settings['driver_can_override_repo_invoice'] ?? 0)===1)?'checked':'' ?>>
                <label class="form-check-label fw-bold" for="driver_override">
                  Allow drivers to override invoice totals (requires approval)
                </label>
              </div>

              <div class="form-check mb-3">
                <input class="form-check-input" type="checkbox" value="1" id="driver_custom" name="driver_can_submit_custom_invoice"
                  <?= ((int)($settings['driver_can_submit_custom_invoice'] ?? 0)===1)?'checked':'' ?>>
                <label class="form-check-label fw-bold" for="driver_custom">
                  Allow drivers to submit custom invoices (future feature)
                </label>
              </div>

              <button class="btn btn-dark" style="border-radius:12px;" type="submit">
                <i class="fa-solid fa-floppy-disk me-2"></i>Save Defaults
              </button>
            </form>
          </div>
        </div>
      </div>

      <div class="col-12 col-xl-7">
        <div class="so-card">
          <div class="so-card-h">
            <h5 class="so-card-title"><i class="fa-solid fa-list me-2" style="color:#cf1f1f"></i>Fee Schedules</h5>
            <button class="btn so-btn btn-sm" type="button" data-bs-toggle="collapse" data-bs-target="#addSchedule">
              <i class="fa-solid fa-plus me-2"></i>Add Schedule
            </button>
          </div>
          <div class="so-card-body">

            <div class="collapse mb-3" id="addSchedule">
              <form method="post" action="<?= app_path('admin/repo_schedule_save.php') ?>" class="row g-2">
                <input type="hidden" name="_csrf" value="<?= h(csrf_token()) ?>">
                <input type="hidden" name="applies_to" value="company">

                <div class="col-12">
                  <label class="form-label fw-bold">Schedule Name</label>
                  <input type="text" name="name" class="form-control" style="border-radius:12px;" required maxlength="190" placeholder="Company Default Schedule">
                </div>

                <div class="col-12 col-md-4">
                  <label class="form-label fw-bold">Included Miles</label>
                  <input type="number" name="included_miles" class="form-control" style="border-radius:12px;" required value="50" min="0">
                </div>
                <div class="col-12 col-md-4">
                  <label class="form-label fw-bold">Max Billable Miles</label>
                  <input type="number" name="max_billable_miles" class="form-control" style="border-radius:12px;" required value="100" min="0">
                </div>
                <div class="col-12 col-md-4">
                  <label class="form-label fw-bold">Per Mile Fee</label>
                  <input type="number" step="0.01" name="per_mile_fee" class="form-control" style="border-radius:12px;" required value="3.00" min="0">
                </div>

                <div class="col-12">
                  <div class="form-check">
                    <input class="form-check-input" type="checkbox" name="is_active" value="1" id="sched_active" checked>
                    <label class="form-check-label fw-bold" for="sched_active">Active</label>
                  </div>
                </div>

                <div class="col-12">
                  <button class="btn btn-dark" style="border-radius:12px;" type="submit">
                    <i class="fa-solid fa-plus me-2"></i>Create Schedule
                  </button>
                </div>
              </form>
              <hr>
            </div>

            <?php if (!$schedules): ?>
              <div class="text-muted">No schedules yet.</div>
            <?php else: ?>
              <div class="accordion" id="schedAcc">
                <?php foreach ($schedules as $idx => $s): ?>
                  <?php if ((string)$s['applies_to'] !== 'company') continue; ?>
                  <?php $sid = (int)$s['id']; ?>
                  <div class="accordion-item" style="border-radius:14px; overflow:hidden; border:1px solid rgba(18,20,24,0.10); margin-bottom:10px;">
                    <h2 class="accordion-header">
                      <button class="accordion-button <?= $idx===0?'':'collapsed' ?>" type="button" data-bs-toggle="collapse" data-bs-target="#sched<?= $sid ?>">
                        <strong><?= h((string)$s['name']) ?></strong>
                        <span class="ms-3 text-muted small">
                          Included <?= (int)$s['included_miles'] ?>mi · Max <?= (int)$s['max_billable_miles'] ?>mi · $<?= number_format((float)$s['per_mile_fee'],2) ?>/mi
                        </span>
                      </button>
                    </h2>
                    <div id="sched<?= $sid ?>" class="accordion-collapse collapse <?= $idx===0?'show':'' ?>" data-bs-parent="#schedAcc">
                      <div class="accordion-body">
                        <form method="post" action="<?= app_path('admin/repo_schedule_save.php') ?>" class="row g-2">
                          <input type="hidden" name="_csrf" value="<?= h(csrf_token()) ?>">
                          <input type="hidden" name="id" value="<?= $sid ?>">
                          <input type="hidden" name="applies_to" value="company">

                          <div class="col-12 col-md-6">
                            <label class="form-label fw-bold">Name</label>
                            <input type="text" name="name" class="form-control" style="border-radius:12px;" value="<?= h((string)$s['name']) ?>" required maxlength="190">
                          </div>
                          <div class="col-6 col-md-2">
                            <label class="form-label fw-bold">Included</label>
                            <input type="number" name="included_miles" class="form-control" style="border-radius:12px;" value="<?= (int)$s['included_miles'] ?>" min="0" required>
                          </div>
                          <div class="col-6 col-md-2">
                            <label class="form-label fw-bold">Max</label>
                            <input type="number" name="max_billable_miles" class="form-control" style="border-radius:12px;" value="<?= (int)$s['max_billable_miles'] ?>" min="0" required>
                          </div>
                          <div class="col-12 col-md-2">
                            <label class="form-label fw-bold">Per Mile</label>
                            <input type="number" step="0.01" name="per_mile_fee" class="form-control" style="border-radius:12px;" value="<?= h((string)$s['per_mile_fee']) ?>" min="0" required>
                          </div>

                          <div class="col-12">
                            <div class="form-check">
                              <input class="form-check-input" type="checkbox" name="is_active" value="1" id="act<?= $sid ?>" <?= ((int)$s['is_active']===1)?'checked':'' ?>>
                              <label class="form-check-label fw-bold" for="act<?= $sid ?>">Active</label>
                            </div>
                          </div>

                          <div class="col-12 d-flex gap-2">
                            <button class="btn btn-dark" style="border-radius:12px;" type="submit">
                              <i class="fa-solid fa-floppy-disk me-2"></i>Save Schedule
                            </button>
                          </div>
                        </form>

                        <hr>

                        <div class="d-flex justify-content-between align-items-center">
                          <div class="fw-bold">Width Rules</div>
                          <button class="btn so-btn btn-sm" type="button" data-bs-toggle="collapse" data-bs-target="#addRule<?= $sid ?>">
                            <i class="fa-solid fa-plus me-2"></i>Add Width Rule
                          </button>
                        </div>

                        <div class="collapse mt-2" id="addRule<?= $sid ?>">
                          <form method="post" action="<?= app_path('admin/repo_width_rule_save.php') ?>" class="row g-2">
                            <input type="hidden" name="_csrf" value="<?= h(csrf_token()) ?>">
                            <input type="hidden" name="schedule_id" value="<?= $sid ?>">

                            <div class="col-6 col-md-2">
                              <label class="form-label fw-bold">Width ft</label>
                              <input type="number" name="width_ft" class="form-control" style="border-radius:12px;" required min="1" value="12">
                            </div>
                            <div class="col-6 col-md-2">
                              <label class="form-label fw-bold">Base Fee</label>
                              <input type="number" step="0.01" name="base_fee" class="form-control" style="border-radius:12px;" required min="0" value="500.00">
                            </div>
                            <div class="col-6 col-md-2">
                              <label class="form-label fw-bold">Base Len Max</label>
                              <input type="number" name="base_length_max_ft" class="form-control" style="border-radius:12px;" required min="1" value="32">
                            </div>
                            <div class="col-6 col-md-3">
                              <label class="form-label fw-bold">Extra Step ft</label>
                              <input type="number" name="extra_length_step_ft" class="form-control" style="border-radius:12px;" required min="1" value="1">
                            </div>
                            <div class="col-12 col-md-3">
                              <label class="form-label fw-bold">Fee per Step</label>
                              <input type="number" step="0.01" name="extra_length_fee_per_step" class="form-control" style="border-radius:12px;" required min="0" value="50.00">
                            </div>

                            <div class="col-12">
                              <button class="btn btn-dark" style="border-radius:12px;" type="submit">
                                <i class="fa-solid fa-plus me-2"></i>Add Width Rule
                              </button>
                            </div>
                          </form>
                          <hr>
                        </div>

                        <?php $rules = $rulesBySchedule[$sid] ?? []; ?>
                        <?php if (!$rules): ?>
                          <div class="text-muted">No width rules yet.</div>
                        <?php else: ?>
                          <div class="table-responsive">
                            <table class="table so-table align-middle">
                              <thead>
                                <tr>
                                  <th>Width</th>
                                  <th>Base Fee</th>
                                  <th>Base Len Max</th>
                                  <th>Step ft</th>
                                  <th>Fee/Step</th>
                                </tr>
                              </thead>
                              <tbody>
                                <?php foreach ($rules as $r): ?>
                                  <tr>
                                    <td><strong><?= (int)$r['width_ft'] ?> ft</strong></td>
                                    <td>$<?= number_format((float)$r['base_fee'],2) ?></td>
                                    <td><?= (int)$r['base_length_max_ft'] ?> ft</td>
                                    <td><?= (int)$r['extra_length_step_ft'] ?> ft</td>
                                    <td>$<?= number_format((float)$r['extra_length_fee_per_step'],2) ?></td>
                                  </tr>
                                <?php endforeach; ?>
                              </tbody>
                            </table>
                          </div>
                        <?php endif; ?>

                      </div>
                    </div>
                  </div>
                <?php endforeach; ?>
              </div>
            <?php endif; ?>

          </div>
        </div>
      </div>
    </div>

  <?php elseif ($tab === 'rto'): ?>
    <!-- RTO OVERRIDES -->
    <div class="so-card">
      <div class="so-card-h">
        <h5 class="so-card-title"><i class="fa-solid fa-building-user me-2" style="color:#cf1f1f"></i>RTO Overrides</h5>
      </div>
      <div class="so-card-body">
        <div class="text-muted mb-3">
          Create schedules and add-on fee types that apply to a specific RTO tenant. These override company defaults.
        </div>

        <?php if (!$linkedRTOs): ?>
          <div class="alert alert-warning mb-0">No RTO tenants linked yet.</div>
        <?php else: ?>
          <form method="post" action="<?= app_path('admin/repo_schedule_save.php') ?>" class="row g-2 mb-4">
            <input type="hidden" name="_csrf" value="<?= h(csrf_token()) ?>">
            <input type="hidden" name="applies_to" value="rto">

            <div class="col-12 col-md-4">
              <label class="form-label fw-bold">RTO Tenant</label>
              <select name="rto_tenant_id" class="form-select" style="border-radius:12px;" required>
                <option value="">— Select RTO —</option>
                <?php foreach ($linkedRTOs as $rt): ?>
                  <option value="<?= (int)$rt['id'] ?>"><?= h((string)$rt['name']) ?></option>
                <?php endforeach; ?>
              </select>
            </div>

            <div class="col-12 col-md-4">
              <label class="form-label fw-bold">Schedule Name</label>
              <input type="text" name="name" class="form-control" style="border-radius:12px;" required maxlength="190" placeholder="RTO Special Schedule">
            </div>

            <div class="col-6 col-md-1">
              <label class="form-label fw-bold">Incl</label>
              <input type="number" name="included_miles" class="form-control" style="border-radius:12px;" required value="50" min="0">
            </div>
            <div class="col-6 col-md-1">
              <label class="form-label fw-bold">Max</label>
              <input type="number" name="max_billable_miles" class="form-control" style="border-radius:12px;" required value="100" min="0">
            </div>
            <div class="col-12 col-md-2">
              <label class="form-label fw-bold">Per Mile</label>
              <input type="number" step="0.01" name="per_mile_fee" class="form-control" style="border-radius:12px;" required value="3.00" min="0">
            </div>

            <div class="col-12">
              <div class="form-check">
                <input class="form-check-input" type="checkbox" name="is_active" value="1" id="rto_sched_active" checked>
                <label class="form-check-label fw-bold" for="rto_sched_active">Active</label>
              </div>
            </div>

            <div class="col-12">
              <button class="btn btn-dark" style="border-radius:12px;" type="submit">
                <i class="fa-solid fa-plus me-2"></i>Create RTO Schedule
              </button>
            </div>
          </form>

          <div class="text-muted">RTO schedules created will be used automatically when a repo has that RTO tenant.</div>
        <?php endif; ?>
      </div>
    </div>

  <?php else: ?>
    <!-- ADD-ONS -->
    <div class="so-card">
      <div class="so-card-h">
        <h5 class="so-card-title"><i class="fa-solid fa-tag me-2" style="color:#cf1f1f"></i>Add-on Fee Types</h5>
      </div>
      <div class="so-card-body">

        <form method="post" action="<?= app_path('admin/repo_addon_save.php') ?>" class="row g-2 mb-4">
          <input type="hidden" name="_csrf" value="<?= h(csrf_token()) ?>">

          <div class="col-12 col-md-3">
            <label class="form-label fw-bold">Applies To</label>
            <select name="applies_to" class="form-select" style="border-radius:12px;" required>
              <option value="company">Company</option>
              <option value="rto">RTO</option>
            </select>
          </div>

          <div class="col-12 col-md-3">
            <label class="form-label fw-bold">RTO Tenant (if RTO)</label>
            <select name="rto_tenant_id" class="form-select" style="border-radius:12px;">
              <option value="">—</option>
              <?php foreach ($linkedRTOs as $rt): ?>
                <option value="<?= (int)$rt['id'] ?>"><?= h((string)$rt['name']) ?></option>
              <?php endforeach; ?>
            </select>
          </div>

          <div class="col-12 col-md-6">
            <label class="form-label fw-bold">Name</label>
            <input type="text" name="name" class="form-control" style="border-radius:12px;" required maxlength="190" placeholder="Electrical Disconnect Fee">
          </div>

          <div class="col-12">
            <label class="form-label fw-bold">Description</label>
            <input type="text" name="description" class="form-control" style="border-radius:12px;" maxlength="500">
          </div>

          <div class="col-6 col-md-2">
            <label class="form-label fw-bold">Default $</label>
            <input type="number" step="0.01" name="default_amount" class="form-control" style="border-radius:12px;" required min="0" value="0.00">
          </div>
          <div class="col-6 col-md-2">
            <label class="form-label fw-bold">Max $</label>
            <input type="number" step="0.01" name="max_amount" class="form-control" style="border-radius:12px;" required min="0" value="0.00">
            <div class="text-muted small mt-1">0 = no override</div>
          </div>

          <div class="col-12 col-md-4">
            <div class="form-check mt-4">
              <input class="form-check-input" type="checkbox" name="driver_can_add" value="1" id="dadd">
              <label class="form-check-label fw-bold" for="dadd">Driver can add</label>
            </div>
          </div>

          <div class="col-12 col-md-4">
            <div class="form-check mt-4">
              <input class="form-check-input" type="checkbox" name="driver_can_override_amount" value="1" id="dovr">
              <label class="form-check-label fw-bold" for="dovr">Driver can override amount (up to max)</label>
            </div>
          </div>

          <div class="col-12">
            <div class="form-check">
              <input class="form-check-input" type="checkbox" name="is_active" value="1" id="aact" checked>
              <label class="form-check-label fw-bold" for="aact">Active</label>
            </div>
          </div>

          <div class="col-12">
            <button class="btn btn-dark" style="border-radius:12px;" type="submit">
              <i class="fa-solid fa-plus me-2"></i>Add Fee Type
            </button>
          </div>
        </form>

        <hr>

        <?php if (!$addons): ?>
          <div class="text-muted">No add-on types yet.</div>
        <?php else: ?>
          <div class="table-responsive">
            <table class="table so-table align-middle">
              <thead>
                <tr>
                  <th>Name</th>
                  <th>Applies</th>
                  <th>Default</th>
                  <th>Max</th>
                  <th>Driver Add</th>
                  <th>Driver Override</th>
                  <th>Active</th>
                </tr>
              </thead>
              <tbody>
                <?php foreach ($addons as $a): ?>
                  <tr>
                    <td><strong><?= h((string)$a['name']) ?></strong><div class="text-muted small"><?= h((string)($a['description'] ?? '')) ?></div></td>
                    <td><?= h((string)$a['applies_to']) ?><?= !empty($a['rto_tenant_id']) ? (' · RTO#'.(int)$a['rto_tenant_id']) : '' ?></td>
                    <td>$<?= number_format((float)$a['default_amount'],2) ?></td>
                    <td>$<?= number_format((float)$a['max_amount'],2) ?></td>
                    <td><?= ((int)$a['driver_can_add']===1)?'Yes':'No' ?></td>
                    <td><?= ((int)$a['driver_can_override_amount']===1)?'Yes':'No' ?></td>
                    <td><?= ((int)$a['is_active']===1)?'Yes':'No' ?></td>
                  </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>
        <?php endif; ?>

      </div>
    </div>
  <?php endif; ?>

</div>
<?php require_once __DIR__ . '/../includes/footer.php'; ?>
