<?php
// /app/includes/auth.php
declare(strict_types=1);

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';

function is_https_request(): bool {
  // Direct HTTPS
  if (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') return true;

  // Common proxy/CDN headers
  $xfp = strtolower((string)($_SERVER['HTTP_X_FORWARDED_PROTO'] ?? ''));
  if ($xfp === 'https') return true;

  $xfs = strtolower((string)($_SERVER['HTTP_X_FORWARDED_SSL'] ?? ''));
  if ($xfs === 'on') return true;

  // Some servers set this
  if ((string)($_SERVER['SERVER_PORT'] ?? '') === '443') return true;

  return false;
}

function start_secure_session(): void {
  if (session_status() === PHP_SESSION_ACTIVE) return;

  $secure = is_https_request();

  session_name(SESSION_NAME);

  // Use "/" so the session works whether you're at /app or later at app.shedoffice.com/
  session_set_cookie_params([
    'lifetime' => SESSION_LIFETIME_SECONDS,
    'path' => '/',
    'domain' => '',
    'secure' => $secure,
    'httponly' => true,
    'samesite' => 'Lax',
  ]);

  ini_set('session.use_strict_mode', '1');
  ini_set('session.cookie_httponly', '1');

  // IMPORTANT: only set cookie_secure when request is truly https
  if ($secure) {
    ini_set('session.cookie_secure', '1');
  } else {
    ini_set('session.cookie_secure', '0');
  }

  session_start();

  // Sliding expiration markers (not enforcing here, just recording)
  if (!isset($_SESSION['_created'])) $_SESSION['_created'] = time();
  $_SESSION['_last_seen'] = time();
}

function csrf_token(): string {
  start_secure_session();
  if (empty($_SESSION['_csrf'])) {
    $_SESSION['_csrf'] = bin2hex(random_bytes(32));
  }
  return $_SESSION['_csrf'];
}

function csrf_validate(?string $token): bool {
  start_secure_session();
  return is_string($token) && !empty($_SESSION['_csrf']) && hash_equals($_SESSION['_csrf'], $token);
}

function is_logged_in(): bool {
  start_secure_session();
  return !empty($_SESSION['auth']['user_id']);
}

function current_user_id(): ?int {
  start_secure_session();
  return isset($_SESSION['auth']['user_id']) ? (int)$_SESSION['auth']['user_id'] : null;
}

function current_tenant_id(): ?int {
  start_secure_session();
  return isset($_SESSION['auth']['tenant_id']) ? (int)$_SESSION['auth']['tenant_id'] : null;
}

function current_tenant_type(): ?string {
  start_secure_session();
  return $_SESSION['auth']['tenant_type'] ?? null; // 'company'|'rto'
}

function active_company_tenant_id(): ?int {
  start_secure_session();
  return isset($_SESSION['auth']['active_company_tenant_id']) && $_SESSION['auth']['active_company_tenant_id']
    ? (int)$_SESSION['auth']['active_company_tenant_id']
    : null;
}

function require_login(): void {
  start_secure_session();
  if (!is_logged_in()) {
    header('Location: ' . app_path('index.php'));
    exit;
  }
}

/**
 * For RTO users, they must select an active company tenant context.
 */
function require_company_context_if_rto(): void {
  require_login();
  if (current_tenant_type() === 'rto' && !active_company_tenant_id()) {
    header('Location: ' . app_path('select_company.php'));
    exit;
  }
}

/**
 * Basic IP-based login throttling.
 */
function login_throttle_key(): string {
  $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
  return 'login_throttle_' . sha1($ip);
}

function login_is_locked(): bool {
  start_secure_session();
  $k = login_throttle_key();
  $state = $_SESSION[$k] ?? null;
  if (!is_array($state)) return false;

  $lockedUntil = $state['locked_until'] ?? 0;
  return (time() < (int)$lockedUntil);
}

function login_register_failure(): void {
  start_secure_session();
  $k = login_throttle_key();
  $state = $_SESSION[$k] ?? ['attempts' => 0, 'locked_until' => 0];
  $state['attempts'] = (int)$state['attempts'] + 1;

  if ($state['attempts'] >= LOGIN_MAX_ATTEMPTS) {
    $state['locked_until'] = time() + (LOGIN_LOCK_MINUTES * 60);
  }
  $_SESSION[$k] = $state;
}

function login_clear_throttle(): void {
  start_secure_session();
  unset($_SESSION[login_throttle_key()]);
}

function auth_login_user(array $userRow, array $tenantRow): void {
  start_secure_session();
  session_regenerate_id(true);

  $_SESSION['auth'] = [
    'user_id' => (int)$userRow['id'],
    'tenant_id' => (int)$userRow['tenant_id'],
    'tenant_type' => (string)$tenantRow['tenant_type'],
    'user_email' => (string)$userRow['email'],
    'user_name' => trim(($userRow['first_name'] ?? '') . ' ' . ($userRow['last_name'] ?? '')) ?: ($userRow['email'] ?? 'User'),
    // only used when tenant_type === 'rto'
    'active_company_tenant_id' => null,
    'active_company_name' => null,

    // ACL cache flags
    'roles' => [],
    'perm_allow' => [],
    'perm_deny' => [],
    'acl_loaded' => 0,
  ];
}

function auth_logout(): void {
  start_secure_session();

  $_SESSION = [];

  if (ini_get("session.use_cookies")) {
    $params = session_get_cookie_params();
    setcookie(session_name(), '', time() - 42000,
      $params["path"], $params["domain"], $params["secure"], $params["httponly"]
    );
  }

  session_destroy();
}

function current_user(): array {
  start_secure_session();
  return [
    'id' => $_SESSION['auth']['user_id'] ?? null,
    'email' => $_SESSION['auth']['user_email'] ?? null,
    'name' => $_SESSION['auth']['user_name'] ?? 'User',
    'tenant_type' => $_SESSION['auth']['tenant_type'] ?? null,
    'active_company_name' => $_SESSION['auth']['active_company_name'] ?? null,
  ];
}
