<?php
// /app/operations/workorder_view.php
declare(strict_types=1);



require_once __DIR__ . '/_bootstrap.php';
require_once __DIR__ . '/includes/bootstrap.php';
require_once __DIR__ . '/_lib.php';

$pageTitle = 'Work Order';
$activeNav = 'operations';

$companyId = ops_require_company_scope();
if (!ops_can_view()) { http_response_code(403); exit('Forbidden'); }

$pdo = db_safe();
$canManage = ops_can_manage();

$idRaw = (string)($_GET['id'] ?? '0');
$isNew = ($idRaw === 'new');
$id = $isNew ? 0 : (int)$idRaw;

if (!$isNew && $id <= 0) { http_response_code(400); exit('Invalid id'); }

/* Load services */
$svcStmt = $pdo->prepare("SELECT id, service_name FROM ops_services WHERE company_id=:cid AND is_active=1 ORDER BY service_name");
$svcStmt->execute([':cid'=>$companyId]);
$services = $svcStmt->fetchAll(PDO::FETCH_ASSOC);
$serviceMap = [];
foreach ($services as $s) $serviceMap[(int)$s['id']] = (string)$s['service_name'];

$wo = null;

if (!$isNew) {
  $stmt = $pdo->prepare("
    SELECT wo.*, s.service_name
    FROM ops_work_orders wo
    JOIN ops_services s ON s.id = wo.service_id
    WHERE wo.id=:id AND wo.company_id=:cid
  ");
  $stmt->execute([':id'=>$id, ':cid'=>$companyId]);
  $wo = $stmt->fetch(PDO::FETCH_ASSOC);

  if (!$wo) { http_response_code(404); exit('Work order not found'); }

  if (!$canManage) {
    // Non-managers must be assigned to view
    $uid = function_exists('current_user_id') ? (int)current_user_id() : 0;
    $chk = $pdo->prepare("SELECT COUNT(*) FROM ops_work_order_assignees a WHERE a.work_order_id=:wo AND a.user_id=:uid");
    $chk->execute([':wo'=>$id, ':uid'=>$uid]);
    if ((int)$chk->fetchColumn() === 0) { http_response_code(403); exit('Forbidden'); }
  }
} else {
  $wo = [
    'id'=>0,
    'service_id'=>0,
    'status'=>'new',
    'priority'=>3,
    'serial_number'=>'',
    'customer_name'=>'',
    'customer_phone'=>'',
    'customer_email'=>'',
    'install_location'=>'',
    'address_line1'=>'',
    'address_line2'=>'',
    'city'=>'',
    'state'=>'',
    'postal_code'=>'',
    'scheduled_date'=>null,
    'due_date'=>null,
    'summary'=>'',
    'internal_notes'=>'',
    'customer_notes'=>'',
    'external_order_id'=>'',
    'external_order_number'=>'',
    'source'=>'manual',
    'created_at'=>null,
    'updated_at'=>null,
    'service_name'=>'',
  ];
}

/* Assignees */
$assignees = [];
$ass = null;
if (!$isNew) {
  $ass = $pdo->prepare("
    SELECT a.user_id, a.role, u.first_name, u.last_name
    FROM ops_work_order_assignees a
    LEFT JOIN users u ON u.id=a.user_id
    WHERE a.work_order_id=:wo
    ORDER BY FIELD(a.role,'lead','tech','helper'), a.assigned_at ASC
  ");
  $ass->execute([':wo'=>$id]);
  $assignees = $ass->fetchAll(PDO::FETCH_ASSOC);
}

$leadUserId = 0;
foreach ($assignees as $a) if (($a['role'] ?? '') === 'lead') { $leadUserId = (int)$a['user_id']; break; }

/* Tech list for assign dropdowns */
$techs = [];
try {
  $qTech = $pdo->prepare("
    SELECT id, CONCAT(first_name,' ',last_name) AS name
    FROM users
    WHERE tenant_type='company' AND tenant_id=:cid
      AND is_active=1
      AND role IN ('electrician','installer','repair_tech','tech','crew_lead')
    ORDER BY first_name, last_name
  ");
  $qTech->execute([':cid'=>$companyId]);
  $techs = $qTech->fetchAll(PDO::FETCH_KEY_PAIR);
} catch (Throwable $e) {
  $techs = $pdo->query("
    SELECT id, CONCAT(first_name,' ',last_name) AS name
    FROM users
    WHERE role IN ('electrician','installer','repair_tech','tech','crew_lead')
    ORDER BY first_name, last_name
  ")->fetchAll(PDO::FETCH_KEY_PAIR);
}

/* Tasks */
$tasks = [];
if (!$isNew) {
  $t = $pdo->prepare("SELECT * FROM ops_work_order_tasks WHERE work_order_id=:wo ORDER BY sort_order ASC, id ASC");
  $t->execute([':wo'=>$id]);
  $tasks = $t->fetchAll(PDO::FETCH_ASSOC);
}

/* Events */
$events = [];
if (!$isNew) {
  $ev = $pdo->prepare("
    SELECT e.*, u.first_name, u.last_name
    FROM ops_work_order_events e
    LEFT JOIN users u ON u.id=e.actor_user_id
    WHERE e.work_order_id=:wo
    ORDER BY e.created_at DESC
    LIMIT 50
  ");
  $ev->execute([':wo'=>$id]);
  $events = $ev->fetchAll(PDO::FETCH_ASSOC);
}

require_once __DIR__ . '/../includes/header.php';
$activeOps = 'workorders';
require __DIR__ . '/_subnav.php';

$badge = ops_status_badge((string)$wo['status']);
$title = $isNew ? 'New Work Order' : ('Work Order #'.(int)$wo['id']);
?>
<div class="d-flex flex-wrap align-items-start gap-2 mb-3">
  <div>
    <div class="h4 mb-0"><?=h($title)?></div>
    <?php if (!$isNew): ?>
      <div class="text-muted small">
        Created <?=h((string)date('M j, Y g:i A', strtotime((string)$wo['created_at'])))?>
        · Updated <?=h((string)date('M j, Y g:i A', strtotime((string)$wo['updated_at'])))?>
        <?php if (!empty($wo['external_order_number']) || !empty($wo['external_order_id'])): ?>
          · External Order: <?=h((string)($wo['external_order_number'] ?: $wo['external_order_id']))?>
        <?php endif; ?>
      </div>
    <?php endif; ?>
  </div>

  <div class="ms-auto d-flex flex-wrap gap-2">
    <a class="btn btn-outline-dark btn-sm" href="/app/operations/workorders_list.php">Back</a>
    <?php if (!$isNew): ?>
      <a class="btn btn-outline-secondary btn-sm" href="javascript:window.print()">Print</a>
    <?php endif; ?>
  </div>
</div>

<form method="post" action="/app/operations/workorder_save.php" class="row g-3">
  <input type="hidden" name="csrf" value="<?=h(csrf_token())?>">
  <input type="hidden" name="id" value="<?= (int)$wo['id'] ?>">

  <div class="col-12 col-lg-8">
    <div class="card shadow-sm">
      <div class="card-header bg-dark text-white d-flex justify-content-between">
        <strong>Details</strong>
        <span class="badge bg-<?=$badge?>"><?=h(ops_status_label((string)$wo['status']))?></span>
      </div>
      <div class="card-body">
        <div class="row g-2">
          <div class="col-12 col-md-6">
            <label class="form-label">Service</label>
            <select class="form-select" name="service_id" <?= $canManage ? '' : 'disabled' ?> required>
              <option value="">— Select Service —</option>
              <?php foreach ($services as $s): ?>
                <option value="<?= (int)$s['id'] ?>" <?= (int)$wo['service_id']===(int)$s['id']?'selected':'' ?>><?=h((string)$s['service_name'])?></option>
              <?php endforeach; ?>
            </select>
            <?php if (!$canManage): ?>
              <input type="hidden" name="service_id" value="<?= (int)$wo['service_id'] ?>">
            <?php endif; ?>
          </div>

          <div class="col-6 col-md-3">
            <label class="form-label">Status</label>
            <select class="form-select" name="status" <?= $canManage ? '' : 'disabled' ?>>
              <?php foreach (['new','scheduled','in_progress','blocked','completed','canceled'] as $st): ?>
                <option value="<?=$st?>" <?=$st===$wo['status']?'selected':''?>><?=h(ops_status_label($st))?></option>
              <?php endforeach; ?>
            </select>
            <?php if (!$canManage): ?>
              <input type="hidden" name="status" value="<?=h((string)$wo['status'])?>">
            <?php endif; ?>
          </div>

          <div class="col-6 col-md-3">
            <label class="form-label">Priority</label>
            <select class="form-select" name="priority" <?= $canManage ? '' : 'disabled' ?>>
              <?php for ($p=1;$p<=5;$p++): ?>
                <option value="<?=$p?>" <?=$p==(int)$wo['priority']?'selected':''?>><?=$p?><?= $p===1?' (Urgent)':($p===5?' (Low)':'') ?></option>
              <?php endfor; ?>
            </select>
            <?php if (!$canManage): ?>
              <input type="hidden" name="priority" value="<?= (int)$wo['priority'] ?>">
            <?php endif; ?>
          </div>

          <div class="col-12">
            <label class="form-label">Summary</label>
            <input class="form-control" name="summary" value="<?=h((string)$wo['summary'])?>" placeholder="Short description of the work order">
          </div>

          <div class="col-6 col-md-4">
            <label class="form-label">Serial #</label>
            <input class="form-control" name="serial_number" value="<?=h((string)$wo['serial_number'])?>">
          </div>

          <div class="col-6 col-md-4">
            <label class="form-label">Install Location</label>
            <input class="form-control" name="install_location" value="<?=h((string)$wo['install_location'])?>" placeholder="Plant / On-site / Hub, etc">
          </div>

          <div class="col-6 col-md-2">
            <label class="form-label">Scheduled</label>
            <input type="date" class="form-control" name="scheduled_date" value="<?=h((string)($wo['scheduled_date'] ?? ''))?>" <?= $canManage ? '' : 'disabled' ?>>
            <?php if (!$canManage): ?>
              <input type="hidden" name="scheduled_date" value="<?=h((string)($wo['scheduled_date'] ?? ''))?>">
            <?php endif; ?>
          </div>

          <div class="col-6 col-md-2">
            <label class="form-label">Due</label>
            <input type="date" class="form-control" name="due_date" value="<?=h((string)($wo['due_date'] ?? ''))?>" <?= $canManage ? '' : 'disabled' ?>>
            <?php if (!$canManage): ?>
              <input type="hidden" name="due_date" value="<?=h((string)($wo['due_date'] ?? ''))?>">
            <?php endif; ?>
          </div>

          <hr class="my-2">

          <div class="col-12 col-md-6">
            <label class="form-label">Customer Name</label>
            <input class="form-control" name="customer_name" value="<?=h((string)$wo['customer_name'])?>">
          </div>
          <div class="col-6 col-md-3">
            <label class="form-label">Phone</label>
            <input class="form-control" name="customer_phone" value="<?=h((string)$wo['customer_phone'])?>">
          </div>
          <div class="col-6 col-md-3">
            <label class="form-label">Email</label>
            <input class="form-control" name="customer_email" value="<?=h((string)$wo['customer_email'])?>">
          </div>

          <div class="col-12">
            <label class="form-label">Address</label>
            <input class="form-control mb-2" name="address_line1" value="<?=h((string)$wo['address_line1'])?>" placeholder="Street address">
            <input class="form-control" name="address_line2" value="<?=h((string)$wo['address_line2'])?>" placeholder="Apt, suite, etc (optional)">
          </div>

          <div class="col-6 col-md-4">
            <label class="form-label">City</label>
            <input class="form-control" name="city" value="<?=h((string)$wo['city'])?>">
          </div>
          <div class="col-3 col-md-4">
            <label class="form-label">State</label>
            <input class="form-control" name="state" value="<?=h((string)$wo['state'])?>">
          </div>
          <div class="col-3 col-md-4">
            <label class="form-label">ZIP</label>
            <input class="form-control" name="postal_code" value="<?=h((string)$wo['postal_code'])?>">
          </div>

          <div class="col-12">
            <label class="form-label">Internal Notes</label>
            <textarea class="form-control" name="internal_notes" rows="3"><?=h((string)$wo['internal_notes'])?></textarea>
          </div>

          <div class="col-12">
            <label class="form-label">Customer Notes</label>
            <textarea class="form-control" name="customer_notes" rows="3"><?=h((string)$wo['customer_notes'])?></textarea>
          </div>

        </div>
      </div>
      <div class="card-footer d-flex flex-wrap gap-2">
        <button class="btn btn-primary" type="submit"><?= $isNew ? 'Create Work Order' : 'Save Changes' ?></button>
        <?php if (!$isNew && $canManage): ?>
          <a class="btn btn-outline-danger" href="/app/operations/workorder_save.php?action=cancel&id=<?= (int)$wo['id'] ?>&csrf=<?=h(csrf_token())?>" onclick="return confirm('Cancel this work order?');">Cancel Work Order</a>
        <?php endif; ?>
      </div>
    </div>

    <?php if (!$isNew): ?>
    <div class="card shadow-sm mt-3">
      <div class="card-header bg-dark text-white"><strong>Tasks</strong></div>
      <div class="card-body">
        <div class="d-flex gap-2 mb-2">
          <input class="form-control form-control-sm" id="newTaskTitle" placeholder="Add a task (e.g., Run wire, Mount condenser, Seal penetrations)">
          <button type="button" class="btn btn-sm btn-primary" id="addTaskBtn">Add</button>
        </div>

        <?php if (!$tasks): ?>
          <div class="text-muted">No tasks yet.</div>
        <?php else: ?>
          <ul class="list-group" id="taskList">
            <?php foreach ($tasks as $t): ?>
              <li class="list-group-item d-flex align-items-center gap-2" data-task-id="<?= (int)$t['id'] ?>">
                <input class="form-check-input js-task-done" type="checkbox" <?= ($t['status'] === 'done')?'checked':''?>>
                <div class="flex-grow-1">
                  <div class="fw-semibold js-task-title"><?=h((string)$t['title'])?></div>
                  <div class="small text-muted">Status: <?=h((string)$t['status'])?></div>
                </div>
                <?php if ($canManage): ?>
                  <button class="btn btn-sm btn-outline-danger js-task-del" type="button">Delete</button>
                <?php endif; ?>
              </li>
            <?php endforeach; ?>
          </ul>
        <?php endif; ?>
      </div>
    </div>
    <?php endif; ?>
  </div>

  <div class="col-12 col-lg-4">
    <div class="card shadow-sm">
      <div class="card-header bg-dark text-white"><strong>Assignment</strong></div>
      <div class="card-body">
        <?php if ($isNew): ?>
          <div class="text-muted">Create the work order first, then assign installers and tasks.</div>
        <?php else: ?>
          <label class="form-label">Lead Installer</label>
          <select class="form-select mb-3 <?= $canManage ? '' : 'disabled' ?>" id="leadSelect" <?= $canManage ? '' : 'disabled' ?>>
            <option value="">— Unassigned —</option>
            <?php foreach ($techs as $uid=>$name): ?>
              <option value="<?= (int)$uid ?>" <?= $leadUserId===(int)$uid?'selected':'' ?>><?=h((string)$name)?></option>
            <?php endforeach; ?>
          </select>

          <div class="small text-muted mb-2">Assigned Team</div>
          <div id="assigneeList">
            <?php if (!$assignees): ?>
              <div class="text-muted">No assignees yet.</div>
            <?php else: ?>
              <?php foreach ($assignees as $a): ?>
                <div class="d-flex justify-content-between align-items-center border rounded-3 p-2 mb-2">
                  <div>
                    <div class="fw-semibold"><?=h(trim((string)($a['first_name'] ?? '').' '.(string)($a['last_name'] ?? '')) ?: ('User #'.(int)$a['user_id']))?></div>
                    <div class="small text-muted">Role: <?=h((string)$a['role'])?></div>
                  </div>
                </div>
              <?php endforeach; ?>
            <?php endif; ?>
          </div>

          <?php if ($canManage): ?>
            <div class="border-top pt-3 mt-3">
              <label class="form-label">Add Team Member</label>
              <div class="d-flex gap-2">
                <select class="form-select" id="addAssigneeUser">
                  <option value="">— Select —</option>
                  <?php foreach ($techs as $uid=>$name): ?>
                    <option value="<?= (int)$uid ?>"><?=h((string)$name)?></option>
                  <?php endforeach; ?>
                </select>
                <select class="form-select" id="addAssigneeRole" style="max-width:140px;">
                  <option value="tech">Tech</option>
                  <option value="helper">Helper</option>
                </select>
                <button class="btn btn-outline-primary" type="button" id="addAssigneeBtn">Add</button>
              </div>
            </div>
          <?php endif; ?>
        <?php endif; ?>
      </div>
    </div>

    <?php if (!$isNew): ?>
    <div class="card shadow-sm mt-3">
      <div class="card-header"><strong>Recent Activity</strong></div>
      <div class="card-body">
        <?php if (!$events): ?>
          <div class="text-muted">No activity yet.</div>
        <?php else: ?>
          <div class="list-group list-group-flush">
            <?php foreach ($events as $e): ?>
              <?php
                $actor = trim((string)($e['first_name'] ?? '').' '.(string)($e['last_name'] ?? ''));
                $when = date('M j, Y g:i A', strtotime((string)$e['created_at']));
              ?>
              <div class="list-group-item">
                <div class="small text-muted"><?=h($when)?><?= $actor ? ' · '.h($actor) : '' ?></div>
                <div class="fw-semibold"><?=h((string)$e['event_type'])?></div>
                <?php if (!empty($e['message'])): ?><div><?=h((string)$e['message'])?></div><?php endif; ?>
              </div>
            <?php endforeach; ?>
          </div>
        <?php endif; ?>
      </div>
    </div>
    <?php endif; ?>
  </div>
</form>

<?php if (!$isNew): ?>
<script>
(function(){
  const csrf = <?= json_encode(csrf_token()) ?>;
  const workOrderId = <?= (int)$wo['id'] ?>;
  const canManage = <?= $canManage ? 'true':'false' ?>;

  async function post(url, data){
    const body = new URLSearchParams(data);
    const res = await fetch(url, { method:'POST', headers:{'Content-Type':'application/x-www-form-urlencoded'}, body: body.toString() });
    const text = await res.text();
    let j = null;
    try { j = JSON.parse(text); } catch(e) {}
    if (!res.ok) throw new Error((j && j.msg) ? j.msg : text || ('HTTP '+res.status));
    if (!j || !j.ok) throw new Error((j && j.msg) ? j.msg : 'Request failed');
    return j;
  }

  // Lead assignment
  const leadSelect = document.getElementById('leadSelect');
  if (leadSelect && canManage) {
    leadSelect.addEventListener('change', async ()=>{
      leadSelect.disabled = true;
      try {
        await post('/app/operations/workorder_team_ajax.php', { action:'set_lead', work_order_id: workOrderId, user_id: leadSelect.value, csrf });
        location.reload();
      } catch (e) { alert(e.message); }
      finally { leadSelect.disabled = false; }
    });
  }

  // Add assignee
  const addBtn = document.getElementById('addAssigneeBtn');
  if (addBtn) {
    addBtn.addEventListener('click', async ()=>{
      const user = document.getElementById('addAssigneeUser').value;
      const role = document.getElementById('addAssigneeRole').value;
      if (!user) return alert('Select a user');
      addBtn.disabled = true;
      try {
        await post('/app/operations/workorder_team_ajax.php', { action:'add', work_order_id: workOrderId, user_id: user, role, csrf });
        location.reload();
      } catch (e) { alert(e.message); }
      finally { addBtn.disabled = false; }
    });
  }

  // Tasks
  const addTaskBtn = document.getElementById('addTaskBtn');
  if (addTaskBtn) {
    addTaskBtn.addEventListener('click', async ()=>{
      const title = document.getElementById('newTaskTitle').value.trim();
      if (!title) return;
      addTaskBtn.disabled = true;
      try {
        await post('/app/operations/workorder_tasks_ajax.php', { action:'add', work_order_id: workOrderId, title, csrf });
        location.reload();
      } catch(e) { alert(e.message); }
      finally { addTaskBtn.disabled = false; }
    });
  }

  document.addEventListener('click', async (e)=>{
    const btn = e.target;
    if (btn.classList.contains('js-task-del')) {
      if (!confirm('Delete this task?')) return;
      const li = btn.closest('[data-task-id]');
      const taskId = li ? li.getAttribute('data-task-id') : '';
      if (!taskId) return;
      btn.disabled = true;
      try {
        await post('/app/operations/workorder_tasks_ajax.php', { action:'delete', work_order_id: workOrderId, task_id: taskId, csrf });
        location.reload();
      } catch(err) { alert(err.message); }
      finally { btn.disabled = false; }
    }
  });

  document.addEventListener('change', async (e)=>{
    const el = e.target;
    if (el.classList.contains('js-task-done')) {
      const li = el.closest('[data-task-id]');
      const taskId = li ? li.getAttribute('data-task-id') : '';
      if (!taskId) return;
      el.disabled = true;
      try {
        await post('/app/operations/workorder_tasks_ajax.php', { action:'toggle', work_order_id: workOrderId, task_id: taskId, done: el.checked ? '1':'0', csrf });
      } catch(err) { alert(err.message); el.checked = !el.checked; }
      finally { el.disabled = false; }
    }
  });
})();
</script>
<?php endif; ?>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
