<?php
// /app/operations/workorder_save.php
declare(strict_types=1);



require_once __DIR__ . '/_bootstrap.php';
require_once __DIR__ . '/includes/bootstrap.php';
require_once __DIR__ . '/_lib.php';

$companyId = ops_require_company_scope();
if (!ops_can_manage()) { http_response_code(403); exit('Forbidden'); }

$pdo = db_safe();

$action = (string)($_GET['action'] ?? '');

if ($action === 'cancel') {
  // CSRF via query for one-click link
  if (session_status() !== PHP_SESSION_ACTIVE) session_start();
  $tok = (string)($_GET['csrf'] ?? '');
  if (empty($_SESSION['csrf_ops']) || !hash_equals((string)$_SESSION['csrf_ops'], $tok)) { http_response_code(419); exit('Invalid CSRF'); }

  $id = (int)($_GET['id'] ?? 0);
  if ($id <= 0) { http_response_code(400); exit('Missing id'); }

  $pdo->prepare("UPDATE ops_work_orders SET status='canceled' WHERE id=:id AND company_id=:cid")
      ->execute([':id'=>$id, ':cid'=>$companyId]);

  $pdo->prepare("INSERT INTO ops_work_order_events (work_order_id, actor_user_id, event_type, message) VALUES (:wo,:uid,'canceled','Work order canceled')")
      ->execute([':wo'=>$id, ':uid'=> function_exists('current_user_id') ? (int)current_user_id() : null]);

  header('Location: /app/operations/workorder_view.php?id='.$id);
  exit;
}

csrf_verify();

$id = (int)($_POST['id'] ?? 0);
$isNew = ($id <= 0);

$serviceId = (int)($_POST['service_id'] ?? 0);
$status = (string)($_POST['status'] ?? 'new');
$priority = (int)($_POST['priority'] ?? 3);

$allowedStatus = ['new','scheduled','in_progress','blocked','completed','canceled'];
if (!in_array($status, $allowedStatus, true)) $status = 'new';
$priority = max(1, min(5, $priority));

$fields = [
  'serial_number'   => trim((string)($_POST['serial_number'] ?? '')),
  'customer_name'   => trim((string)($_POST['customer_name'] ?? '')),
  'customer_phone'  => trim((string)($_POST['customer_phone'] ?? '')),
  'customer_email'  => trim((string)($_POST['customer_email'] ?? '')),
  'install_location'=> trim((string)($_POST['install_location'] ?? '')),
  'address_line1'   => trim((string)($_POST['address_line1'] ?? '')),
  'address_line2'   => trim((string)($_POST['address_line2'] ?? '')),
  'city'            => trim((string)($_POST['city'] ?? '')),
  'state'           => trim((string)($_POST['state'] ?? '')),
  'postal_code'     => trim((string)($_POST['postal_code'] ?? '')),
  'scheduled_date'  => ($_POST['scheduled_date'] ?? '') ?: null,
  'due_date'        => ($_POST['due_date'] ?? '') ?: null,
  'summary'         => trim((string)($_POST['summary'] ?? '')),
  'internal_notes'  => (string)($_POST['internal_notes'] ?? ''),
  'customer_notes'  => (string)($_POST['customer_notes'] ?? ''),
];

if ($serviceId <= 0) { http_response_code(400); exit('Service is required'); }

$svc = $pdo->prepare("SELECT * FROM ops_services WHERE id=:id AND company_id=:cid AND is_active=1");
$svc->execute([':id'=>$serviceId, ':cid'=>$companyId]);
$service = $svc->fetch(PDO::FETCH_ASSOC);
if (!$service) { http_response_code(400); exit('Invalid service'); }

/* Auto due date if missing */
if (empty($fields['due_date'])) {
  $days = (int)($service['default_due_days'] ?? 7);
  $base = new DateTime('now', new DateTimeZone('UTC'));
  $base->modify('+'.$days.' days');
  $fields['due_date'] = $base->format('Y-m-d');
}

$actor = function_exists('current_user_id') ? (int)current_user_id() : null;

if ($isNew) {
  $ins = $pdo->prepare("
    INSERT INTO ops_work_orders
      (company_id, service_id, status, priority, source, summary,
       serial_number, customer_name, customer_phone, customer_email,
       install_location, address_line1, address_line2, city, state, postal_code,
       scheduled_date, due_date, internal_notes, customer_notes)
    VALUES
      (:cid, :sid, :st, :prio, 'manual', :summary,
       :serial_number, :customer_name, :customer_phone, :customer_email,
       :install_location, :address_line1, :address_line2, :city, :state, :postal_code,
       :scheduled_date, :due_date, :internal_notes, :customer_notes)
  ");

  $ins->execute([
    ':cid'=>$companyId,
    ':sid'=>$serviceId,
    ':st'=>$status,
    ':prio'=>$priority,
    ':summary'=>$fields['summary'],
    ':serial_number'=>$fields['serial_number'] ?: null,
    ':customer_name'=>$fields['customer_name'] ?: null,
    ':customer_phone'=>$fields['customer_phone'] ?: null,
    ':customer_email'=>$fields['customer_email'] ?: null,
    ':install_location'=>$fields['install_location'] ?: null,
    ':address_line1'=>$fields['address_line1'] ?: null,
    ':address_line2'=>$fields['address_line2'] ?: null,
    ':city'=>$fields['city'] ?: null,
    ':state'=>$fields['state'] ?: null,
    ':postal_code'=>$fields['postal_code'] ?: null,
    ':scheduled_date'=>$fields['scheduled_date'],
    ':due_date'=>$fields['due_date'],
    ':internal_notes'=>$fields['internal_notes'] ?: null,
    ':customer_notes'=>$fields['customer_notes'] ?: null,
  ]);

  $newId = (int)$pdo->lastInsertId();

  // Default lead assignee (service-level) if exists
  if (!empty($service['default_assignee_user_id'])) {
    $pdo->prepare("INSERT IGNORE INTO ops_work_order_assignees (work_order_id, user_id, role) VALUES (:wo,:uid,'lead')")
        ->execute([':wo'=>$newId, ':uid'=>(int)$service['default_assignee_user_id']]);
  }

  $pdo->prepare("INSERT INTO ops_work_order_events (work_order_id, actor_user_id, event_type, message) VALUES (:wo,:uid,'created',:msg)")
      ->execute([':wo'=>$newId, ':uid'=>$actor, ':msg'=>'Work order created']);

  header('Location: /app/operations/workorder_view.php?id='.$newId);
  exit;
}

/* Update */
$existing = $pdo->prepare("SELECT * FROM ops_work_orders WHERE id=:id AND company_id=:cid");
$existing->execute([':id'=>$id, ':cid'=>$companyId]);
$old = $existing->fetch(PDO::FETCH_ASSOC);
if (!$old) { http_response_code(404); exit('Work order not found'); }

$completedAt = $old['completed_at'] ?? null;
if ($status === 'completed' && empty($completedAt)) {
  $completedAt = (new DateTime('now', new DateTimeZone('UTC')))->format('Y-m-d H:i:s');
}
if ($status !== 'completed') {
  // Keep completed_at for history; do not clear automatically.
}

$upd = $pdo->prepare("
  UPDATE ops_work_orders
     SET service_id=:sid,
         status=:st,
         priority=:prio,
         summary=:summary,
         serial_number=:serial_number,
         customer_name=:customer_name,
         customer_phone=:customer_phone,
         customer_email=:customer_email,
         install_location=:install_location,
         address_line1=:address_line1,
         address_line2=:address_line2,
         city=:city,
         state=:state,
         postal_code=:postal_code,
         scheduled_date=:scheduled_date,
         due_date=:due_date,
         internal_notes=:internal_notes,
         customer_notes=:customer_notes,
         completed_at=COALESCE(:completed_at, completed_at)
   WHERE id=:id AND company_id=:cid
");
$upd->execute([
  ':sid'=>$serviceId,
  ':st'=>$status,
  ':prio'=>$priority,
  ':summary'=>$fields['summary'] ?: null,
  ':serial_number'=>$fields['serial_number'] ?: null,
  ':customer_name'=>$fields['customer_name'] ?: null,
  ':customer_phone'=>$fields['customer_phone'] ?: null,
  ':customer_email'=>$fields['customer_email'] ?: null,
  ':install_location'=>$fields['install_location'] ?: null,
  ':address_line1'=>$fields['address_line1'] ?: null,
  ':address_line2'=>$fields['address_line2'] ?: null,
  ':city'=>$fields['city'] ?: null,
  ':state'=>$fields['state'] ?: null,
  ':postal_code'=>$fields['postal_code'] ?: null,
  ':scheduled_date'=>$fields['scheduled_date'],
  ':due_date'=>$fields['due_date'],
  ':internal_notes'=>$fields['internal_notes'] ?: null,
  ':customer_notes'=>$fields['customer_notes'] ?: null,
  ':completed_at'=>$completedAt,
  ':id'=>$id,
  ':cid'=>$companyId,
]);

// Event logging (minimal diffs)
$msgParts = [];
if ((int)$old['service_id'] !== $serviceId) $msgParts[] = 'Service updated';
if ((string)$old['status'] !== $status) $msgParts[] = 'Status '.$old['status'].' → '.$status;
if ((int)$old['priority'] !== $priority) $msgParts[] = 'Priority updated';
if ($msgParts) {
  $pdo->prepare("INSERT INTO ops_work_order_events (work_order_id, actor_user_id, event_type, message) VALUES (:wo,:uid,'updated',:msg)")
      ->execute([':wo'=>$id, ':uid'=>$actor, ':msg'=>implode(' · ', $msgParts)]);
}

header('Location: /app/operations/workorder_view.php?id='.$id);
exit;
