<?php
// /app/operations/cron/cron_process_webhook_orders.php
declare(strict_types=1);



require_once __DIR__ . '/../_bootstrap.php';
require_once __DIR__ . '/../_bootstrap.php';
/**
 * Processes pending ops_webhook_inbox rows:
 * - Extracts order + addons from stored payload
 * - Matches add-on names to ops_services via ops_addon_service_map
 * - Upserts ops_work_orders (idempotent per company+external_order_id+service_id)
 * - Writes ops_work_order_events for creations/updates
 *
 * Recommended schedule: every 2 minutes.
 */

require_once __DIR__ . '/../_bootstrap.php';
require_once __DIR__ . '/../../includes/db.php';

function dbw(): PDO {
  if (function_exists('db')) return db();
  if (isset($GLOBALS['pdo']) && $GLOBALS['pdo'] instanceof PDO) return $GLOBALS['pdo'];
  throw new RuntimeException('No DB connection');
}

$pdo = dbw();
$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

function pick(array $a, array $keys, $default=null) {
  foreach ($keys as $k) {
    if (array_key_exists($k, $a) && $a[$k] !== null && $a[$k] !== '') return $a[$k];
  }
  return $default;
}

function flatten_strings($v, array &$out): void {
  if (is_string($v)) {
    $s = trim($v);
    if ($s !== '') $out[] = $s;
    return;
  }
  if (is_array($v)) {
    foreach ($v as $vv) flatten_strings($vv, $out);
  }
}

function extract_order(array $payload): array {
  $order = $payload['order'] ?? $payload['data']['order'] ?? $payload['data'] ?? $payload;
  return is_array($order) ? $order : [];
}

/** Best-effort add-on extraction; returns list of strings */
function extract_addons(array $order): array {
  $candidates = [];

  foreach (['addons','add_ons','addOns','options','line_items','items','products'] as $k) {
    if (!empty($order[$k]) && is_array($order[$k])) {
      foreach ($order[$k] as $item) {
        if (is_array($item)) {
          $name = pick($item, ['addon_name','name','title','description','label'], null);
          if (is_string($name) && trim($name) !== '') $candidates[] = trim($name);
        } elseif (is_string($item)) {
          $candidates[] = trim($item);
        }
      }
    }
  }

  // Some payloads store add-ons nested (e.g. order['building']['addons'])
  if (!empty($order['building']) && is_array($order['building'])) {
    foreach (['addons','options','add_ons'] as $k) {
      if (!empty($order['building'][$k]) && is_array($order['building'][$k])) {
        foreach ($order['building'][$k] as $item) {
          if (is_array($item)) {
            $name = pick($item, ['addon_name','name','title','description','label'], null);
            if (is_string($name) && trim($name) !== '') $candidates[] = trim($name);
          } elseif (is_string($item)) {
            $candidates[] = trim($item);
          }
        }
      }
    }
  }

  // Last resort: scan any string fields that look like addon labels
  if (!$candidates) {
    $all = [];
    flatten_strings($order, $all);
    foreach ($all as $s) {
      if (stripos($s, 'package') !== false || stripos($s, 'mini') !== false || stripos($s, 'spray') !== false || stripos($s, 'finish') !== false) {
        $candidates[] = $s;
      }
    }
  }

  // Normalize and de-dupe
  $out = [];
  foreach ($candidates as $s) {
    $s = preg_replace('/\s+/', ' ', trim($s));
    if ($s === '') continue;
    $out[strtolower($s)] = $s;
  }
  return array_values($out);
}

function load_mappings(PDO $pdo, int $companyId): array {
  $stmt = $pdo->prepare("
    SELECT m.*, s.service_code, s.service_name, s.default_priority, s.default_due_days, s.default_assignee_user_id
    FROM ops_addon_service_map m
    JOIN ops_services s ON s.id=m.service_id
    WHERE m.company_id=:cid AND m.is_active=1 AND s.is_active=1
    ORDER BY m.priority ASC, m.id ASC
  ");
  $stmt->execute([':cid'=>$companyId]);
  return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

function match_services(array $addons, array $mappings): array {
  // Returns service_id => ['service'=>mapping row, 'addons'=>[]]
  $hits = [];
  foreach ($addons as $addonName) {
    foreach ($mappings as $m) {
      $mt = (string)$m['match_type'];
      $mv = (string)$m['match_value'];
      $ok = false;
      if ($mt === 'exact') $ok = strcasecmp($addonName, $mv) === 0;
      if ($mt === 'contains') $ok = stripos($addonName, $mv) !== false;
      if ($mt === 'regex') {
        $pattern = $mv;
        if (@preg_match($pattern, '') === false) $pattern = '/'.str_replace('/','\/', $pattern).'/i';
        $ok = @preg_match($pattern, $addonName) === 1;
      }
      if ($ok) {
        $sid = (int)$m['service_id'];
        if (!isset($hits[$sid])) $hits[$sid] = ['service'=>$m, 'addons'=>[]];
        $hits[$sid]['addons'][] = $addonName;
        break; // first matching rule wins for this addon
      }
    }
  }
  // de-dupe addon list per service
  foreach ($hits as $sid=>$v) {
    $u = [];
    foreach ($v['addons'] as $a) $u[strtolower($a)] = $a;
    $hits[$sid]['addons'] = array_values($u);
  }
  return $hits;
}

$batchSize = 50;

$pending = $pdo->prepare("
  SELECT *
  FROM ops_webhook_inbox
  WHERE status='pending'
  ORDER BY received_at ASC
  LIMIT :lim
");
$pending->bindValue(':lim', $batchSize, PDO::PARAM_INT);
$pending->execute();
$rows = $pending->fetchAll(PDO::FETCH_ASSOC);

if (php_sapi_name() === 'cli') {
  echo "Pending: ".count($rows).PHP_EOL;
}

foreach ($rows as $inbox) {
  $inboxId = (int)$inbox['id'];
  $companyId = (int)$inbox['company_id'];

  try {
    $payload = json_decode((string)$inbox['payload_json'], true, 512, JSON_THROW_ON_ERROR);
    $order = extract_order($payload);

    $externalOrderId = (string)($inbox['external_order_id'] ?? '');
    if ($externalOrderId === '') $externalOrderId = (string)(pick($order, ['order_id','id','uuid'], ''));

    $externalOrderNumber = (string)(pick($order, ['order_number','number','invoice_number'], ''));
    $serial = (string)(pick($order, ['serial_number','building_serial_number','serial'], ''));

    $customerName = '';
    if (!empty($order['customer']) && is_array($order['customer'])) {
      $customerName = trim((string)pick($order['customer'], ['name','full_name'], ''));
      if ($customerName === '') {
        $customerName = trim((string)pick($order['customer'], ['first_name'], '').' '.(string)pick($order['customer'], ['last_name'], ''));
      }
    }
    if ($customerName === '') $customerName = trim((string)pick($order, ['customer_name','name'], ''));

    $phone = '';
    $email = '';
    if (!empty($order['customer']) && is_array($order['customer'])) {
      $phone = (string)pick($order['customer'], ['phone','phone_number','mobile'], '');
      $email = (string)pick($order['customer'], ['email'], '');
    } else {
      $phone = (string)pick($order, ['customer_phone','phone'], '');
      $email = (string)pick($order, ['customer_email','email'], '');
    }

    $addr = $order['delivery_address'] ?? $order['address'] ?? ($order['customer']['address'] ?? null);
    $addr = is_array($addr) ? $addr : [];
    $address1 = (string)pick($addr, ['line1','address1','street','address'], '');
    $address2 = (string)pick($addr, ['line2','address2','unit'], '');
    $city     = (string)pick($addr, ['city'], '');
    $state    = (string)pick($addr, ['state','region'], '');
    $zip      = (string)pick($addr, ['zip','postal_code'], '');

    $addons = extract_addons($order);

    $mappings = load_mappings($pdo, $companyId);
    $serviceHits = match_services($addons, $mappings);

    // Mark processed if no relevant services (still processed - nothing to do)
    if (!$serviceHits) {
      $pdo->prepare("UPDATE ops_webhook_inbox SET status='processed', processed_at=UTC_TIMESTAMP() WHERE id=:id")
          ->execute([':id'=>$inboxId]);
      continue;
    }

    foreach ($serviceHits as $serviceId=>$hit) {
      $svc = $hit['service'];
      $matchedAddons = $hit['addons'];

      $prio = (int)($svc['default_priority'] ?? 3);
      $dueDays = (int)($svc['default_due_days'] ?? 7);
      $dueDate = (new DateTime('now', new DateTimeZone('UTC')))->modify('+'.$dueDays.' days')->format('Y-m-d');

      $summary = (string)$svc['service_name'];
      if ($matchedAddons) $summary .= ' — '.implode(', ', $matchedAddons);

      $payloadSlim = [
        'matched_addons' => $matchedAddons,
        'external_order_id' => $externalOrderId,
        'external_order_number' => $externalOrderNumber,
      ];

      // Upsert work order (idempotent)
      $ins = $pdo->prepare("
        INSERT INTO ops_work_orders
          (company_id, service_id, status, priority, source, external_order_id, external_order_number,
           serial_number, customer_name, customer_phone, customer_email,
           address_line1, address_line2, city, state, postal_code,
           due_date, summary, source_payload_json)
        VALUES
          (:cid,:sid,'new',:prio,'shedsuite',:oid,:onum,
           :serial,:cname,:phone,:email,
           :a1,:a2,:city,:state,:zip,
           :due,:summary,:payload)
        ON DUPLICATE KEY UPDATE
          serial_number=COALESCE(VALUES(serial_number), serial_number),
          customer_name=COALESCE(VALUES(customer_name), customer_name),
          customer_phone=COALESCE(VALUES(customer_phone), customer_phone),
          customer_email=COALESCE(VALUES(customer_email), customer_email),
          address_line1=COALESCE(VALUES(address_line1), address_line1),
          address_line2=COALESCE(VALUES(address_line2), address_line2),
          city=COALESCE(VALUES(city), city),
          state=COALESCE(VALUES(state), state),
          postal_code=COALESCE(VALUES(postal_code), postal_code),
          summary=VALUES(summary),
          source_payload_json=VALUES(source_payload_json)
      ");

      $ins->execute([
        ':cid'=>$companyId,
        ':sid'=>$serviceId,
        ':prio'=>$prio,
        ':oid'=>$externalOrderId ?: null,
        ':onum'=>$externalOrderNumber ?: null,
        ':serial'=>$serial ?: null,
        ':cname'=>$customerName ?: null,
        ':phone'=>$phone ?: null,
        ':email'=>$email ?: null,
        ':a1'=>$address1 ?: null,
        ':a2'=>$address2 ?: null,
        ':city'=>$city ?: null,
        ':state'=>$state ?: null,
        ':zip'=>$zip ?: null,
        ':due'=>$dueDate,
        ':summary'=>$summary,
        ':payload'=>json_encode($payloadSlim, JSON_UNESCAPED_SLASHES),
      ]);

      // Determine WO id
      $woId = (int)$pdo->lastInsertId();
      if ($woId === 0) {
        $find = $pdo->prepare("SELECT id FROM ops_work_orders WHERE company_id=:cid AND external_order_id=:oid AND service_id=:sid");
        $find->execute([':cid'=>$companyId, ':oid'=>$externalOrderId, ':sid'=>$serviceId]);
        $woId = (int)$find->fetchColumn();
      }

      // If created and service has default assignee, set lead.
      if ($woId > 0 && !empty($svc['default_assignee_user_id'])) {
        $pdo->prepare("INSERT IGNORE INTO ops_work_order_assignees (work_order_id, user_id, role) VALUES (:wo,:uid,'lead')")
            ->execute([':wo'=>$woId, ':uid'=>(int)$svc['default_assignee_user_id']]);
      }

      // Event (lightweight)
      if ($woId > 0) {
        $pdo->prepare("INSERT INTO ops_work_order_events (work_order_id, actor_user_id, event_type, message, meta_json)
                       VALUES (:wo, NULL, 'webhook', :msg, :meta)")
            ->execute([
              ':wo'=>$woId,
              ':msg'=>'Webhook processed (ShedSuite)'.($matchedAddons ? ' — '.implode(', ', $matchedAddons) : ''),
              ':meta'=>json_encode(['inbox_id'=>$inboxId,'addons'=>$matchedAddons], JSON_UNESCAPED_SLASHES),
            ]);
      }
    }

    $pdo->prepare("UPDATE ops_webhook_inbox SET status='processed', processed_at=UTC_TIMESTAMP() WHERE id=:id")
        ->execute([':id'=>$inboxId]);

  } catch (Throwable $e) {
    $pdo->prepare("UPDATE ops_webhook_inbox SET status='failed', error_message=:msg WHERE id=:id")
        ->execute([':msg'=>substr($e->getMessage(), 0, 250), ':id'=>$inboxId]);

    if (php_sapi_name() === 'cli') {
      echo "Inbox {$inboxId} failed: ".$e->getMessage().PHP_EOL;
    }
  }
}
