<?php
// /app/operations/addon_mapping.php
declare(strict_types=1);



require_once __DIR__ . '/_bootstrap.php';
require_once __DIR__ . '/includes/bootstrap.php';
require_once __DIR__ . '/_lib.php';

$pageTitle = 'Add-on Mapping';
$activeNav = 'operations';

$companyId = ops_require_company_scope();
if (!ops_can_configure()) { http_response_code(403); exit('Forbidden'); }

$pdo = db_safe();

$servicesStmt = $pdo->prepare("SELECT id, service_name FROM ops_services WHERE company_id=:cid AND is_active=1 ORDER BY service_name");
$servicesStmt->execute([':cid'=>$companyId]);
$services = $servicesStmt->fetchAll(PDO::FETCH_ASSOC);
$serviceMap = [];
foreach ($services as $s) $serviceMap[(int)$s['id']] = (string)$s['service_name'];

$maps = $pdo->prepare("
  SELECT m.*, s.service_name
  FROM ops_addon_service_map m
  JOIN ops_services s ON s.id=m.service_id
  WHERE m.company_id=:cid
  ORDER BY m.is_active DESC, m.priority ASC, m.id ASC
");
$maps->execute([':cid'=>$companyId]);
$maps = $maps->fetchAll(PDO::FETCH_ASSOC);

$testAddon = trim((string)($_GET['test'] ?? ''));
$testResult = [];
if ($testAddon !== '') {
  $sql = "
    SELECT m.*, s.service_name
    FROM ops_addon_service_map m
    JOIN ops_services s ON s.id=m.service_id
    WHERE m.company_id=:cid AND m.is_active=1
    ORDER BY m.priority ASC, m.id ASC
  ";
  $st = $pdo->prepare($sql);
  $st->execute([':cid'=>$companyId]);
  $rows = $st->fetchAll(PDO::FETCH_ASSOC);
  foreach ($rows as $r) {
    $ok = false;
    $mv = (string)$r['match_value'];
    $mt = (string)$r['match_type'];
    if ($mt === 'exact') $ok = strcasecmp($testAddon, $mv) === 0;
    if ($mt === 'contains') $ok = stripos($testAddon, $mv) !== false;
    if ($mt === 'regex') {
      $pattern = $mv;
      if (@preg_match($pattern, '') === false) {
        // allow users to store naked patterns; wrap if needed
        $pattern = '/'.str_replace('/','\/', $pattern).'/i';
      }
      $ok = @preg_match($pattern, $testAddon) === 1;
    }
    if ($ok) $testResult[] = $r;
  }
}

require_once __DIR__ . '/../includes/header.php';
$activeOps = 'mapping';
require __DIR__ . '/_subnav.php';
?>
<div class="row g-3">
  <div class="col-12 col-lg-8">
    <div class="card shadow-sm">
      <div class="card-header bg-dark text-white d-flex justify-content-between">
        <strong>Add-on → Service Mapping</strong>
        <button class="btn btn-sm btn-primary" type="button" data-bs-toggle="modal" data-bs-target="#mapModal">Add Mapping</button>
      </div>
      <div class="card-body">
        <div class="small text-muted mb-3">
          These rules translate incoming ShedSuite add-on names into services to auto-create work orders.
          Rules are evaluated by <strong>priority</strong> (lower first), then by ID.
        </div>

        <?php if (!$maps): ?>
          <div class="text-muted">No mappings configured.</div>
        <?php else: ?>
          <div class="table-responsive">
            <table class="table table-sm table-striped align-middle">
              <thead class="table-dark">
                <tr>
                  <th>Service</th>
                  <th>Match</th>
                  <th>Value</th>
                  <th>Priority</th>
                  <th>Active</th>
                  <th></th>
                </tr>
              </thead>
              <tbody>
                <?php foreach ($maps as $m): ?>
                  <tr>
                    <td class="fw-semibold"><?=h((string)$m['service_name'])?></td>
                    <td><code><?=h((string)$m['match_type'])?></code></td>
                    <td><?=h((string)$m['match_value'])?></td>
                    <td><?= (int)$m['priority'] ?></td>
                    <td><?= ((int)$m['is_active']===1)?'<span class="badge bg-success">Yes</span>':'<span class="badge bg-secondary">No</span>' ?></td>
                    <td class="text-end">
                      <button class="btn btn-sm btn-outline-primary js-edit"
                        data-id="<?= (int)$m['id'] ?>"
                        data-service_id="<?= (int)$m['service_id'] ?>"
                        data-match_type="<?=h((string)$m['match_type'])?>"
                        data-match_value="<?=h((string)$m['match_value'])?>"
                        data-priority="<?= (int)$m['priority'] ?>"
                        data-is_active="<?= (int)$m['is_active'] ?>"
                      >Edit</button>
                    </td>
                  </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>
        <?php endif; ?>
      </div>
    </div>
  </div>

  <div class="col-12 col-lg-4">
    <div class="card shadow-sm">
      <div class="card-header"><strong>Test a ShedSuite Add-on</strong></div>
      <div class="card-body">
        <form method="get">
          <label class="form-label">Add-on name</label>
          <input class="form-control" name="test" value="<?=h($testAddon)?>" placeholder="e.g., Deluxe Electrical Package">
          <button class="btn btn-dark mt-2" type="submit">Test</button>
        </form>

        <?php if ($testAddon !== ''): ?>
          <hr>
          <div class="fw-semibold">Matches</div>
          <?php if (!$testResult): ?>
            <div class="text-muted">No mappings matched.</div>
          <?php else: ?>
            <ul class="list-group list-group-flush">
              <?php foreach ($testResult as $r): ?>
                <li class="list-group-item">
                  <div class="fw-semibold"><?=h((string)$r['service_name'])?></div>
                  <div class="small text-muted">Rule: <?=h((string)$r['match_type'])?> “<?=h((string)$r['match_value'])?>” · Priority <?= (int)$r['priority'] ?></div>
                </li>
              <?php endforeach; ?>
            </ul>
          <?php endif; ?>
        <?php endif; ?>
      </div>
    </div>
  </div>
</div>

<!-- Modal -->
<div class="modal fade" id="mapModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog">
    <form class="modal-content" method="post" action="/app/operations/addon_mapping_save.php">
      <input type="hidden" name="csrf" value="<?=h(csrf_token())?>">
      <input type="hidden" name="id" id="mapId" value="">
      <div class="modal-header">
        <h5 class="modal-title">Mapping</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <div class="mb-2">
          <label class="form-label">Service</label>
          <select class="form-select" name="service_id" id="mapService" required>
            <option value="">— select —</option>
            <?php foreach ($services as $s): ?>
              <option value="<?= (int)$s['id'] ?>"><?=h((string)$s['service_name'])?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div class="row g-2">
          <div class="col-6">
            <label class="form-label">Match Type</label>
            <select class="form-select" name="match_type" id="mapType">
              <option value="contains">contains</option>
              <option value="exact">exact</option>
              <option value="regex">regex</option>
            </select>
          </div>
          <div class="col-6">
            <label class="form-label">Priority</label>
            <input class="form-control" type="number" min="1" name="priority" id="mapPriority" value="100">
          </div>
        </div>

        <div class="mt-2">
          <label class="form-label">Match Value</label>
          <input class="form-control" name="match_value" id="mapValue" required placeholder="Electrical Package">
          <div class="form-text">
            For <code>contains</code>: substring (case-insensitive). For <code>regex</code>: either a full pattern (e.g. <code>/mini\s*split/i</code>) or a bare pattern.
          </div>
        </div>

        <div class="mt-2">
          <label class="form-label">Active</label>
          <select class="form-select" name="is_active" id="mapActive">
            <option value="1">Yes</option>
            <option value="0">No</option>
          </select>
        </div>
      </div>
      <div class="modal-footer">
        <button class="btn btn-primary" type="submit">Save</button>
        <button class="btn btn-outline-dark" type="button" data-bs-dismiss="modal">Close</button>
      </div>
    </form>
  </div>
</div>

<script>
(function(){
  const modalEl = document.getElementById('mapModal');
  const btns = document.querySelectorAll('.js-edit');
  btns.forEach(btn=>{
    btn.addEventListener('click', ()=>{
      document.getElementById('mapId').value = btn.dataset.id || '';
      document.getElementById('mapService').value = btn.dataset.service_id || '';
      document.getElementById('mapType').value = btn.dataset.match_type || 'contains';
      document.getElementById('mapValue').value = btn.dataset.match_value || '';
      document.getElementById('mapPriority').value = btn.dataset.priority || '100';
      document.getElementById('mapActive').value = btn.dataset.is_active || '1';
      const m = new bootstrap.Modal(modalEl);
      m.show();
    });
  });
})();
</script>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
