<?php
// /app/operations/_lib.php
declare(strict_types=1);



require_once __DIR__ . '/_bootstrap.php';
require_once __DIR__ . '/includes/bootstrap.php';
/**
 * Operations module shared helpers.
 * Designed to work with the ShedOffice.com framework:
 * - Uses require_login(), current_user(), user_has_role(), user_can()
 * - Uses db() helper when available; falls back to $pdo from legacy db_config.php
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/acl.php';
require_once __DIR__ . '/../includes/db.php';

require_login();

// Ensure ACL is loaded (roles + permissions cached in session)
if (function_exists('acl_bootstrap')) {
  acl_bootstrap(db_safe());
}

/** @return PDO */
function db_safe(): PDO {
  /** @var PDO $pdo */
  $pdo = db();
  return $pdo;
}

function h(string $v): string {
  return htmlspecialchars($v, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
}

/**
 * Determine active company scope for the current session.
 * - In company tenant mode: company_id = current_tenant_id()
 * - In platform/admin mode with active company: company_id = active_company_tenant_id()
 */
function ops_company_id(): int {
  $type = function_exists('current_tenant_type') ? (string)current_tenant_type() : 'company';
  if ($type === 'company' && function_exists('current_tenant_id')) {
    return (int)current_tenant_id();
  }
  if (function_exists('active_company_tenant_id')) {
    $cid = (int)(active_company_tenant_id() ?: 0);
    if ($cid > 0) return $cid;
  }
  // Last-resort: if current tenant id exists and looks like a company, use it.
  if (function_exists('current_tenant_id')) return (int)current_tenant_id();
  return 0;
}

function ops_require_company_scope(): int {
  $cid = ops_company_id();
  if ($cid <= 0) {
    http_response_code(403);
    echo "<div style='padding:16px;font-family:system-ui'>No active company context. Please select a company tenant.</div>";
    exit;
  }
  return $cid;
}

/** Permission gates */
function ops_can_view(): bool {
  return user_has_role('company_admin') || user_has_role('manager') || user_can('operations.view') || user_can('workorders.view');
}
function ops_can_manage(): bool {
  return user_has_role('company_admin') || user_has_role('manager') || user_can('workorders.edit') || user_can('operations.manage');
}
function ops_can_configure(): bool {
  return user_has_role('company_admin') || user_can('operations.configure') || user_can('services.manage');
}
function ops_can_webhooks(): bool {
  return user_has_role('company_admin') || user_can('webhooks.manage') || user_can('operations.configure');
}

/** CSRF */
function csrf_token(): string {
  if (session_status() !== PHP_SESSION_ACTIVE) session_start();
  if (empty($_SESSION['csrf_ops'])) {
    $_SESSION['csrf_ops'] = bin2hex(random_bytes(32));
  }
  return (string)$_SESSION['csrf_ops'];
}
function csrf_verify(): void {
  if ($_SERVER['REQUEST_METHOD'] !== 'POST') return;
  if (session_status() !== PHP_SESSION_ACTIVE) session_start();
  $tok = (string)($_POST['csrf'] ?? '');
  $ok = !empty($_SESSION['csrf_ops']) && hash_equals((string)$_SESSION['csrf_ops'], $tok);
  if (!$ok) {
    http_response_code(419);
    header('Content-Type: text/plain; charset=utf-8');
    echo "Invalid CSRF token.";
    exit;
  }
}

/** Status labels */
function ops_status_label(string $s): string {
  return match ($s) {
    'new' => 'New',
    'scheduled' => 'Scheduled',
    'in_progress' => 'In Progress',
    'blocked' => 'Blocked',
    'completed' => 'Completed',
    'canceled' => 'Canceled',
    default => $s,
  };
}
function ops_status_badge(string $s): string {
  return match ($s) {
    'new' => 'secondary',
    'scheduled' => 'info',
    'in_progress' => 'warning',
    'blocked' => 'danger',
    'completed' => 'success',
    'canceled' => 'dark',
    default => 'secondary',
  };
}
